package neutrino.text;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;
import java.util.Random;
import javax.swing.Timer;


/**
 * Persists backup copy of document. By default is turned off.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class BackupManager {

	public static final String BACKUPS_DIRECTORY = "backups";
	public static final String BACKUP_FILE_EXTENSION = "dat";
	public static final String NONAME_FILE = "noname";
	
	private TextEditor textEditor = null;
	private long id = 0L;
	private boolean isEnabled = false;
	private boolean isBackupPerformed = false;
	private Timer timer = null;
	private File backupsDir = null;
	
	private ActionListener timerListener = new ActionListener() {
		public synchronized void actionPerformed(ActionEvent e) {
			if (!isBackupPerformed) {
				persistDocument();
			}
		}
	};
	
	public BackupManager(TextEditor textComponent) {
		this.textEditor = textComponent;
		id = Math.abs(new Random().nextLong());
		backupsDir = new File(BACKUPS_DIRECTORY);
		if (!backupsDir.exists()) backupsDir.mkdir();
		
		timer = new Timer(1200, timerListener);
		timer.start();
	}

	
	private synchronized void persistDocument() {
		if (!isEnabled) {
			return;
		}
		if (!backupsDir.exists()) backupsDir.mkdir();		
		ObjectOutputStream stream = null;
		try {
			stream = new ObjectOutputStream(new FileOutputStream(getBackupFile()));
			if (textEditor.isFileLoaded()) {
				stream.writeObject(textEditor.getFile().getAbsolutePath());
			} else {
				stream.writeObject(NONAME_FILE);
			}
			stream.writeObject(textEditor.getTextComponent().getDocument());
			stream.writeObject(textEditor.getTextComponent().getClass().getCanonicalName());
			isBackupPerformed = true;
		} catch (FileNotFoundException e) {
			e.printStackTrace();
		} catch (IOException e) {
			e.printStackTrace();
		} finally {
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e) {
					e.printStackTrace();
				}
			}
		}
	}
	
	/**
	 * Force the backup manager to recreate backup from current document.
	 */
	public synchronized void persistBackup() {
		this.isBackupPerformed = false;
	}
	
	/**
	 * Sets the backup manager enabled when parameter is true
	 * @param isEnabled - boolean
	 */
	public synchronized void setEnabled(boolean isEnabled) {
		this.isEnabled = isEnabled;
		if (isEnabled) {
			persistBackup();
		} else {
			getBackupFile().delete();
		}
	}
	
	/**
	 * Returns true if the backup manager is enabled
	 * @return boolean
	 */
	public synchronized boolean isEnabled() {
		return this.isEnabled;
	}
	
	private File getBackupFile() {
		return new File(BACKUPS_DIRECTORY + "/backup" + Long.toString(id) + "." + BACKUP_FILE_EXTENSION);
	}
	
	/**
	 * Turns on the backup manager
	 */
	public synchronized void turnOn() {
		setEnabled(true);
	}

	/**
	 * Turns off the backup manager
	 */
	public synchronized void turnOff() {
		setEnabled(false);
	}

}
