package neutrino.text;

import java.awt.Font;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.*;
import java.util.*;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.JTextArea;
import javax.swing.KeyStroke;
import javax.swing.event.*;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.Element;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.CompoundEdit;
import javax.swing.undo.UndoManager;

/**
 * Incapsulates text component.
 * Also supports undo and redo operations.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class PlainTextArea extends JTextArea {
	
	private UndoManager undoManager = null;
 	private Stack<CompoundEdit> edits = null;  	
	private int numberOfCurrentEdit = 0;	
	private boolean autoIndentMode = true;
	private JPopupMenu popupMenu = null;
	private boolean undoable = true;
	
	public PlainTextArea() {
		super();
		setTabSize(4);
		undoManager = new UndoManager();
		undoManager.setLimit(-1);
		edits = new Stack<CompoundEdit>();
		getDocument().addUndoableEditListener(undoableEditListener);
		addKeyListener(startLineKeyListener);
		final PlainTextArea instance = this;
		addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (popupMenu != null && e.getButton() == e.BUTTON3) {
					popupMenu.show(instance, e.getX(), e.getY());
				}
			}
		});
		createDefaultPopupMenu();
	}
	
	protected void createDefaultPopupMenu() {
		// create menu items
		final JMenuItem pmiUndo = new JMenuItem("Undo");
		final JMenuItem pmiRedo = new JMenuItem("Redo");
		final JMenuItem pmiCut = new JMenuItem("Cut");
		final JMenuItem pmiCopy = new JMenuItem("Copy");
		final JMenuItem pmiPaste = new JMenuItem("Paste");
		final JMenuItem pmiSelectAll = new JMenuItem("Select all");
		// create popup menu
		popupMenu = new JPopupMenu();
		popupMenu.addPopupMenuListener(new PopupMenuListener() {
			@Override
			public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
				pmiUndo.setEnabled(canUndo());
				pmiRedo.setEnabled(canRedo());
				pmiCut.setEnabled(canCut());
				pmiCopy.setEnabled(canCopy());
				pmiPaste.setEnabled(canPaste());
				pmiSelectAll.setEnabled(canSelectAll());
			}
			@Override
			public void popupMenuWillBecomeInvisible(PopupMenuEvent e) { }
			@Override
			public void popupMenuCanceled(PopupMenuEvent e) { }
		});
		// build popup menu
		popupMenu.add(pmiUndo);
		popupMenu.add(pmiRedo);
		popupMenu.addSeparator();		
		popupMenu.add(pmiCut);
		popupMenu.add(pmiCopy);
		popupMenu.add(pmiPaste);
		popupMenu.addSeparator();
		popupMenu.add(pmiSelectAll);
		// build mnemonics
		pmiUndo.setMnemonic(KeyEvent.VK_U);
		pmiRedo.setMnemonic(KeyEvent.VK_R);
		pmiCut.setMnemonic(KeyEvent.VK_C);
		pmiCopy.setMnemonic(KeyEvent.VK_O);
		pmiPaste.setMnemonic(KeyEvent.VK_P);
		pmiSelectAll.setMnemonic(KeyEvent.VK_S);
		// build accelerators
		pmiUndo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, InputEvent.CTRL_DOWN_MASK));
		pmiRedo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y, InputEvent.CTRL_DOWN_MASK));
		pmiCut.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.CTRL_DOWN_MASK));
		pmiCopy.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_DOWN_MASK));
		pmiPaste.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, InputEvent.CTRL_DOWN_MASK));
		pmiSelectAll.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_DOWN_MASK));
		// build actions
		ActionListener listener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (e.getSource() == pmiUndo) {
					undo();
				} else if (e.getSource() == pmiRedo) {
					redo();
				} else if (e.getSource() == pmiCut) {
					cut();
				} else if (e.getSource() == pmiCopy) {
					copy();
				} else if (e.getSource() == pmiPaste) {
					paste();
				} else if (e.getSource() == pmiSelectAll) {
					selectAll();
				}
			}
		};
		pmiUndo.addActionListener(listener);
		pmiRedo.addActionListener(listener);
		pmiCut.addActionListener(listener);
		pmiCopy.addActionListener(listener);
		pmiPaste.addActionListener(listener);
		pmiSelectAll.addActionListener(listener);
	}
	
	/**
	 * Sets popup menu for text component
	 * @param popupMenu - a popup menu
	 */
	public void setPopupMenu(JPopupMenu popupMenu) {
		this.popupMenu = popupMenu;
	}
	
	public void setFont(Font font) {
		int tabSize = super.getTabSize();
		super.setFont(font);
		super.setTabSize(tabSize);
	}
	
	/**
	 * Notices that text is not changed when new document is set
	 * Prepares undo manager for set document
	 */
	public void setDocument(Document doc) {
		int tabSize = super.getTabSize();
		super.setDocument(doc);
		super.setTabSize(tabSize);
		doc.addUndoableEditListener(undoableEditListener);
		reset(true);
		System.gc();
	}

	/**
	 * Returns true when all text may be selected
	 * @return boolean
	 */
	public boolean canSelectAll() {
		return !isTextEmpty() && !isAllTextSelected();
	}
	
	/**
	 * Returns true when text may be printed 
	 * @return boolean
	 */
	public boolean canPrint() {
		return !isTextEmpty();
	}
	
	/**
	 * Chack is text component is empty.
	 * @return true when text is empty.
	 */
	public boolean isTextEmpty() {
		return getDocument().getLength() == 0;
	}
	
	/**
	 * Returns true when all text is selected
	 * @return boolean
	 */
	public boolean isAllTextSelected() {
		if (!isTextSelected()) return false;
		else return (getSelectionStart() == 0) && (getSelectionLength() == getDocument().getLength());
	}
	
	/**
	 * Set type of line start. If true then line start as previous. 
	 * If false then line start from beginning. 
	 * @param flag - boolean value
	 */
	public void setAutoIndentMode(boolean flag) {
		this.autoIndentMode = flag;
	}

	/**
	 * Return true if line starts as previous line.
	 * @return boolean value
	 */
	public boolean isAutoIndentMode() {
		return this.autoIndentMode;
	}
	
	/**
	 * Returns true if text is selected.
	 */
	public boolean isTextSelected() {
		return getSelectionStart() != getSelectionEnd();
	}
	
	/**
	 * Return length of selection or 0 if text is not selected. 
	 */
	public int getSelectionLength() {
		return getSelectionEnd() - getSelectionStart();
	}
	
	/**
	 * Make selection uppercase.
	 * If text is not selected then do nothing.
	 */
	public void makeUppercase() {
		if (!isTextSelected()) return;
		int startPosition = getSelectionStart();
		int length = getSelectionLength();
		String sourceText = getSelectedText();
		String uppercaseText = sourceText.toUpperCase();
		try {
			beginEdit();
			getDocument().remove(startPosition, length);
			getDocument().insertString(startPosition, uppercaseText, null);
			endEdit();
			select(startPosition, startPosition + length);
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Make selection lowercase.
	 * If text is not selected then do nothing.
	 */
	public void makeLowercase() {
		if (!isTextSelected()) return;
		int startPosition = getSelectionStart();
		int length = getSelectionLength();
		String sourceText = getSelectedText();
		String lowercaseText = sourceText.toLowerCase();
		try {
			beginEdit();
			getDocument().remove(startPosition, length);
			getDocument().insertString(startPosition, lowercaseText, null);
			endEdit();
			select(startPosition, startPosition + length);
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Shift text on one symbol in left.
	 * @param text - text to be shifted
	 * @return shifted text
	 */
	private String shiftTextInLeft(String text) {
		if (text == null) return "";
		StringBuffer buffer = new StringBuffer();
		StringTokenizer tokenizer = new StringTokenizer(text, "\n", true);
		boolean isPreviousTokenNotEmpty = false;
		while (tokenizer.hasMoreTokens()) {
			String row = tokenizer.nextToken();
			if (row.equals("\n")) {
				if (isPreviousTokenNotEmpty) {
					isPreviousTokenNotEmpty = false;
				} else {
					buffer.append('\n');
				}
				continue;
			}
			if (row.length() > 0 && Character.isSpace(row.charAt(0))) {
				buffer.append(row.substring(1));
			} else {
				buffer.append(row);
			}
			isPreviousTokenNotEmpty = true;
			if (tokenizer.hasMoreTokens()) { 
				buffer.append('\n');
			}
		}
		return buffer.toString();
	}
	
	/**
	 * Shift text on one symbol in right.
	 * @param text - text to be shifted
	 * @return shifted text
	 */
	private String shiftTextInRight(String text) {
		if (text == null) return "";
		StringBuffer buffer = new StringBuffer();
		StringTokenizer tokenizer = new StringTokenizer(text, "\n", true);
		boolean isPreviousTokenNotEmpty = false;
		while (tokenizer.hasMoreTokens()) {
			String row = tokenizer.nextToken();
			if (row.equals("\n")) {
				if (isPreviousTokenNotEmpty) {
					isPreviousTokenNotEmpty = false;
				} else {
					buffer.append('\n');
				}
				continue;
			}
			buffer.append(' ');
			buffer.append(row);
			isPreviousTokenNotEmpty = true;
			if (tokenizer.hasMoreTokens()) { 
				buffer.append('\n');
			}
		}
		return buffer.toString();
	} 
	
	/**
	 * Shift selected text in left on one character.
	 * If text is not selected shift current line in left.
	 */
	public void shiftInLeft() {
		if (isTextSelected()) {
			String shiftedText = shiftTextInLeft(getSelectedText());
			int startPosition = getSelectionStart();
			int length = getSelectionLength();
			try {
				beginEdit();
				getDocument().remove(startPosition, length);
				getDocument().insertString(startPosition, shiftedText, null);
				endEdit();
				select(startPosition, startPosition + shiftedText.length());
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				// get current caret position
				char c;
				int position = getCaretPosition();
				// find first character position of current line
				c = getDocument().getText(position, 1).charAt(0);
				if (c == '\n') position--;
				if (position < 0) return;
				for (; position > 0; position--) {
					c = getDocument().getText(position, 1).charAt(0);
					if (c == '\n') {
						position++;
						break;
					}
				}
				// remove first space
				c = getDocument().getText(position, 1).charAt(0);
				if (c == ' ' || c == '\t') {
					getDocument().remove(position, 1);
				}
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		}
	}
	
	/**
	 * Shift selected text in right on one character.
	 * If text is not selected shift current line in right.
	 */
	public void shiftInRight() {
		if (isTextSelected()) {
			String shiftedText = shiftTextInRight(getSelectedText());
			int startPosition = getSelectionStart();
			int length = getSelectionLength();
			try {
				beginEdit();
				getDocument().remove(startPosition, length);
				getDocument().insertString(startPosition, shiftedText, null);
				endEdit();
				select(startPosition, startPosition + shiftedText.length());
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				// get current caret position
				char c;
				int position = getCaretPosition();
				// find first character position of current line
				c = getDocument().getText(position, 1).charAt(0);
				if (c == '\n') position--;
				if (position < 0) return;
				for (; position > 0; position--) {
					c = getDocument().getText(position, 1).charAt(0);
					if (c == '\n') {
						position++;
						break;
					}
				}
				// add space
				getDocument().insertString(position, " ", null);
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		}
	}
	
	/**
	 * Replaces groups of spaces with tabs in string. Returns result of replacement
	 * @param source - string for replacement
	 * @return String
	 */
	private String tabifyString(String source) {
		int tabSize = getTabSize();
		StringBuffer standard = new StringBuffer();
		for (int i = 0; i < tabSize; i++) {
			standard.append(' ');
		}
		return source.replaceAll(standard.toString(), "\t");
	}
	
	/*
	 * Replaces groups of spaces with tabs in selection. 
	 * If text is not selected do nothing.
	 */
	public void tabifySelectedLines() {
		if (isTextSelected()) {
			int selectionStart = getSelectionStart();
			String selectedText = getSelectedText();
			String result = tabifyString(selectedText);
			beginEdit();
			try {
				getDocument().remove(selectionStart, selectedText.length());
				getDocument().insertString(selectionStart, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(selectionStart, selectionStart + result.length());
		}
	}
	
	/**
	 * Replaces tabs with groups of spaces in string. 
	 * Returns result of replacement
	 * @param source - string for replacement
	 * @return String
	 */
	private String untabifyString(String source) {
		int tabSize = getTabSize();
		StringBuffer standard = new StringBuffer();
		for (int i = 0; i < tabSize; i++) {
			standard.append(' ');
		}
		return source.replace("\t", standard.toString());
	}
	
	/**
	 * Replaces tabs with groups of spaces in selection. 
	 * If text is not selected do nothing 
	 */
	public void untabifySelectedLines() {
		if (isTextSelected()) {
			int selectionStart = getSelectionStart();
			String selectedText = getSelectedText();
			String result = untabifyString(selectedText);
			beginEdit();
			try {
				getDocument().remove(selectionStart, selectedText.length());
				getDocument().insertString(selectionStart, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(selectionStart, selectionStart + result.length());
		}
	}
	
	/**
	 * Returns result of alignment source in left
	 * @param source - string to alignment
	 * @return String
	 */
	private String alignStringInLeft(String source) {
		StringTokenizer tokenizer = new StringTokenizer(source, "\n", true);
		StringBuffer result = new StringBuffer();
		while (tokenizer.hasMoreTokens()) {
			String token = tokenizer.nextToken();
			int startPosition = 0;
			while (startPosition < token.length() 
					&& (token.charAt(startPosition) == ' ' || token.charAt(startPosition) == '\t')) {
				startPosition++;
			}
			result.append(token.substring(startPosition));
		}
		return result.toString();
	}
	
	/**
	 * Removes start spaces from current line or from selection.
	 */
	public void deleteHorizontalWhiteSpace() {
		String source, result;
		int startPosition;
		if (isTextSelected()) {
			source = getSelectedText();
			startPosition = getSelectionStart();
			result = alignStringInLeft(source);
			beginEdit();
			try {
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(startPosition, startPosition + result.length());
		} else if (!isTextEmpty()) {
			try {
				Element rootElement = getDocument().getDefaultRootElement();
				Element currentLine = rootElement.getElement(rootElement.getElementIndex(getCaretPosition()));
				startPosition = currentLine.getStartOffset();
				source = getText(startPosition, currentLine.getEndOffset() - startPosition);
				result = alignStringInLeft(source);
				beginEdit();
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
				endEdit();
				setCaretPosition(startPosition);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * Removes trailing whitespaces from source and returns result
	 * @param source - String
	 * @return String
	 */
	private String removeTrailingWhitespacesFromString(String source) {
		StringTokenizer tokenizer = new StringTokenizer(source, "\n", true);
		StringBuffer result = new StringBuffer();
		while (tokenizer.hasMoreTokens()) {
			String token = tokenizer.nextToken();
			int endPosition = token.length() - 1;
			while (endPosition >= 0 
					&& (token.charAt(endPosition) == ' ' || token.charAt(endPosition) == '\t')) {
				endPosition--;
			}
			result.append(token.substring(0, endPosition + 1));
		}
		return result.toString();
	}
	
	/**
	 * Removes trailing whitespaces from current line or selected text
	 */
	public void removeTrailingWhitespaces() {
		String source, result;
		int startPosition;
		if (isTextSelected()) {
			source = getSelectedText();
			startPosition = getSelectionStart();
			result = removeTrailingWhitespacesFromString(source);
			beginEdit();
			try {
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(startPosition, startPosition + result.length());
		} else if (!isTextEmpty()) {
			try {
				Element rootElement = getDocument().getDefaultRootElement();
				Element currentLine = rootElement.getElement(rootElement.getElementIndex(getCaretPosition()));
				startPosition = currentLine.getStartOffset();
				source = getText(startPosition, currentLine.getEndOffset() - startPosition);
				result = removeTrailingWhitespacesFromString(source);
				beginEdit();
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
				endEdit();
				if (getCaretPosition() >= startPosition + result.length()) {
					setCaretPosition( startPosition + result.length() - 1);
				}
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		}
	}
	
	private KeyListener startLineKeyListener = new KeyListener() {
		
		public void keyPressed(KeyEvent e) { }

		public void keyReleased(KeyEvent e) { }

		public void keyTyped(KeyEvent e) {
			if (isAutoIndentMode() && e.getKeyChar() == KeyEvent.VK_ENTER) {
				int startPosition = 0;
				int length = 0;
				String text;
				int i = 0;
				do {
					i++;
					try {
						text = getDocument().getText(getCaretPosition() - 1 - i, 1);
					} catch (BadLocationException e1) {
						break;
					}
				} while(text.charAt(0) != KeyEvent.VK_ENTER);
				startPosition = getCaretPosition() - 1 - i + 1;
				do {
					try {
						text = getDocument().getText(startPosition + length, 1);
						if (Character.isSpace(text.charAt(0)) && text.charAt(0) != KeyEvent.VK_ENTER) {
							length++;
						}
					} catch (BadLocationException e1) {
						break;
					}
				} while(Character.isSpace(text.charAt(0)) && text.charAt(0) != KeyEvent.VK_ENTER);
				
				try {
					String spaceLine = getDocument().getText(startPosition, length);
					getDocument().insertString(getCaretPosition(), spaceLine, null);
				} catch (BadLocationException e1) {
					e1.printStackTrace();
				}
			}
		}
		
	};

	// undoable capability
	
	/**
	 * Sets undoable for text component
	 * {@value value - boolean}
	 */
	public void setUndoable(boolean value) {
		this.undoable = value;
	}
	
	/**
	 * Returns true when text component is undoable
	 * @return boolean
	 */
	public boolean isUndoable() {
		return this.undoable;
	}
	
	/**
	 * Chack is text is changed.
	 * @return true when text is changed.
	 */
	public boolean isTextChanged() {
		return numberOfCurrentEdit != 0;
	}	

	/**
	 * Returns true when text changes may be reverted
	 * @return boolean
	 */
	public boolean canRevert() {
		return undoable && isTextChanged();
	}	
	
	/**
	 * Returns true if the text change may be undone
	 */
	public boolean canUndo() {
		return undoable && undoManager.canUndo();
	}
	
	/**
	 * Returns true if the text change may be redone
	 */
	public boolean canRedo() {
		return undoable && undoManager.canRedo();
	}
	
	/**
	 * Undoes the last text change if possible. 
	 */
	public void undo() {
		if (undoable && undoManager.canUndo()) {
			undoManager.undo();
			numberOfCurrentEdit--;
		}
	}

	/**
	 * Redose the last text change if possible.
	 */
	public void redo() {
		if (undoable && undoManager.canRedo()) {
			undoManager.redo();
			numberOfCurrentEdit++;
		}
	}
	
 	/**
 	 * Returns true when compound edit is begun
 	 * @return boolean
 	 */
	public boolean isEditBegun() {
		return !edits.isEmpty();
	}
	
	/**
	 * Marks beginning of coalesce edit operation
	 */
	public void beginEdit() {
		if (!undoable) return;
		edits.push(new CompoundEdit());
	}
	
	/**
	 * Marks end of coalesce edit operation
	 */
	public void endEdit() {
		if (undoable) {
			if (!isEditBegun()) {
				return;
			}
			CompoundEdit currentEdit = edits.pop();
			if (isEditBegun()) {
				edits.peek().addEdit(currentEdit);
			} else {
				undoManager.addEdit(currentEdit);
			}
			currentEdit.end();
		}
		numberOfCurrentEdit++;
	}
	
	public void revert() {
		if (!undoable) return;
		try {
			while (isTextChanged()) {
				undoManager.undo();
				numberOfCurrentEdit--;
			}
		} catch (CannotUndoException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Marks text as not changed
	 */
	public void reset(boolean discardAllEdits) {
		if (discardAllEdits) {
			if (undoManager != null) {
				undoManager.discardAllEdits();
			}
			if (edits != null) {
				edits.clear();
			}
		}
		numberOfCurrentEdit = 0;
	}
	
	private UndoableEditListener undoableEditListener = new UndoableEditListener() {
		
		public void undoableEditHappened(UndoableEditEvent e) {
			if (isEditBegun()) {
				if (undoable) edits.peek().addEdit(e.getEdit());
			} else {
				if (undoable) undoManager.addEdit(e.getEdit());
				numberOfCurrentEdit++;
			}
		}
		
	};

	/**
	 * Returns true if clipboard contain text.
	 * @throws IllegalStateException
	 * @return boolean value
	 */
	public boolean isClipboardContainText() throws IllegalStateException {
		for (DataFlavor dataFlavor : getToolkit().getSystemClipboard().getAvailableDataFlavors()) {
			if (dataFlavor.isFlavorTextType()) {
				return true;
			}
		}
		return false;
	}
	
	public void cut() {
		if (canCut()) super.cut();
	}
	
	public void copy() {
		if (canCopy()) super.copy();
	}
	
	public void paste() {
		try {
			if (!isClipboardContainText()) return;
		} catch (IllegalStateException e) {
			e.printStackTrace();
		}
		super.paste();
	}
	
	/**
	 * Returns true when text fragment may be cut
	 * @return boolean
	 */
	public boolean canCut() {
		return isTextSelected();
	}
	
	/**
	 * Returns true when text fragment may be copied
	 * @return boolean
	 */
	public boolean canCopy() {
		return isTextSelected();
	}
	
	/**
	 * Returns true when text fragment may be pasted
	 * @return boolean
	 */
	public boolean canPaste() {
		try {
			return isClipboardContainText();
		} catch (IllegalStateException e1) {
			return true;
		}
	}
	
	public int getNumberOfCurrentLine() {
		Element root = getDocument().getDefaultRootElement();
		return root.getElementIndex(getCaretPosition()) + 1;
	}
	
	public int getNumberOfCurrentColumn() {
		Element root = getDocument().getDefaultRootElement();
		Element element = root.getElement(root.getElementIndex(getCaretPosition()));		
		return getCaretPosition() - element.getStartOffset() + 1;
	}
	
}
