package neutrino.text;

import java.awt.event.KeyEvent;
import java.awt.event.KeyListener;
import java.util.StringTokenizer;

import javax.swing.text.BadLocationException;
import javax.swing.text.Element;


public class PlainTextComponent extends TextComponent {
	
	private boolean autoIndentMode = true;
	
	public PlainTextComponent() {
		super();
		addKeyListener(startLineKeyListener);
	}
	
	/**
	 * Set type of line start. If true then line start as previous. 
	 * If false then line start from beginning. 
	 * @param flag - boolean value
	 */
	public void setAutoIndentMode(boolean flag) {
		this.autoIndentMode = flag;
	}

	/**
	 * Return true if line starts as previous line.
	 * @return boolean value
	 */
	public boolean isAutoIndentMode() {
		return this.autoIndentMode;
	}
	
	/**
	 * Make selection uppercase.
	 * If text is not selected then do nothing.
	 */
	public void makeUppercase() {
		if (!isTextSelected()) return;
		int startPosition = getSelectionStart();
		int length = getSelectionLength();
		String sourceText = getSelectedText();
		String uppercaseText = sourceText.toUpperCase();
		try {
			beginEdit();
			getDocument().remove(startPosition, length);
			getDocument().insertString(startPosition, uppercaseText, null);
			endEdit();
			select(startPosition, startPosition + length);
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Make selection lowercase.
	 * If text is not selected then do nothing.
	 */
	public void makeLowercase() {
		if (!isTextSelected()) return;
		int startPosition = getSelectionStart();
		int length = getSelectionLength();
		String sourceText = getSelectedText();
		String lowercaseText = sourceText.toLowerCase();
		try {
			beginEdit();
			getDocument().remove(startPosition, length);
			getDocument().insertString(startPosition, lowercaseText, null);
			endEdit();
			select(startPosition, startPosition + length);
		} catch (BadLocationException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Shift text on one symbol in left.
	 * @param text - text to be shifted
	 * @return shifted text
	 */
	private String shiftTextInLeft(String text) {
		if (text == null) return "";
		StringBuffer buffer = new StringBuffer();
		StringTokenizer tokenizer = new StringTokenizer(text, "\n", true);
		boolean isPreviousTokenNotEmpty = false;
		while (tokenizer.hasMoreTokens()) {
			String row = tokenizer.nextToken();
			if (row.equals("\n")) {
				if (isPreviousTokenNotEmpty) {
					isPreviousTokenNotEmpty = false;
				} else {
					buffer.append('\n');
				}
				continue;
			}
			if (row.length() > 0 && Character.isSpace(row.charAt(0))) {
				buffer.append(row.substring(1));
			} else {
				buffer.append(row);
			}
			isPreviousTokenNotEmpty = true;
			if (tokenizer.hasMoreTokens()) { 
				buffer.append('\n');
			}
		}
		return buffer.toString();
	}
	
	/**
	 * Shift text on one symbol in right.
	 * @param text - text to be shifted
	 * @return shifted text
	 */
	private String shiftTextInRight(String text) {
		if (text == null) return "";
		StringBuffer buffer = new StringBuffer();
		StringTokenizer tokenizer = new StringTokenizer(text, "\n", true);
		boolean isPreviousTokenNotEmpty = false;
		while (tokenizer.hasMoreTokens()) {
			String row = tokenizer.nextToken();
			if (row.equals("\n")) {
				if (isPreviousTokenNotEmpty) {
					isPreviousTokenNotEmpty = false;
				} else {
					buffer.append('\n');
				}
				continue;
			}
			buffer.append(' ');
			buffer.append(row);
			isPreviousTokenNotEmpty = true;
			if (tokenizer.hasMoreTokens()) { 
				buffer.append('\n');
			}
		}
		return buffer.toString();
	} 
	
	/**
	 * Shift selected text in left on one character.
	 * If text is not selected shift current line in left.
	 */
	public void shiftInLeft() {
		if (isTextSelected()) {
			String shiftedText = shiftTextInLeft(getSelectedText());
			int startPosition = getSelectionStart();
			int length = getSelectionLength();
			try {
				beginEdit();
				getDocument().remove(startPosition, length);
				getDocument().insertString(startPosition, shiftedText, null);
				endEdit();
				select(startPosition, startPosition + shiftedText.length());
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				// get current caret position
				char c;
				int position = getCaretPosition();
				// find first character position of current line
				c = getDocument().getText(position, 1).charAt(0);
				if (c == '\n') position--;
				if (position < 0) return;
				for (; position > 0; position--) {
					c = getDocument().getText(position, 1).charAt(0);
					if (c == '\n') {
						position++;
						break;
					}
				}
				// remove first space
				c = getDocument().getText(position, 1).charAt(0);
				if (c == ' ' || c == '\t') {
					getDocument().remove(position, 1);
				}
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		}
	}
	
	/**
	 * Shift selected text in right on one character.
	 * If text is not selected shift current line in right.
	 */
	public void shiftInRight() {
		if (isTextSelected()) {
			String shiftedText = shiftTextInRight(getSelectedText());
			int startPosition = getSelectionStart();
			int length = getSelectionLength();
			try {
				beginEdit();
				getDocument().remove(startPosition, length);
				getDocument().insertString(startPosition, shiftedText, null);
				endEdit();
				select(startPosition, startPosition + shiftedText.length());
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		} else if (!isTextEmpty()) {
			try {
				// get current caret position
				char c;
				int position = getCaretPosition();
				// find first character position of current line
				c = getDocument().getText(position, 1).charAt(0);
				if (c == '\n') position--;
				if (position < 0) return;
				for (; position > 0; position--) {
					c = getDocument().getText(position, 1).charAt(0);
					if (c == '\n') {
						position++;
						break;
					}
				}
				// add space
				getDocument().insertString(position, " ", null);
			} catch (BadLocationException e1) {
				e1.printStackTrace();
			}
		}
	}
	
	/**
	 * Replaces groups of spaces with tabs in string. Returns result of replacement
	 * @param source - string for replacement
	 * @return String
	 */
	private String tabifyString(String source) {
		int tabSize = 4; // getTabSize();
		StringBuffer standard = new StringBuffer();
		for (int i = 0; i < tabSize; i++) {
			standard.append(' ');
		}
		return source.replaceAll(standard.toString(), "\t");
	}
	
	/*
	 * Replaces groups of spaces with tabs in selection. 
	 * If text is not selected do nothing.
	 */
	public void tabifySelectedLines() {
		if (isTextSelected()) {
			int selectionStart = getSelectionStart();
			String selectedText = getSelectedText();
			String result = tabifyString(selectedText);
			beginEdit();
			try {
				getDocument().remove(selectionStart, selectedText.length());
				getDocument().insertString(selectionStart, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(selectionStart, selectionStart + result.length());
		}
	}
	
	/**
	 * Replaces tabs with groups of spaces in string. 
	 * Returns result of replacement
	 * @param source - string for replacement
	 * @return String
	 */
	private String untabifyString(String source) {
		int tabSize = 4; // getTabSize();
		StringBuffer standard = new StringBuffer();
		for (int i = 0; i < tabSize; i++) {
			standard.append(' ');
		}
		return source.replace("\t", standard.toString());
	}
	
	/**
	 * Replaces tabs with groups of spaces in selection. 
	 * If text is not selected do nothing 
	 */
	public void untabifySelectedLines() {
		if (isTextSelected()) {
			int selectionStart = getSelectionStart();
			String selectedText = getSelectedText();
			String result = untabifyString(selectedText);
			beginEdit();
			try {
				getDocument().remove(selectionStart, selectedText.length());
				getDocument().insertString(selectionStart, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(selectionStart, selectionStart + result.length());
		}
	}
	
	/**
	 * Returns result of alignment source in left
	 * @param source - string to alignment
	 * @return String
	 */
	private String alignStringInLeft(String source) {
		StringTokenizer tokenizer = new StringTokenizer(source, "\n", true);
		StringBuffer result = new StringBuffer();
		while (tokenizer.hasMoreTokens()) {
			String token = tokenizer.nextToken();
			int startPosition = 0;
			while (startPosition < token.length() 
					&& (token.charAt(startPosition) == ' ' || token.charAt(startPosition) == '\t')) {
				startPosition++;
			}
			result.append(token.substring(startPosition));
		}
		return result.toString();
	}
	
	/**
	 * Removes start spaces from current line or from selection.
	 */
	public void deleteHorizontalWhiteSpace() {
		String source, result;
		int startPosition;
		if (isTextSelected()) {
			source = getSelectedText();
			startPosition = getSelectionStart();
			result = alignStringInLeft(source);
			beginEdit();
			try {
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(startPosition, startPosition + result.length());
		} else if (!isTextEmpty()) {
			try {
				Element rootElement = getDocument().getDefaultRootElement();
				Element currentLine = rootElement.getElement(rootElement.getElementIndex(getCaretPosition()));
				startPosition = currentLine.getStartOffset();
				source = getText(startPosition, currentLine.getEndOffset() - startPosition);
				result = alignStringInLeft(source);
				beginEdit();
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
				endEdit();
				setCaretPosition(startPosition);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		}
	}
	
	/**
	 * Removes trailing whitespaces from source and returns result
	 * @param source - String
	 * @return String
	 */
	private String removeTrailingWhitespacesFromString(String source) {
		StringTokenizer tokenizer = new StringTokenizer(source, "\n", true);
		StringBuffer result = new StringBuffer();
		while (tokenizer.hasMoreTokens()) {
			String token = tokenizer.nextToken();
			int endPosition = token.length() - 1;
			while (endPosition >= 0 
					&& (token.charAt(endPosition) == ' ' || token.charAt(endPosition) == '\t')) {
				endPosition--;
			}
			result.append(token.substring(0, endPosition + 1));
		}
		return result.toString();
	}
	
	/**
	 * Removes trailing whitespaces from current line or selected text
	 */
	public void removeTrailingWhitespaces() {
		String source, result;
		int startPosition;
		if (isTextSelected()) {
			source = getSelectedText();
			startPosition = getSelectionStart();
			result = removeTrailingWhitespacesFromString(source);
			beginEdit();
			try {
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
			endEdit();
			select(startPosition, startPosition + result.length());
		} else if (!isTextEmpty()) {
			try {
				Element rootElement = getDocument().getDefaultRootElement();
				Element currentLine = rootElement.getElement(rootElement.getElementIndex(getCaretPosition()));
				startPosition = currentLine.getStartOffset();
				source = getText(startPosition, currentLine.getEndOffset() - startPosition);
				result = removeTrailingWhitespacesFromString(source);
				beginEdit();
				getDocument().remove(startPosition, source.length());
				getDocument().insertString(startPosition, result, null);
				endEdit();
				if (getCaretPosition() >= startPosition + result.length()) {
					setCaretPosition( startPosition + result.length() - 1);
				}
			} catch (BadLocationException e) {
				e.printStackTrace();
			}
		}
	}
	
	private KeyListener startLineKeyListener = new KeyListener() {
		
		public void keyPressed(KeyEvent e) { }

		public void keyReleased(KeyEvent e) { }

		public void keyTyped(KeyEvent e) {
			if (isAutoIndentMode() && e.getKeyChar() == KeyEvent.VK_ENTER) {
				int startPosition = 0;
				int length = 0;
				String text;
				int i = 0;
				do {
					i++;
					try {
						text = getDocument().getText(getCaretPosition() - 1 - i, 1);
					} catch (BadLocationException e1) {
						break;
					}
				} while(text.charAt(0) != KeyEvent.VK_ENTER);
				startPosition = getCaretPosition() - 1 - i + 1;
				do {
					try {
						text = getDocument().getText(startPosition + length, 1);
						if (Character.isSpace(text.charAt(0)) && text.charAt(0) != KeyEvent.VK_ENTER) {
							length++;
						}
					} catch (BadLocationException e1) {
						break;
					}
				} while(Character.isSpace(text.charAt(0)) && text.charAt(0) != KeyEvent.VK_ENTER);
				
				try {
					String spaceLine = getDocument().getText(startPosition, length);
					getDocument().insertString(getCaretPosition(), spaceLine, null);
				} catch (BadLocationException e1) {
					e1.printStackTrace();
				}
			}
		}
		
	};

	public int getNumberOfCurrentLine() {
		Element root = getDocument().getDefaultRootElement();
		return root.getElementIndex(getCaretPosition()) + 1;
	}
	
	public int getNumberOfCurrentColumn() {
		Element root = getDocument().getDefaultRootElement();
		Element element = root.getElement(root.getElementIndex(getCaretPosition()));		
		return getCaretPosition() - element.getStartOffset() + 1;
	}
	
}
