package neutrino.text;

import java.awt.Insets;
import java.awt.datatransfer.DataFlavor;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.InputEvent;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Stack;

import javax.swing.JEditorPane;
import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.KeyStroke;
import javax.swing.event.PopupMenuEvent;
import javax.swing.event.PopupMenuListener;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.Document;
import javax.swing.undo.CannotUndoException;
import javax.swing.undo.CompoundEdit;
import javax.swing.undo.UndoManager;

/**
 * Incapsulates text component.
 * Also supports undo and redo operations.
 * @author Oleh Radvanskyj
 * @version 1.0
 */
public class TextComponent extends JEditorPane {
	
	private UndoManager undoManager = null;
 	private Stack<CompoundEdit> edits = null;  	
	private int numberOfCurrentEdit = 0;	
	private JPopupMenu popupMenu = null;
	private boolean undoable = true;
	
	public TextComponent() {
		super();
		undoManager = new UndoManager();
		undoManager.setLimit(-1);
		edits = new Stack<CompoundEdit>();
		getDocument().addUndoableEditListener(undoableEditListener);
		final TextComponent instance = this;
		addMouseListener(new MouseAdapter() {
			@Override
			public void mouseReleased(MouseEvent e) {
				if (popupMenu != null && e.getButton() == e.BUTTON3) {
					popupMenu.show(instance, e.getX(), e.getY());
				}
			}
		});
		createDefaultPopupMenu();
	}
	
	protected void createDefaultPopupMenu() {
		// create menu items
		final JMenuItem pmiUndo = new JMenuItem("Undo");
		final JMenuItem pmiRedo = new JMenuItem("Redo");
		final JMenuItem pmiCut = new JMenuItem("Cut");
		final JMenuItem pmiCopy = new JMenuItem("Copy");
		final JMenuItem pmiPaste = new JMenuItem("Paste");
		final JMenuItem pmiSelectAll = new JMenuItem("Select all");
		// create popup menu
		popupMenu = new JPopupMenu();
		popupMenu.addPopupMenuListener(new PopupMenuListener() {
			@Override
			public void popupMenuWillBecomeVisible(PopupMenuEvent e) {
				pmiUndo.setEnabled(canUndo());
				pmiRedo.setEnabled(canRedo());
				pmiCut.setEnabled(canCut());
				pmiCopy.setEnabled(canCopy());
				pmiPaste.setEnabled(canPaste());
				pmiSelectAll.setEnabled(canSelectAll());
			}
			@Override
			public void popupMenuWillBecomeInvisible(PopupMenuEvent e) { }
			@Override
			public void popupMenuCanceled(PopupMenuEvent e) { }
		});
		// build popup menu
		popupMenu.add(pmiUndo);
		popupMenu.add(pmiRedo);
		popupMenu.addSeparator();		
		popupMenu.add(pmiCut);
		popupMenu.add(pmiCopy);
		popupMenu.add(pmiPaste);
		popupMenu.addSeparator();
		popupMenu.add(pmiSelectAll);
		// build mnemonics
		pmiUndo.setMnemonic(KeyEvent.VK_U);
		pmiRedo.setMnemonic(KeyEvent.VK_R);
		pmiCut.setMnemonic(KeyEvent.VK_C);
		pmiCopy.setMnemonic(KeyEvent.VK_O);
		pmiPaste.setMnemonic(KeyEvent.VK_P);
		pmiSelectAll.setMnemonic(KeyEvent.VK_S);
		// build accelerators
		pmiUndo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Z, InputEvent.CTRL_DOWN_MASK));
		pmiRedo.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Y, InputEvent.CTRL_DOWN_MASK));
		pmiCut.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_X, InputEvent.CTRL_DOWN_MASK));
		pmiCopy.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, InputEvent.CTRL_DOWN_MASK));
		pmiPaste.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_V, InputEvent.CTRL_DOWN_MASK));
		pmiSelectAll.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_A, InputEvent.CTRL_DOWN_MASK));
		// build actions
		ActionListener listener = new ActionListener() {
			@Override
			public void actionPerformed(ActionEvent e) {
				if (e.getSource() == pmiUndo) {
					undo();
				} else if (e.getSource() == pmiRedo) {
					redo();
				} else if (e.getSource() == pmiCut) {
					cut();
				} else if (e.getSource() == pmiCopy) {
					copy();
				} else if (e.getSource() == pmiPaste) {
					paste();
				} else if (e.getSource() == pmiSelectAll) {
					selectAll();
				}
			}
		};
		pmiUndo.addActionListener(listener);
		pmiRedo.addActionListener(listener);
		pmiCut.addActionListener(listener);
		pmiCopy.addActionListener(listener);
		pmiPaste.addActionListener(listener);
		pmiSelectAll.addActionListener(listener);
	}
	
	/**
	 * Sets popup menu for text component
	 * @param popupMenu - a popup menu
	 */
	public void setPopupMenu(JPopupMenu popupMenu) {
		this.popupMenu = popupMenu;
	}
	
	/**
	 * Notices that text is not changed when new document is set
	 * Prepares undo manager for set document
	 */
	public void setDocument(Document doc) {
		super.setDocument(doc);
		doc.addUndoableEditListener(undoableEditListener);
		reset(true);
		System.gc();
	}

	/**
	 * Returns true when all text may be selected
	 * @return boolean
	 */
	public boolean canSelectAll() {
		return !isTextEmpty() && !isAllTextSelected();
	}
	
	/**
	 * Returns true when text may be printed 
	 * @return boolean
	 */
	public boolean canPrint() {
		return !isTextEmpty();
	}
	
	/**
	 * Chack is text component is empty.
	 * @return true when text is empty.
	 */
	public boolean isTextEmpty() {
		return getDocument().getLength() == 0;
	}
	
	/**
	 * Returns true when all text is selected
	 * @return boolean
	 */
	public boolean isAllTextSelected() {
		if (!isTextSelected()) return false;
		else return (getSelectionStart() == 0) && (getSelectionLength() == getDocument().getLength());
	}

	/**
	 * Returns true if text is selected.
	 */
	public boolean isTextSelected() {
		return getSelectionStart() != getSelectionEnd();
	}
	
	/**
	 * Return length of selection or 0 if text is not selected. 
	 */
	public int getSelectionLength() {
		return getSelectionEnd() - getSelectionStart();
	}

	// undoable capability
	
	/**
	 * Sets undoable for text component
	 * {@value value - boolean}
	 */
	public void setUndoable(boolean value) {
		this.undoable = value;
	}
	
	/**
	 * Returns true when text component is undoable
	 * @return boolean
	 */
	public boolean isUndoable() {
		return this.undoable;
	}
	
	/**
	 * Chack is text is changed.
	 * @return true when text is changed.
	 */
	public boolean isTextChanged() {
		return numberOfCurrentEdit != 0;
	}	

	/**
	 * Returns true when text changes may be reverted
	 * @return boolean
	 */
	public boolean canRevert() {
		return undoable && isTextChanged();
	}	
	
	/**
	 * Returns true if the text change may be undone
	 */
	public boolean canUndo() {
		return undoable && undoManager.canUndo();
	}
	
	/**
	 * Returns true if the text change may be redone
	 */
	public boolean canRedo() {
		return undoable && undoManager.canRedo();
	}
	
	/**
	 * Undoes the last text change if possible. 
	 */
	public void undo() {
		if (undoable && undoManager.canUndo()) {
			undoManager.undo();
			numberOfCurrentEdit--;
		}
	}

	/**
	 * Redose the last text change if possible.
	 */
	public void redo() {
		if (undoable && undoManager.canRedo()) {
			undoManager.redo();
			numberOfCurrentEdit++;
		}
	}
	
 	/**
 	 * Returns true when compound edit is begun
 	 * @return boolean
 	 */
	public boolean isEditBegun() {
		return !edits.isEmpty();
	}
	
	/**
	 * Marks beginning of coalesce edit operation
	 */
	public void beginEdit() {
		if (!undoable) return;
		edits.push(new CompoundEdit());
	}
	
	/**
	 * Marks end of coalesce edit operation
	 */
	public void endEdit() {
		if (undoable) {
			if (!isEditBegun()) {
				return;
			}
			CompoundEdit currentEdit = edits.pop();
			if (isEditBegun()) {
				edits.peek().addEdit(currentEdit);
			} else {
				undoManager.addEdit(currentEdit);
			}
			currentEdit.end();
		}
		numberOfCurrentEdit++;
	}
	
	public void revert() {
		if (!undoable) return;
		try {
			while (isTextChanged()) {
				undoManager.undo();
				numberOfCurrentEdit--;
			}
		} catch (CannotUndoException e) {
			e.printStackTrace();
		}
	}
	
	/**
	 * Marks text as not changed
	 */
	public void reset(boolean discardAllEdits) {
		if (discardAllEdits) {
			if (undoManager != null) {
				undoManager.discardAllEdits();
			}
			if (edits != null) {
				edits.clear();
			}
		}
		numberOfCurrentEdit = 0;
	}
	
	private UndoableEditListener undoableEditListener = new UndoableEditListener() {
		
		public void undoableEditHappened(UndoableEditEvent e) {
			if (isEditBegun()) {
				if (undoable) edits.peek().addEdit(e.getEdit());
			} else {
				if (undoable) undoManager.addEdit(e.getEdit());
				numberOfCurrentEdit++;
			}
		}
		
	};

	/**
	 * Returns true if clipboard contain text.
	 * @throws IllegalStateException
	 * @return boolean value
	 */
	public boolean isClipboardContainText() throws IllegalStateException {
		for (DataFlavor dataFlavor : getToolkit().getSystemClipboard().getAvailableDataFlavors()) {
			if (dataFlavor.isFlavorTextType()) {
				return true;
			}
		}
		return false;
	}
	
	public void cut() {
		if (canCut()) super.cut();
	}
	
	public void copy() {
		if (canCopy()) super.copy();
	}
	
	public void paste() {
		try {
			if (!isClipboardContainText()) return;
		} catch (IllegalStateException e) {
			e.printStackTrace();
		}
		super.paste();
	}
	
	/**
	 * Returns true when text fragment may be cut
	 * @return boolean
	 */
	public boolean canCut() {
		return isTextSelected();
	}
	
	/**
	 * Returns true when text fragment may be copied
	 * @return boolean
	 */
	public boolean canCopy() {
		return isTextSelected();
	}
	
	/**
	 * Returns true when text fragment may be pasted
	 * @return boolean
	 */
	public boolean canPaste() {
		try {
			return isClipboardContainText();
		} catch (IllegalStateException e1) {
			return true;
		}
	}
	
}
