package neutrino.text;

import static javax.swing.JOptionPane.*;
import java.io.*;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Iterator;
import javax.swing.JEditorPane;
import javax.swing.JFileChooser;
import javax.swing.JTextArea;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.filechooser.FileFilter;
import javax.swing.text.BadLocationException;
import javax.swing.text.Document;
import javax.swing.text.EditorKit;
import javax.swing.text.JTextComponent;
import javax.swing.text.PlainDocument;


public class TextEditor {
	
	private JTextComponent textComponent = null;
	private File currentDirectory = null;
	private BackupManager backupManager = null;
	private File currentFile = null;
	private ArrayList<FileChangeListener> fileChangeListeners = null;
	private long lastModified = 0;
	private Charset currentEncoding = null;
	private ArrayList<FileFilter> choosableFileFilters = new ArrayList<FileFilter>(); 
	private FileFilter m_fileFilter = null;
	private boolean acceptAllFileFilterUsed = true;
	
	public TextEditor(JTextComponent textComponent) {
		this.textComponent = textComponent;
		fileChangeListeners = new ArrayList<FileChangeListener>();
		currentEncoding = Charset.defaultCharset();
		backupManager = new BackupManager(this);
		textComponent.getDocument().addDocumentListener(textChangedDocumentListener);
		addFileChangeListener(fileChangeListener);
	}
	
	public JTextComponent getTextComponent() {
		return this.textComponent;
	}	
	
	/**
	 * Returns true if file is ready for opening. Otherwise shows message dialog and returns false.
	 * @param file - the file
	 * @return boolean 
	 */
	private boolean isFileReadyForOpening(File file) {
		if (file == null) {
			try {
				throw new NullPointerException("File parameter is not initialized");
			} catch (NullPointerException e) {
				e.printStackTrace();
			}
			return false;
		} else if (file.exists() == false) {
			showMessageDialog(textComponent.getParent(), "File is not exist. \n Please verify the corrent file name was given", "Error", ERROR_MESSAGE);
			return false;
		} else if (file.isFile() == false) {
			showMessageDialog(textComponent.getParent(), file.getAbsolutePath() + " is not a file", "Error", ERROR_MESSAGE);
			return false;
		} else if (file.canRead() == false) {
			showMessageDialog(textComponent.getParent(), "File is protected from reading", "Error", ERROR_MESSAGE);
			return false;
		}
		return true;
	}
	
	/**
	 * Load text from file into text component. Returns true when success.
	 * precondition - file is ready for opening
	 * @param file - loaded file.
	 * @param encoding - encoding for reading.
	 * @return boolean
	 */
	private boolean openFile(File file, Charset encoding) {
		FileInputStream stream = null;
		InputStreamReader reader = null;
		boolean isSuccess;
		try {
			stream = new FileInputStream(file);
			if (textComponent instanceof JEditorPane) {
				EditorKit editorKit = ((JEditorPane) textComponent).getEditorKit();
				Document document = editorKit.createDefaultDocument();
				editorKit.read(stream, document, 0);
				textComponent.setDocument(document);
			} else {
				reader = new InputStreamReader(stream, encoding);
				textComponent.read(reader, null);
			}
			isSuccess = true;
		} catch (FileNotFoundException e) {
			showMessageDialog(textComponent.getParent(), "File not found", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} catch (IOException e) {
			showMessageDialog(textComponent.getParent(), "Cannot read file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} catch (BadLocationException e) {
			showMessageDialog(textComponent.getParent(), "Cannot read file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} finally {
			if (reader != null) {
				try {
					reader.close();
				} catch (IOException e) {
					showMessageDialog(textComponent.getParent(), "Cannot close file", "Error", ERROR_MESSAGE);
					e.printStackTrace();
					isSuccess = false;
				}
			}
		}
		return isSuccess;
	}

	/**
	 * Loads text file in text component using given encoding. Returns true when success
	 * @param file - file for loading
	 * @return boolean
	 */
	public boolean open(File file) {
		if (!isFileReadyForOpening(file)) {
			return false;
		}
		boolean isSuccess = openFile(file, currentEncoding);
		if (isSuccess) {
			setFile(file);
			fireFileChanged(FileChangeEventType.OPENING);
		}
		return isSuccess;
	}
	
	/**
	 * Loads file selected by user using given encoding. Returns true when file is opened
	 * @return boolean
	 */
	public boolean open() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setMultiSelectionEnabled(false);
		for (FileFilter fileFilter : choosableFileFilters) {
			fileChooser.addChoosableFileFilter(fileFilter);
		}
		if (m_fileFilter != null) fileChooser.setFileFilter(m_fileFilter);
		fileChooser.setAcceptAllFileFilterUsed(acceptAllFileFilterUsed);
		if (isFileLoaded()) 
			fileChooser.setCurrentDirectory(getFile().getParentFile());
		else 
			fileChooser.setCurrentDirectory(getCurrentDirectory());
		while (fileChooser.showOpenDialog(textComponent.getParent()) == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			if (!isFileReadyForOpening(file)) continue;
			if (!openFile(file, currentEncoding)) continue;
			setFile(file);
			fireFileChanged(FileChangeEventType.OPENING);
			return true;
		}
		return false;
	}
	
	/**
	 * Returns true if file is ready for saving. Otherwise show message dialog and return false.
	 * @param file - the file
	 * @return boolean
	 */
	private boolean isFileReadyForSaving(File file) {
		if (file == null) {
			try {
				throw new NullPointerException("File parameter is not initialized");
			} catch (NullPointerException e) {
				e.printStackTrace();
			}
			return false;
		} else if (file.exists() == true && file.isFile() == false) {
			showMessageDialog(textComponent.getParent(), file.getAbsoluteFile() + " is not a file. \n Please verify the corrent file name was given", "Error", ERROR_MESSAGE);
			return false;
		} else if (file.exists() == true && file.canWrite() == false) {
			showMessageDialog(textComponent.getParent(), "File is protected from writing", "Error", ERROR_MESSAGE);
			return false;
		}
		return true;
	}
	
	/**
	 * Save text in file. Returns true when success.
	 * precondition - file is ready for saving
	 * @param file - file for saving.
	 * @param encoding - encoding for writing.
	 * @return boolean
	 */ 
	private boolean saveFile(File file, Charset encoding) {
		FileOutputStream stream = null;
		OutputStreamWriter writer = null;
		boolean isSuccess;
		try {
			stream = new FileOutputStream(file);
			if (textComponent instanceof JEditorPane) {
				writer = new OutputStreamWriter(stream);
				EditorKit editorKit = ((JEditorPane) textComponent).getEditorKit();
				Document document = textComponent.getDocument();
				editorKit.write(stream, document, 0, document.getLength());
			} else {
				writer = new OutputStreamWriter(stream, encoding);
				textComponent.write(writer);
			}
			isSuccess = true;
		} catch (IOException e) {
			showMessageDialog(textComponent.getParent(), "Cannot write file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} catch (BadLocationException e) {
			showMessageDialog(textComponent.getParent(), "Cannot write file", "Error", ERROR_MESSAGE);
			e.printStackTrace();
			isSuccess = false;
		} finally {
			if (writer != null) {
				try {
					writer.close();
				} catch (IOException e) {
					showMessageDialog(textComponent.getParent(), "Cannot close file", "Error", ERROR_MESSAGE);
					e.printStackTrace();
					isSuccess = false;
				}
			}
		}
		return isSuccess;
	}

	/**
	 * Saves text in file using given encoding. Returns true when success
	 * @param file - file for saving
	 * @return boolean
	 */
	public boolean save(File file) {
		if (!isFileReadyForSaving(file)) {
			return false;
		}
		boolean isSuccess = saveFile(file, currentEncoding);
		if (isSuccess) {
			setFile(file);
			resetTextComponent();
			fireFileChanged(FileChangeEventType.SAVING);
		}
		return isSuccess;
	}
	
	private void resetTextComponent() {
		if (textComponent instanceof PlainTextArea) {
			((PlainTextArea) textComponent).reset(false);
		} else if (textComponent instanceof TextComponent) {
			((TextComponent) textComponent).reset(false);
		} else if (textComponent instanceof StyledTextComponent) {
			((StyledTextComponent) textComponent).reset(false);
		}
	}
	
	/**
	 * Save text in selected file. Returns true when file is saved
	 * @return boolean
	 */
	public boolean saveAs() {
		JFileChooser fileChooser = new JFileChooser();
		fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
		fileChooser.setMultiSelectionEnabled(false);
		for (FileFilter fileFilter : choosableFileFilters) {
			fileChooser.addChoosableFileFilter(fileFilter);
		}
		if (m_fileFilter != null) fileChooser.setFileFilter(m_fileFilter);
		fileChooser.setAcceptAllFileFilterUsed(acceptAllFileFilterUsed);
		fileChooser.setDialogTitle("Save As");
		if (isFileLoaded()) 
			fileChooser.setSelectedFile(getFile());
		else
			fileChooser.setCurrentDirectory(getCurrentDirectory());
		while (fileChooser.showSaveDialog(textComponent.getParent()) == JFileChooser.APPROVE_OPTION) {
			File file = fileChooser.getSelectedFile();
			if (!isFileReadyForSaving(file)) continue;
			if (file.exists() && file.isFile()) {
				if (showConfirmDialog(textComponent.getParent(), "Do want to replace file", "Saving confirmation", YES_NO_OPTION) == NO_OPTION) {
					continue;
				}
			}
			if (!saveFile(file, currentEncoding)) continue;
			setFile(file);
			resetTextComponent();
			fireFileChanged(FileChangeEventType.SAVING);
			return true;
		}
		return false;
	}
	
	/**
	 * Saves current file or selected file. Returns true when file is saved
	 * @return boolean
	 */
	public boolean save() {
		if (isFileLoaded()) {
			return save(getFile());
		} else {
			return saveAs();
		}
	}
	
	/**
	 * Clear text component.
	 */
	public void clear() {
		if (textComponent instanceof JTextArea) {
			PlainDocument document = new PlainDocument();
			textComponent.setDocument(document);
		} else if (textComponent instanceof JEditorPane) {
			EditorKit editorKit = ((JEditorPane) textComponent).getEditorKit();
			Document document = editorKit.createDefaultDocument();
			textComponent.setDocument(document);
		} else {
			textComponent.setText("");
		}
		setFile(null);
		fireFileChanged(FileChangeEventType.CLEARING);
	}
	
	public void reload() {
		if (isFileLoaded()) {
			open(getFile());
		}
	}
	
	/**
	 * Returns true when text may be cleared
	 * @return boolean
	 */
	public boolean canClear() {
		if (textComponent instanceof PlainTextArea) {
			return isFileLoaded() || !((PlainTextArea) textComponent).isTextEmpty() || ((PlainTextArea) textComponent).isTextChanged();
		} else if (textComponent instanceof TextComponent) {
			return isFileLoaded() || !((TextComponent) textComponent).isTextEmpty() || ((TextComponent) textComponent).isTextChanged();
		} else if (textComponent instanceof StyledTextComponent) {
			return isFileLoaded() || !((StyledTextComponent) textComponent).isTextEmpty() || ((StyledTextComponent) textComponent).isTextChanged();
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true when text may be saved
	 * @return boolean
	 */
	public boolean canSave() {
		if (textComponent instanceof PlainTextArea) {
			return ((PlainTextArea) textComponent).isTextChanged() || !isFileLoaded() || isFileChanged();
		} else if (textComponent instanceof TextComponent) {
			return ((TextComponent) textComponent).isTextChanged() || !isFileLoaded() || isFileChanged();
		} else if (textComponent instanceof StyledTextComponent) {
			return ((StyledTextComponent) textComponent).isTextChanged() || !isFileLoaded() || isFileChanged();
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true when text may be saved in another file
	 * @return boolean
	 */
	public boolean canSaveAs() {
		return isFileLoaded();
	}
	
	/**
	 * Returns true when file may be reloaded
	 * @return boolean
	 */
	public boolean canReload() {
		if (textComponent instanceof PlainTextArea) {
			return isFileChanged() || (isFileLoaded() && ((PlainTextArea) textComponent).isTextChanged());
		} else if (textComponent instanceof TextComponent) {
			return isFileChanged() || (isFileLoaded() && ((TextComponent) textComponent).isTextChanged());
		} else if (textComponent instanceof StyledTextComponent) {
			return isFileChanged() || (isFileLoaded() && ((StyledTextComponent) textComponent).isTextChanged());
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true when need to save file
	 * @return boolean
	 */
	public boolean conditionOfSaving() {
		if (textComponent instanceof PlainTextArea) {
			return (isFileLoaded() && ((PlainTextArea) textComponent).isTextChanged()) 
				|| (!isFileLoaded() && !((PlainTextArea) textComponent).isTextEmpty()) 
				|| isFileChanged();
		} else if (textComponent instanceof TextComponent) {
			return (isFileLoaded() && ((TextComponent) textComponent).isTextChanged()) 
				|| (!isFileLoaded() && !((TextComponent) textComponent).isTextEmpty()) 
				|| isFileChanged();
		} else if (textComponent instanceof StyledTextComponent) {
			return (isFileLoaded() && ((StyledTextComponent) textComponent).isTextChanged()) 
			|| (!isFileLoaded() && !((StyledTextComponent) textComponent).isTextEmpty()) 
			|| isFileChanged();
		} else {
			return true;
		}
	}
	
	/**
	 * Returns true when text is seved. Causes saving when need
	 * @return boolean
	 */
	public boolean isTextSaved() {
		if (conditionOfSaving()) {
			int option;
			if (isFileChanged()) {
				option = showConfirmDialog(textComponent.getParent(), "The file has been changed on the file system.\nDo you want to overwrite the changes?", "Update conflict", YES_NO_CANCEL_OPTION);
			} else {
				option = showConfirmDialog(textComponent.getParent(), "Do you want to save file?", "Saving confirmation", YES_NO_CANCEL_OPTION); 
			}
			switch (option) {
			case YES_OPTION:
				if (!save()) return false;
				break;
			case NO_OPTION:
				break;
			case CANCEL_OPTION:
				return false;
			}
		}
		return true;		
	}
			
	/**
	 * Get current edited file
	 * @return file
	 */
	public File getFile() {
		return this.currentFile;
	}
	
	private void setFile(File file) {
		currentFile = file;
		if (currentFile == null) { 
			lastModified = 0;
		} else {
			lastModified = file.lastModified();
		}
	}
	
	/**
	 * Returns true when current file modified outside of the program
	 * @return boolean
	 */
	public boolean isFileChanged() {
		if (!isFileLoaded()) return false;
		return currentFile.lastModified() != lastModified; 
	}
	
	/**
	 * Check is file for editing is loaded
	 * @return boolean
	 */
	public boolean isFileLoaded() {
		return currentFile != null;
	}
	
	/**
	 * Return current encoding used for io operations.
	 * @return
	 */
	public Charset getEncoding() {
		return this.currentEncoding;
	}
	
	/**
	 * Set current encoding used for io operations.
	 * @param encoding
	 */
	public void setEncoding(Charset encoding) {
		if (encoding == null) return;
		this.currentEncoding = encoding;
	} 
	
	/**
	 * Returns the backup manager for saving text changes
	 * @return BackupManager
	 */
	public BackupManager getBackupManager() {
		return this.backupManager;
	}
	
	/**
	 * Restores document from backup
	 * @param backup - File
	 */
	public void restore(File backup) {
		if (backup == null) return;
		if (!backup.isFile()) {
			showMessageDialog(textComponent.getParent(), backup.getAbsolutePath() + " - is not a file", "Error massage", ERROR_MESSAGE);
			return;
		}
		String fileName = null;
		Document doc = null;
		boolean isBackupReaded = true;
		ObjectInputStream stream = null;
		try {
			stream = new ObjectInputStream(new FileInputStream(backup));
			try {
				fileName = (String) stream.readObject();
				doc = (Document) stream.readObject();
			} catch (EOFException e) {
				isBackupReaded = false;
			} 
		} catch (FileNotFoundException e) {
			isBackupReaded = false;
			e.printStackTrace();
		} catch (IOException e) {
			isBackupReaded = false;
			e.printStackTrace();
		} catch (ClassNotFoundException e) {
			isBackupReaded = false;
			e.printStackTrace();
		} finally {
			if (stream != null) {
				try {
					stream.close();
				} catch (IOException e) {
					isBackupReaded = false;
					e.printStackTrace();
				}
			}
		}
		if (!isBackupReaded) {
			showMessageDialog(textComponent.getParent(), "Error of reading backup file", "Error massage", ERROR_MESSAGE);
			return;
		}
		if (fileName.equalsIgnoreCase(BackupManager.NONAME_FILE)) {
			setFile(null);
		} else {
			currentFile = new File(fileName);
			lastModified = 0;
		}
		textComponent.setDocument(doc);
		backup.delete();
		fireFileChanged(FileChangeEventType.RESTORING);
	}
	
	private FileChangeListener fileChangeListener = new FileChangeListener() {

		@Override
		public void fileChanged(FileChangeEvent event) {
			switch (event.getType()) {
			case OPENING:
			case CLEARING:
			case RESTORING:
			textComponent.getDocument().addDocumentListener(textChangedDocumentListener);
			getBackupManager().persistBackup();
			}
		}
		
	};
	
	private DocumentListener textChangedDocumentListener = new DocumentListener() {

		public void changedUpdate(DocumentEvent e) {
			getBackupManager().persistBackup();
		}

		public void insertUpdate(DocumentEvent e) {
			getBackupManager().persistBackup();
		}

		public void removeUpdate(DocumentEvent e) {
			getBackupManager().persistBackup();
		}
		
	};
	
	/**
	 * Returns current directory for save and open operations. 
	 * @return directory
	 */
	public File getCurrentDirectory() {
		return this.currentDirectory;
	}
	
	/**
	 * Sets current directory for save and open operations.
	 * @param dir - current directory
	 */
	public void setCurrentDirectory(File dir) {
		this.currentDirectory = dir;
	}
	
	
	/**
	 * Adds the specify file change listener to intercept file loading events.
	 * If paramater is null no action is perfomed. 
	 * @param listener - the file change listener.
	 */
	public void addFileChangeListener(FileChangeListener listener) {
		if (listener == null) return; 
		fileChangeListeners.add(listener);
	}
	
	/**
	 * Removes the specify file change listener.
	 * If paramater is null no action is perfomed. 
	 * @param listener - the file change listener.
	 */
	public void removeFileChangeListener(FileChangeListener listener) {
		if (listener == null) return; 
		fileChangeListeners.remove(listener);
	}
		
	private void fireFileChanged(FileChangeEventType type) {
		Iterator<FileChangeListener> iterator = fileChangeListeners.iterator();
		while (iterator.hasNext()) {
			FileChangeListener listener = iterator.next();
			FileChangeEvent event = new FileChangeEvent(this, type);
			listener.fileChanged(event);
		}
	} 
	
	public void addChoosableFileFilter(FileFilter fileFilter) {
		choosableFileFilters.add(fileFilter);
	}
	
	public boolean removeChoosableFileFilter(FileFilter fileFilter) {
		return choosableFileFilters.remove(fileFilter);
	}
	
	public void setFileFilter(FileFilter fileFilter) {
		this.m_fileFilter = fileFilter;
	}
	
	public void setAcceptAllFileFilterUsed(boolean value) {
		acceptAllFileFilterUsed = value;
	}
	
	public boolean isAcceptAllFileFilterUsed() {
		return acceptAllFileFilterUsed;
	}

}
