package com.neilmoomey.jpegbrowser2;



import javax.swing.*;

import javax.swing.tree.*;

import javax.swing.event.*;

import java.io.*;

import java.util.*;



/** FileTree.java

 * @author Neil Moomey, www.neilmoomey.com

 * @version 1.0

 **/



public class FileTree extends JTree {

	private String path;



	public FileTree(){

		super((TreeModel)null);



		// Horizontal and vertical line style

		putClientProperty("JTree.lineStyle", "Angled");

	}



	public void createTree(String path) throws FileNotFoundException, SecurityException {



 		// Create the first node

		FileTreeNode rootNode = new FileTreeNode(null, path);



		// Populate the root node with its subdirectories

		boolean addedNodes = rootNode.populateDirectories(true);

		setModel(new DefaultTreeModel(rootNode));



		// Listen for Tree Selection Events

		addTreeExpansionListener(new TreeExpansionHandler());



		// Set one click to expand folders instead of two

		setToggleClickCount(1);

	}



	// Returns the full pathname for a path, or null if not a known path

	public String getPathName(TreePath treePath) {

		Object o = treePath.getLastPathComponent();

		if (o instanceof FileTreeNode) {

			return ((FileTreeNode)o).file.getAbsolutePath();

		}

		return null;

	}



	// Returns the File for a path, or null if not a known path

	public File getFile(TreePath treePath) {

		Object o = treePath.getLastPathComponent();

		if (o instanceof FileTreeNode) {

			return ((FileTreeNode)o).file;

		}

		return null;

	}



	// Inner class that represents a node in this file system tree

	protected static class FileTreeNode extends DefaultMutableTreeNode {

		public FileTreeNode(File parent, String name) throws SecurityException, FileNotFoundException {

			this.name = name;



			// See if this node exists and whether it is a directory

			file = new File(parent, name);

			if (!file.exists()) {

				throw new FileNotFoundException("File " + name + " does not exist");

			}



			isDir = file.isDirectory();



			// Hold the File as the user object.

			setUserObject(file);

		}



		// Override isLeaf to check whether this is a directory

		public boolean isLeaf() {

			return !isDir;

		}



		// Override getAllowsChildren to check whether this is a directory

		public boolean getAllowsChildren() {

			return isDir;

		}



		// For display purposes, we return our own name

		public String toString() {

			return name;

		}



		// If we are a directory, scan our contents and populate

		// with children. In addition, populate those children

		// if the "descend" flag is true. We only descend once,

		// to avoid recursing the whole subtree.

		// Returns true if some nodes were added

		boolean populateDirectories(boolean descend) {

			boolean addedNodes = false;



			// Do this only once

			if (populated == false) {

				if (interim == true) {

					// We have had a quick look here before:

					// remove the dummy node that we added last time

					removeAllChildren();

					interim = false;

				}



				String[] names = file.list();		// Get list of contents

				Arrays.sort(names, String.CASE_INSENSITIVE_ORDER);



				// Process the directories

				for (int i = 0; i < names.length; i++) {

					String name = names[i];

					File d = new File(file, name);

					if (!d.isHidden()) {

					try {

						if (d.isDirectory()) {

							FileTreeNode node = new FileTreeNode(file, name);

							this.add(node);

							if (descend) {

								node.populateDirectories(false);

							}

							addedNodes = true;

							if (descend == false) {

								// Only add one node if not descending

								break;

							}

						}

					} catch (Throwable t) {

						// Ignore phantoms or access problems

					}

					}

				}



				// Now for files only

				for (int i = 0; i < names.length; i++) {

					String name = names[i];

					File d = new File(file, name);

					if (!d.isHidden()) {

					try {

						if (d.isFile()) {

							if (name.toLowerCase().endsWith(".jpg") || name.toLowerCase().endsWith(".jpeg") || name.toLowerCase().endsWith(".gif") || name.toLowerCase().endsWith(".png")) {

								FileTreeNode node = new FileTreeNode(file, name);

								this.add(node);

							}

						}

					} catch (Throwable t) {

						// Ignore phantoms or access problems

					}

					}

				}



				// If we were scanning to get all subdirectories,

				// or if we found no subdirectories, there is no

				// reason to look at this directory again, so

				// set populated to true. Otherwise, we set interim

				// so that we look again in the future if we need to

				if (descend == true || addedNodes == false) {

					populated = true;

				} else {

					// Just set interim state

					interim = true;

				}

			}

			return addedNodes;

		}



		protected File file;		// File object for this node

		protected String name;		// Name of this node

		protected boolean populated;// true if we have been populated

		protected boolean interim;	// true if we are in interim state

		protected boolean isDir;	// true if this is a directory

	}



	// Inner class that handles Tree Expansion Events

	protected class TreeExpansionHandler implements TreeExpansionListener {

		public void treeExpanded(TreeExpansionEvent evt) {

			TreePath treePath = evt.getPath();		// The expanded path

			JTree tree = (JTree)evt.getSource();	// The tree



			// Get the last component of the path and

			// arrange to have it fully populated.

			FileTreeNode node = (FileTreeNode)treePath.getLastPathComponent();

			if (node.populateDirectories(true)) {

				((DefaultTreeModel)tree.getModel()).nodeStructureChanged(node);

			}

		}



		public void treeCollapsed(TreeExpansionEvent evt) {

			// Nothing to do

		}

	}



	public static void main(String[] args) {

		// For testing only

		FileTree fileTree = new FileTree();
		JFrame f = new JFrame("File Tree Test");

		f.getContentPane().add(fileTree);

		f.setSize(220, 500);

		f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
		f.setVisible(true);



	}

}
