<?php 
/**
 * Radical Flash Chat
 *
 * LICENSE
 *
 * This source file is subject of Jaromir Sivic's ownership
 * and the commercial license on http://www.cze.cz.
 * It is not allowed to redistribute or modify source code
 * of this file. It is also not allowed to use any part
 * of the following Radical Chat source code in another projects.
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to radicalchat@cze.cz and you will be issued with a copy immediately.
 *
 * @category   Radical Chat
 * @copyright  Jaromir Sivic 2009
 * @license    http://www.cze.cz commercial license
 */
?>
<?php



class Configuration
{
	///////////////////////////////////////////////////////
	//             Configuration part
	// Administrator have to setup following properties
	///////////////////////////////////////////////////////
	
	
	/**
     * Path to the log file
     * @var sstring
     */
	public static $LOGFILE = "./logs/phplog.txt";
	
	/**
	 * Default encoding of operation system where php is running
	 * This is the iconv in charset value
	 * http://www.php.net/manual/en/function.iconv.php
	 * @var sstring
	 */
	public static $LOCAL_ENCODING = "utf-8";
	
	/**
	 * Unique id of this application
	 * Use only lower case characters
         * type random letters
	 * @var sstring
	 */
	public static $APP_UID = "g5dt7jk8qzs4g8df";
	
	/**
	 * Each computer which wants to reach this web service
	 * must pass following regular expression test.
	 * Obviously the variable containts IP address
	 * of the stream server and IP addresses
	 * of the developer computer, for example
	 * /(127\.0\.0\.1)|(192\.168\.0\.{1,3})/
	 * The test uses preg_match function
	 * http://www.php.net/manual/en/function.preg-match.php
	 * @var sstring
	 */
	public static $BIND_ADDRESSES_REGEXP = "/.*/";
	
	/**
	 * Maximum life time for sessions and persistent objects
	 */
	public static $SESSIONS_EXPIREMINUTES = 900;
	
	
	///////////////////////////////////////////////////////
	//          End of configuration part
	///////////////////////////////////////////////////////
	
	
	
	
	
	/**
	 * If configuration is prepared
	 * this value will be changed in prepare configuration
	 * @var unknown_type
	 */
	public static $PREPARED = false;
	
	/**
	 * Sequence unique id describes the order
	 * when this script has been started
	 * @var int
	 */
	public static $FIRST_SEQUENCE_UID = -1;	
	public static $ACTUAL_SEQUENCE_UID = -1;
	public static $LAST_BOOKED_SEQUENCE_UID = -1;
	
	/**
	 * How many sequence uids will be booked for each request
	 * @var unknown_type
	 */
	public static $SQUENCE_UIDS_TO_BOOK = 50;
	
	
	
	/**
	 * Test if application can started
	 */
	public static function canStart()
	{
		//test if ip address matches  $BIND_ADDRESSES_REGEXP
		//if does not then terminate this script
		if(preg_match(Configuration::$BIND_ADDRESSES_REGEXP,
		   $_SERVER["REMOTE_ADDR"]) == 0)
		{
		   	exit('Your computer is attempting to reach this '.
		   	     'web service from IP address '.$_SERVER["REMOTE_ADDR"].
		   	     ' which does not match the regular expression stored in '.
		   	     'Configuration::$BIND_ADDRESSES_REGEXP. For more '.
		   	     'informations please contact the administrator.');
		}
	}
	
	/**
	 * Prepare variables in configuration file
	 */
	public static function prepareConfiguration()
	{
		if(!Configuration::$PREPARED)
		{
			//test if application can start
			Configuration::canStart();
			//get sequence uid
			Configuration::$ACTUAL_SEQUENCE_UID = 
					Configuration::generateNextContinuousUID();
			//configuration is prepared now
			Configuration::$PREPARED = true;
			//set when sessions will expire
			ini_set('session.gc_maxlifetime',
					    Configuration::$SESSIONS_EXPIREMINUTES*60);
		}
	}
	
	/**
	 * Will generate next continuous UID
	 * @return int
	 */
	public static function generateNextContinuousUID()
	{
		if(Configuration::$FIRST_SEQUENCE_UID == -1)
		{
			//save old session id
			$oldid = session_id();
			//close old session and start new
			Configuration::startSession(Configuration::$APP_UID,10000);
			
			//get next uid
			if(!isset($_SESSION["NEXTUID"]))
			{
				$_SESSION["NEXTUID"] = 100000000;
			}
			Configuration::$FIRST_SEQUENCE_UID = $_SESSION["NEXTUID"];
			$_SESSION["NEXTUID"] += Configuration::$SQUENCE_UIDS_TO_BOOK;
			Configuration::$LAST_BOOKED_SEQUENCE_UID = $_SESSION["NEXTUID"]-1;
			Configuration::$ACTUAL_SEQUENCE_UID = 
					Configuration::$FIRST_SEQUENCE_UID;
			//close new session and start old
			Configuration::startSession($oldid);
			
			//return actual sequence uid
			return Configuration::$ACTUAL_SEQUENCE_UID;
		}
		else
		{
			Configuration::$ACTUAL_SEQUENCE_UID++;
			if(Configuration::$ACTUAL_SEQUENCE_UID >
					Configuration::$LAST_BOOKED_SEQUENCE_UID)
			{
				Logger::warn('You have reached maximum '.
				             'of booked sequence unique id. '.
				             'Please increase value '.
				             'Configuration::$HOW_MANY_SEQUENCE_UIDS_BOOK.');
			}
			return Configuration::$ACTUAL_SEQUENCE_UID;
		}
	}
	
	
	
	/**
	 * Return text representation of miliseconds from unix time
	 * @return double
	 */
	public static function getMicroTime()
	{
		/*$microTime = microtime();
		$arr = explode(" ",$microTime);
		$result = $arr[1] . substr($arr[0].'000',2,3);
		return ((float) $result)/1000;*/
		return microtime(true);
	}
	
	
	
	/**
	 * Forcefully start session and wait until the session is not started
	 * @param sstring $sessionID
	 */
	public static function startSession($sessionID, $expireMinutes = -1)
	{
		//close old session
		session_write_close();
		
		if(strlen($sessionID)>0)
		{
			//load configuration session
			session_id($sessionID);
			if($expireMinutes < 0)
			{
				$expireMinutes = Configuration::$SESSIONS_EXPIREMINUTES;
			}
			session_cache_expire($expireMinutes);
			//wait until session is opened
			if(!(@session_start()))
			{
				usleep(100);
			}
		}
	}
}

//prepare configuration
Configuration::prepareConfiguration();



class PersistentObject
{
	private $name;
	
	/**
	 * Array which contains user defined variables
	 * @example
	 * 		//simple page load counter<br />
	 * 		$po = new PersistentObject("persistentobject1");<br />
	 *		if(!isset($po->vars["property1"]))<br />
	 *		{<br />
	 *			$po->vars["property1"] = 0;<br />
	 *		}<br />
	 *		$po->vars["property1"]++;<br />
	 *		echo $po->vars["property1"];<br />
	 *		$po->save();<br />
	 * @var mixed
	 */
	public $vars;
	
	/**
	 * Create a persistent object
	 * @param $name - name of the object
	 * @param $expireMinutes - amount of minutes when the object
         * will be deleted by garbage collector
	 */
	public function PersistentObject($objectName,$expireMinutes = -1)
	{
		//save object name
		$this->name = $objectName;
		//save old session id
		$oldid = session_id();
		//load properties
		Configuration::startSession(Configuration::$APP_UID."po".
							        $this->name, $expireMinutes);
		$this->vars = $_SESSION["VARS"];
		//start old session
		Configuration::startSession($oldid);
	}
	
	/**
	 * Save all changes in the array vars
	 */
	public function save()
	{
		//save old session id
		$oldid = session_id();
		//save properties
		Configuration::startSession(Configuration::$APP_UID."po".
							        $this->name, $expireMinutes);
		$_SESSION["VARS"] = $this->vars;
		//start old session
		Configuration::startSession($oldid);
	}
}





class Mutex
{
	/**
	 * Name of the mutex used for the session
	 * @var unknown_type
	 */
	private $name;
	
	/**
	 * Id of this object
	 * @var unknown_type
	 */
	private $id;
	
	/**
	 * Constructor
	 * @param $mutexName Unique name of the mutex
	 */
	public function Mutex($mutexName)
	{
		$this->name = strtolower($mutexName);
		$this->id = Configuration::$FIRST_SEQUENCE_UID . rand(100000000,999999999);
	}
	
	
	
	/**
	 * Lock the mutex. Wait until the mutex is locked.
	 * @param int $autoUnlockMiliseconds If the mutex is locked and the
         * developer forget to unlock it, then the mutex will
         * automatically unlock itself after $autoUnlockMilliseconds
	 */
	public function lock($autoUnlockMiliseconds = 1000)
	{
		do
		{                          
			//try to lock the mutex
			$locked = $this->tryToLock($autoUnlockMiliseconds);
			//if mutex was not locked wait some time
			if(!$locked)
			{
				usleep(rand(10000,19999));
			}
			                            
		}while(!$locked);
	}
	
	
	
	/**
	 * Try to lock mutex, if succeed then return true
	 * otherwise return false
	 * @param $autoUnlockMiliseconds
	 * @return boolean
	 */
	public function tryToLock($autoUnlockMiliseconds = 1000)
	{
		//save id of the session
		$oldid = session_id();
		//start mutex session
		Configuration::startSession(Configuration::$APP_UID."mutex".
		                            $this->name,5);
		
		//save actual micro time
		$microTime = Configuration::getMicroTime();
		$result = true;
		//test if mutex is not owned by anyone else
		if((!(isset($_SESSION["MUTEX_OWNER_ID"]))) ||
		   ($_SESSION["MUTEX_OWNER_ID"] == $this->id) ||
		    ($_SESSION["AUTO_UNLOCK"]<$microTime))
		{
			//save that i am the owner of the mutex
			$_SESSION["MUTEX_OWNER_ID"] = $this->id;
			$_SESSION["AUTO_UNLOCK"] = $microTime + 
										($autoUnlockMiliseconds/1000);
		}
		//if mutex is owned by someone else
		else
		{
			$result = false;
		}
		
		//start old session
		Configuration::startSession($oldid);
		
		return $result;
	}
	
	
	
	/**
	 * Unlock the mutex
	 */
	public function unlock()
	{
		//save id of the session
		$oldid = session_id();
		//start mutex session
		Configuration::startSession(Configuration::$APP_UID."mutex".
		                            $this->name,5);
		
		//save actual micro time
		$microTime = Configuration::getMicroTime();
		//test if mutex is owned by me
		if((isset($_SESSION["MUTEX_OWNER_ID"])) &&
		   ($_SESSION["MUTEX_OWNER_ID"] == $this->id))
		{
			//release the mutex
			$_SESSION["AUTO_UNLOCK"] = 0.0;
		}
		
		//start old session
		Configuration::startSession($oldid);
	}
}


/**
 * Using this class you can log events inside web service class
 */
class Logger
{
    public static function loglog($value)
    {
        if(file_exists(Configuration::$LOGFILE))
        {
            $f = fopen(Configuration::$LOGFILE,"a+");
        }
        else
        {
            $f = fopen(Configuration::$LOGFILE,"w");
        }
        fputs($f,$value);
        fclose($f);
    }
	
	
    /**
     * Write a string into the logfile
     * @param $value
     * @param $prefix
     * @param $level
     */
    private static function log($value, $prefix, $level)
    {
        //test if file exists
        if(!file_exists(Configuration::$LOGFILE))
        {
                return;
        }

        //get actual date
        $date = date("Y-m-d H:i:s", time());

        $value = $level . ";" . $date .";" . $prefix . str_replace("\n","<br/>", $value);

        $f = fopen(Configuration::$LOGFILE,"a+");
        fputs($f,$value."\n");
        fclose($f);
    }
	


    /**
     * Write information message into the log file.
     * @param $value - message
     */
    public static function info($value)
    {
	Logger::log($value, "info; ",0);
    }
    
    
    
    /**
     * Write note message into the log file.
     * @param $value - message
     */
    public static function note($value)
    {
	Logger::log($value, "note; ",0);
    }
    
    
    
    /**
     * Write warning message into the log file.
     * @param $value - message
     */
    public static function warn($value)
    {
	Logger::log($value, "warn; ",0);
    }

    
    
    /**
     * Write error message into the log file.
     * @param $value - message
     */
    public static function error($value)
    {
	Logger::log($value, "error;",0);
    }
    
    
    /**
     * Write function header into the log file.
     * @param $value - message
     */
    public static function func($value)
    {
	Logger::log("-------------------------","func; ",0);
    	Logger::log($value, "func; ",0);
    	Logger::log("-------------------------","func; ",0);
    }
    
    
    
    /**
     * Clear the log file
     */
    public static function logClear()
    {
        fclose(fopen(Configuration::$LOGFILE,"w+"));
    }
}
    
?>