/*
 * Copyright (C) 2005 Luca Veltri - University of Parma - Italy
 * 
 * This source code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Author(s):
 * Luca Veltri (luca.veltri@unipr.it)
 * 
 * Greg Dorfuss 2009/02/16 patch for registration to match rfc3261 - uses same callid and unique cseq.
 * Greg Dorfuss 2009/03/28 patch keepalive variable not getting set
 * Greg Dorfuss 2009/04/21 added 407 proxy authentication
 */

package local.ua;


import local.net.KeepAliveSip;

import org.apache.log4j.Logger;
import org.zoolu.net.SocketAddress;
import org.zoolu.sip.address.*;
import org.zoolu.sip.provider.SipStack;
import org.zoolu.sip.provider.SipProvider;
import org.zoolu.sip.header.*;
import org.zoolu.sip.message.*;
import org.zoolu.sip.transaction.TransactionClient;
import org.zoolu.sip.transaction.TransactionClientListener;
import org.zoolu.sip.authentication.DigestAuthentication;

import java.util.Vector;


/** Register User Agent.
  * It registers (one time or periodically) a contact address with a registrar server.
  */
public class RegisterAgent extends Thread implements TransactionClientListener
{
   /** Max number of registration attempts. */
   static final int MAX_ATTEMPTS=3;

   /** RegisterAgent listener */
   RegisterAgentListener listener;
   
   /** SipProvider */
   SipProvider sip_provider;

   /** User's URI with the fully qualified domain name of the registrar server. */
   NameAddress target;

   /** User name. */
   String username;

   /** User name. */
   String realm;

   /** User's passwd. */
   String passwd;

   /** Nonce for the next authentication. */
   String next_nonce;

   /** Qop for the next authentication. */
   String qop;

   /** User's contact address. */
   NameAddress contact; 

   /** Expiration time. */
   int expire_time;

   /** Renew time. */
   int renew_time;
   int orig_renew_time;
   
   int minRenewTime=30;
   int regFailRetryTime=15;

   /** Whether keep on registering. */
   boolean loop;
   
   boolean lastRegFailed=false;
   boolean regInprocess=false;

   /** Whether the thread is running. */
   boolean is_running;

   /** Event logger. */
   private Logger log = Logger.getLogger(this.getClass());

   /** Number of registration attempts. */
   int attempts;
   
   /** KeepAliveSip daemon. */
   KeepAliveSip keep_alive;

   /** Creates a new RegisterAgent. */
   /*
   public RegisterAgent(SipProvider sip_provider, RegisterAgentListener listener,UserAgentProfile uaprof)
   {  
	   init(sip_provider,uaprof.from_url,uaprof.contact_url,listener,uaprof);
   }
   */
   
   /** Creates a new RegisterAgent with authentication credentials (i.e. username, realm, and passwd). */
   public RegisterAgent(SipProvider sip_provider, RegisterAgentListener listener,UserAgentProfile uaprof)
   {  
	  this.setName(this.getClass().getName()+".T"+this.getName().replaceAll("Thread-", "")); 
      //if no contact_url and/or from_url has been set, create it now
	  uaprof.initContactAddress(sip_provider); 
	   
	  init(sip_provider,uaprof.from_url,uaprof.contact_url,listener,uaprof);
      // authentication
      this.username=uaprof.username;
      this.realm=uaprof.realm;
      this.passwd=uaprof.passwd;
   }

   /** Inits the RegisterAgent. */
   private void init(SipProvider sip_provider, String target_url, String contact_url, RegisterAgentListener listener,UserAgentProfile uaprof)
   {  this.listener=listener;
      this.sip_provider=sip_provider;
      this.target=new NameAddress(target_url);
      this.contact=new NameAddress(contact_url);
      this.expire_time=SipStack.default_expires;
      this.renew_time=uaprof.expires/2;
      this.orig_renew_time=this.renew_time;
      this.is_running=false;
      this.keep_alive=null;
      // authentication
      this.username=null;
      this.realm=null;
      this.passwd=null;
      this.next_nonce=null;
      this.qop=null;
      this.attempts=0;
      this.minRenewTime=uaprof.minRegRenewTime;
      this.regFailRetryTime=uaprof.regFailRetryTime;
   }


   /** Whether it is periodically registering. */
   public boolean isRegistering()
   {  return is_running;
   }


   /** Registers with the registrar server. */
   public void register()
   {  register(expire_time);
   }


   /** Registers with the registrar server for <i>expire_time</i> seconds. */
   private CallIdHeader registerCallID=null;
   private int registerCSeq=1;
   public void register(int expire_time)
   {  
	  attempts=0;
      lastRegFailed=false;
      regInprocess=true;
      if (expire_time>0) this.expire_time=expire_time;
      Message req=MessageFactory.createRegisterRequest(sip_provider,target,target,contact);
      if (registerCallID == null)
          registerCallID = req.getCallIdHeader();
      else
          req.setCallIdHeader(registerCallID);
      req.setCSeqHeader(new CSeqHeader(registerCSeq++, SipMethods.REGISTER));
      
      req.setExpiresHeader(new ExpiresHeader(String.valueOf(expire_time)));
      if (next_nonce!=null)
      {  AuthorizationHeader ah=new AuthorizationHeader("Digest");
         SipURL target_url=target.getAddress();
         ah.addUsernameParam(username);
         ah.addRealmParam(realm);
         ah.addNonceParam(next_nonce);
         ah.addUriParam(req.getRequestLine().getAddress().toString());
         ah.addQopParam(qop);
         String response=(new DigestAuthentication(SipMethods.REGISTER,ah,null,passwd)).getResponse();
         ah.addResponseParam(response);
         req.setAuthorizationHeader(ah);
      }
      if (expire_time>0) log.debug("Registering contact "+contact+" (it expires in "+expire_time+" secs)");
      else log.debug("Unregistering contact "+contact);
      TransactionClient t=new TransactionClient(sip_provider,req,this);
      t.request(); 
   }


   /** Unregister with the registrar server */
   public void unregister()
   {  register(0);
   } 


   /** Unregister all contacts with the registrar server */
   public void unregisterall()
   {  attempts=0;
      NameAddress user=new NameAddress(target);
      Message req=MessageFactory.createRegisterRequest(sip_provider,target,target,null);
      //ContactHeader contact_star=new ContactHeader(); // contact is *
      //req.setContactHeader(contact_star);
      req.setExpiresHeader(new ExpiresHeader(String.valueOf(0)));
      log.debug("Unregistering all contacts");
      TransactionClient t=new TransactionClient(sip_provider,req,this); 
      t.request(); 
   }


   /** Periodically registers with the registrar server.
     * @param expire_time expiration time in seconds
     * @param renew_time renew time in seconds */
   public void loopRegister(int expire_time, int renew_time)
   {  this.expire_time=expire_time;
      this.renew_time=renew_time;
      loop=true;
      //if (!is_running) (new Thread(this,this.getClass().getName())).start();
      if (!is_running)
    	  start();
   }


   /** Periodically registers with the registrar server.
     * @param expire_time expiration time in seconds
     * @param renew_time renew time in seconds
     * @param keepalive_time keep-alive packet rate (inter-arrival time) in milliseconds */
   public void loopRegister(int expire_time, int renew_time, long keepalive_time)
   {  loopRegister(expire_time,renew_time);
      // keep-alive
      if (keepalive_time>0)
      {  SipURL target_url=target.getAddress();
         String target_host=target_url.getHost();
         int targe_port=target_url.getPort();
         if (targe_port<0) targe_port=SipStack.default_port;
         keep_alive=new KeepAliveSip(sip_provider,new SocketAddress(target_host,targe_port),null,keepalive_time);
      }
   }


   /** Halts the periodic registration. */
   public void halt()
   {  
	  if (is_running) loop=false;
      if (keep_alive!=null)
    	  keep_alive.halt();
   }

   public void haltRA()
   {
	   halt();
	   listener=null;
	   sip_provider=null;
	   interrupt();
   }
   
   
   // ***************************** run() *****************************

   /** Run method */
   public void run()
   {  
      is_running=true;
      try
      {  while (loop)
         {  
    	    register();
    	    long waitCnt=0;
    	    while (regInprocess)
    	    {	
    	    	sleep(1000);
    	    	waitCnt+=1000;
    	    }

    	    if (lastRegFailed)
    	    {
    	    	log.warn("Failed Registration retrying in 15 seconds.");
	    		sleep(regFailRetryTime*1000);
    	    }	
    	    else	
    	    	sleep(renew_time*1000-waitCnt);
         }
      }
      catch (InterruptedException e)
      {
    	 if (loop)
    		 e.printStackTrace();
      }
      catch (Exception e) {  log.error("error",e);  }
      is_running=false;
   }

   
   // **************** Transaction callback functions *****************

   /** Callback function called when client sends back a failure response. */

   /** Callback function called when client sends back a provisional response. */
   public void onTransProvisionalResponse(TransactionClient transaction, Message resp)
   {  // do nothing..
   }

   /** Callback function called when client sends back a success response. */
   public void onTransSuccessResponse(TransactionClient transaction, Message resp)
   {  
	  if (transaction.getTransactionMethod().equals(SipMethods.REGISTER))
      {  
		 if (resp.hasAuthenticationInfoHeader())
         {  
			 next_nonce=resp.getAuthenticationInfoHeader().getNextnonceParam();
         }
         
		 StatusLine status=resp.getStatusLine();
         String result=status.getCode()+" "+status.getReason();
         
         // update the renew_time
         int newRenew=0;
         if (resp.hasExpiresHeader())
         {  
        	 newRenew=resp.getExpiresHeader().getDeltaSeconds()/2;
         }
         else if (resp.hasContactHeader())
         {  
        	// look for the max expires - should be last one
        	Vector contacts=resp.getContacts().getHeaders();
            if (contacts.size()>0)
            	newRenew=(new ContactHeader((Header)contacts.elementAt(contacts.size()-1))).getExpires()/2;
         }
         
         if (newRenew>0 && newRenew<renew_time)
         {	 
        	 renew_time=newRenew;
        	 if (renew_time<minRenewTime)
        	 {	 
        		 log.warn("Attempt to set renew time below min renew. Attempted="+renew_time+" min="+minRenewTime+"\r\nResponse="+resp.toString());
        		 renew_time=minRenewTime;
        	 }
         }
         else if (newRenew>orig_renew_time)
         {	 
       		 log.warn("Attempt to set renew time above original renew. Attempted="+newRenew+" origrenew="+orig_renew_time+"\r\nResponse="+resp.toString());
         }

         
         regInprocess=false;
         if (listener!=null) 
        	 listener.onUaRegistrationSuccess(this,target,contact,result);
         else
             log.info("Registration success: "+result);
      }
   }

   /** Callback function called when client sends back a failure response. */
   public void onTransFailureResponse(TransactionClient transaction, Message resp)
   {  if (transaction.getTransactionMethod().equals(SipMethods.REGISTER))
      {  StatusLine status=resp.getStatusLine();
         int code=status.getCode();
         if (
        	 (code==401 && attempts<MAX_ATTEMPTS && resp.hasWwwAuthenticateHeader() && resp.getWwwAuthenticateHeader().getRealmParam().equalsIgnoreCase(realm))
          || (code==407 && attempts<MAX_ATTEMPTS && resp.hasProxyAuthenticateHeader() && resp.getProxyAuthenticateHeader().getRealmParam().equalsIgnoreCase(realm))		
         )
         {  attempts++;
           
            log.info(code+" Registration Authentication");
            
         	Message req=transaction.getRequestMessage();
            req.setCSeqHeader(new CSeqHeader(registerCSeq++, SipMethods.REGISTER));
            
            WwwAuthenticateHeader wah;
            if (code==401) 
           	 wah=resp.getWwwAuthenticateHeader();
            else 
           	 wah=resp.getProxyAuthenticateHeader();
            String qop_options=wah.getQopOptionsParam();
            //log.debug("DEBUG: qop-options: "+qop_options);
            // select a new branch - rfc3261 says should be new on each request
            ViaHeader via=req.getViaHeader();
            req.removeViaHeader();
            via.setBranch(SipProvider.pickBranch());
            req.addViaHeader(via);
            qop=(qop_options!=null)? "auth" : null;
            /*
            AuthorizationHeader ah=(new DigestAuthentication(SipMethods.REGISTER,req.getRequestLine().getAddress().toString(),wah,qop,null,username,passwd)).getAuthorizationHeader();
            req.setAuthorizationHeader(ah);
            */
            
            DigestAuthentication digest=new DigestAuthentication(SipMethods.REGISTER,req.getRequestLine().getAddress().toString().toString(),wah,qop,null,username,passwd);
            AuthorizationHeader ah;
            if (code==401) 
           	 ah=digest.getAuthorizationHeader();
            else 
           	 ah=digest.getProxyAuthorizationHeader();
            req.setAuthorizationHeader(ah);
            
            TransactionClient t=new TransactionClient(sip_provider,req,this);
            t.request();
            
         }
         else
         {  String result=code+" "+status.getReason();
         	lastRegFailed=true;
         	regInprocess=false;
         	if (listener==null)
         		log.error("Registration failure: "+result);
         	else
         		log.debug("Registration failure: "+result);
            if (listener!=null) listener.onUaRegistrationFailure(this,target,contact,result);
         }
      }
   }

   /** Callback function called when client expires timeout. */
   public void onTransTimeout(TransactionClient transaction)
   {  if (transaction.getTransactionMethod().equals(SipMethods.REGISTER))
      {  
	     if (listener==null)
	    	 log.error("Registration failure: No response from server.");
	     else
	    	 log.debug("Registration failure: No response from server.");
      	 lastRegFailed=true;
      	 regInprocess=false;
         if (listener!=null) listener.onUaRegistrationFailure(this,target,contact,"Timeout");
      }
   }

}
