/*
 * Copyright (C) 2005 Luca Veltri - University of Parma - Italy
 * 
 * This source code is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Author(s):
 * Luca Veltri (luca.veltri@unipr.it)
 */

/*
 * Greg Dorfuss - http://mhspot.com 2008/03/01-2008/09/10
 * Added RFC2833 support
 * Added resetLocalSession - it was taking on previous incoming call origins and using it as it's own for outgoing.
 * remove username from origin owner
 * Added hold support sort of
 * Added support for DTMF over INFO
 * added ability for multiple codecs
 * 
 * Greg Dorfuss - http://mhspot.com 2008/11/30
 * Added ability for call to override user,password and realm in target url
 * Greg Dorfuss - 2009/04/25
 * Added ability to return public ip in rtp connection field if remote is not local address
 */

package local.ua;


import local.media.AudioClipPlayer;
import java.net.InetAddress;

import org.apache.log4j.Logger;
import org.zoolu.sip.call.*;
import org.zoolu.sip.address.*;
import org.zoolu.sip.provider.SipProvider;
import org.zoolu.sip.header.StatusLine;
import org.zoolu.sip.message.*;
import org.zoolu.sdp.*;
import org.zoolu.tools.Parser;

import java.util.Enumeration;
import java.util.Vector;
import java.util.regex.*;

/** Simple SIP user agent (UA).
  * It includes audio/video applications.
  * <p>
  * It can use external audio/video tools as media applications.
  * Currently only RAT (Robust Audio Tool) and VIC are supported as external applications.
  */
public class UserAgent extends CallListenerAdapter
{           
   /** Event logger. */
   protected Logger log = Logger.getLogger(this.getClass());

   /** UserAgentProfile */
   protected UserAgentProfile user_profile;

   /** SipProvider */
   protected SipProvider sip_provider;

   /** Call */
   //Call call;
   protected ExtendedCall call=null;

   /** Call transfer */
   protected ExtendedCall call_transfer;
   
   /** Audio application */
   protected MediaLauncher audio_app=null;
   /** Video application */
   protected MediaLauncher video_app=null;
   
   /** Local sdp */
   protected String local_session=null;
   protected int rtpPort=0;

   /** UserAgent listener */
   protected UserAgentListener listener=null;

   /** Media file path */
   final String MEDIA_PATH="media/local/ua/";

   /** On wav file */
   final String CLIP_ON=MEDIA_PATH+"on.wav";
   /** Off wav file */
   final String CLIP_OFF=MEDIA_PATH+"off.wav";
   /** Ring wav file */
   final String CLIP_RING=MEDIA_PATH+"ring.wav";
   
   /** Ring sound */
   AudioClipPlayer clip_ring;
   /** On sound */
   AudioClipPlayer clip_on;
   /** Off sound */
   AudioClipPlayer clip_off;
   
   protected String dtmfBuffer="";
   
   protected boolean onRemoteSipHold=false;
   protected boolean onLocalSipHold=false; // not implemented

   private String dtmf2833RtpParams="telephone-event/8000";
   private String dtmf2833FmtpParams="0-15";
   protected int dtmf2833PayloadType_session=-1;


   // *********************** Startup Configuration ***********************   
     
   /** UA_IDLE=0 */
   static final String UA_IDLE="IDLE";
   /** UA_INCOMING_CALL=1 */
   static final String UA_INCOMING_CALL="INCOMING_CALL";
   /** UA_OUTGOING_CALL=2 */
   static final String UA_OUTGOING_CALL="OUTGOING_CALL";
   /** UA_ONCALL=3 */
   static final String UA_ONCALL="ONCALL";

   /** Call state
     * <P>UA_IDLE=0, <BR>UA_INCOMING_CALL=1, <BR>UA_OUTGOING_CALL=2, <BR>UA_ONCALL=3 */
   String call_state=UA_IDLE;
   


   // *************************** Basic methods ***************************   

   /** Changes the call state */
   protected void changeStatus(String state)
   {  
      call_state=state;
      //log.debug("state: "+call_state); 
   }

   /** Checks the call state */
   protected boolean statusIs(String state)
   {  return call_state.equals(state); 
   }

   /** Gets the call state */
   protected String getStatus()
   {  return call_state; 
   }
   
   /** Sets the automatic answer time (default is -1 that means no auto accept mode) */
   public void setAcceptTime(int accept_time)
   {  user_profile.accept_time=accept_time; 
   }

   /** Sets the automatic hangup time (default is 0, that corresponds to manual hangup mode) */
   public void setHangupTime(int time)
   {  user_profile.hangup_time=time; 
   }
      
   /** Sets the redirection url (default is null, that is no redircetion) */
   public void setRedirection(String url)
   {  user_profile.redirect_to=url; 
   }
      
   /** Sets the no offer mode for the invite (default is false) */
   public void setNoOfferMode(boolean nooffer)
   {  user_profile.no_offer=nooffer;
   }

   /** Enables audio */
   public void setAudio(boolean enable)
   {  user_profile.audio=enable;
   }

   /** Enables video */
   public void setVideo(boolean enable)
   {  user_profile.video=enable;
   }
   
   /** Sets the receive only mode */
   public void setReceiveOnlyMode(boolean r_only)
   {  user_profile.recv_only=r_only;
   }

   /** Sets the send only mode */
   public void setSendOnlyMode(boolean s_only)
   {  user_profile.send_only=s_only;
   }

   /** Sets the send tone mode */
   public void setSendToneMode(boolean s_tone)
   {  user_profile.send_tone=s_tone;
   }

   /** Sets the send file */
   public void setSendFile(String file_name)
   {  user_profile.send_file=file_name;
   }

   /** Sets the recv file */
   public void setRecvFile(String file_name)
   {  user_profile.recv_file=file_name;
   }

   /** Gets the local SDP */
   public String getSessionDescriptor()
   {  return local_session;
   }   

   /** Sets the local SDP */
   public void setSessionDescriptor(String sdp)
   {  local_session=sdp;
   }

   /** Inits the local SDP (no media spec) */
   public void initSessionDescriptor()
   {  SessionDescriptor sdp=new SessionDescriptor(user_profile.username,sip_provider.getViaAddress());
      local_session=sdp.toString();
   }

   /** Adds a media to the SDP */
   public void addMediaDescriptor(String media, int port, int avp, String codec, int rate)
   {  
	  if (local_session==null) initSessionDescriptor();
      SessionDescriptor sdp=new SessionDescriptor(local_session);
      String attr_param=String.valueOf(avp);
      if (codec!=null) attr_param+=" "+codec+"/"+rate;
      
      sdp.addMedia(new MediaField(media,port,0,"RTP/AVP",String.valueOf(avp)),new AttributeField("rtpmap",attr_param));
      local_session=sdp.toString();
   }

   public void addMediaDescriptor(String media, int port, int[] avp, String[] codec, int[] rate)
   {  
	  if (local_session==null) initSessionDescriptor();
      
	  SessionDescriptor sdp=setupMedia(new SessionDescriptor(local_session),media,port,avp,codec,rate);
      
      local_session=sdp.toString();
   }

   private SessionDescriptor setupMedia(SessionDescriptor sdp,String media, int port, int[] avp, String[] codec, int[] rate)
   {
	      Vector<AttributeField> af=new Vector<AttributeField>();
	      Vector<String> mf=new Vector<String>();
	      for (int c=0;c<avp.length;c++)
	      {	  
	          String attr_param=String.valueOf(avp[c]);
	          if (codec!=null) attr_param+=" "+codec[c]+"/"+rate[c];
	    	  af.add(new AttributeField("rtpmap",attr_param));
	    	  mf.add(String.valueOf(avp[c]));
	      }
	      if (media.equals("audio"))
	      {
	    	  if (user_profile.dtmf2833PayloadType>0)
	    	  {	  
	    		  mf.add(String.valueOf(user_profile.dtmf2833PayloadType));
	              af.add(new AttributeField("rtpmap",this.user_profile.dtmf2833PayloadType+" "+dtmf2833RtpParams));
	              af.add(new AttributeField("fmtp",this.user_profile.dtmf2833PayloadType+" "+dtmf2833FmtpParams));
	    	  }

	          af.add(new AttributeField("sendrecv"));
	          af.add(new AttributeField("silenceSupp","off"));

	      }
	      sdp.addMedia(new MediaField(media,port,0,"RTP/AVP",mf),af);
	      
	      return sdp;
   }
   
   private void resetLocalSession()
   {
	  // set local sdp
	  dtmf2833PayloadType_session=this.user_profile.dtmf2833PayloadType; // set to default
      initSessionDescriptor();
      
      if (user_profile.audio || !user_profile.video)
    	  addMediaDescriptor("audio",this.rtpPort,user_profile.audio_avp,user_profile.audio_codec,user_profile.audio_sample_rate);

      if (user_profile.video) addMediaDescriptor("video",user_profile.video_port,user_profile.video_avp,null,0);    

      //log.debug("*** localSdp:"+local_session.toString());
      
   } 

 
   private SessionDescriptor selectOptionalParams(String strSdp,String remotesdp)
   {
	   SessionDescriptor sdp=new SessionDescriptor(strSdp);

	   MediaDescriptor md=sdp.getMediaDescriptor("audio");
	   
	   if (md==null)
		   return sdp;
	   
       /* no longer needed
        if (user_profile.dtmf2833PayloadType>0)
        {	
	        if (remotesdp.toLowerCase().indexOf(dtmf2833RtpParams)>=0 && strSdp.toLowerCase().indexOf(dtmf2833RtpParams)<0)
	        {
			   md.getMedia().appendFormat(String.valueOf(user_profile.dtmf2833PayloadType));
			   md.addAttribute(new AttributeField("rtpmap",this.user_profile.dtmf2833PayloadType+" "+dtmf2833RtpParams));
			   md.addAttribute(new AttributeField("fmtp",this.user_profile.dtmf2833PayloadType+" "+dtmf2833FmtpParams));
	        }
	        else
	        {	
	        	md.getMedia().removeFormat(Integer.toString(this.user_profile.dtmf2833PayloadType));
	        }
        }
       */ 
        
	   md.addAttribute(new AttributeField("sendrecv"));
	   md.addAttribute(new AttributeField("silenceSupp","off"));
        
       return sdp;
   }
   
   private SessionDescriptor selectCodec(String str_local_sdp,String str_remote_sdp,boolean remoteHasPriority)
   {
	   SessionDescriptor local_sdp=new SessionDescriptor(str_local_sdp);
	   SessionDescriptor remote_sdp=new SessionDescriptor(str_remote_sdp);
       SessionDescriptor new_sdp=new SessionDescriptor(remote_sdp.getOrigin(),remote_sdp.getSessionName(),local_sdp.getConnection(),local_sdp.getTime());
       new_sdp.addMediaDescriptors(local_sdp.getMediaDescriptors()); // puts in local descriptors
       if (remoteHasPriority)
    	   new_sdp=SdpTools.selectMedia(new_sdp,remote_sdp,local_sdp);
       else 
    	   new_sdp=SdpTools.selectMedia(new_sdp,local_sdp,remote_sdp);
       
       String testMatch=new_sdp.toString().replaceAll("a=rtpmap:.*telephone-event", "");
       if (testMatch.indexOf("a=rtpmap:")<0)
       {
      	 log.error("Failed to select RTP format");
	     log.error("### local descriptor="+local_sdp.toString());
	     log.error("### remote descriptor="+remote_sdp.toString());
	     log.debug("### mjsipAfterSelect="+new_sdp.toString());
      	 return null;
       }
       else
	       	 new_sdp=selectOptionalParams(new_sdp.toString(),remote_sdp.toString());

       log.debug("### selectCodec mjsipAfterSelect="+new_sdp.toString());
	   
     return new_sdp;	 
   }
   
   // *************************** Public Methods **************************

   /** Costructs a UA with a default media port */
   public UserAgent(SipProvider sip_provider, UserAgentProfile user_profile, UserAgentListener listener,int argRtpPort)
   {  
	  this.sip_provider=sip_provider;
      this.listener=listener;
      this.user_profile=user_profile;
      this.rtpPort=argRtpPort;
      
      // if no contact_url and/or from_url has been set, create it now
      user_profile.initContactAddress(sip_provider);

      // load sounds  

      // ################# patch to make audio working with javax.sound.. #################
      // currently AudioSender must be started before any AudioClipPlayer is initialized,
      // since there is a problem with the definition of the audio format
     
      /*
      if (!user_profile.use_rat && !user_profile.use_jmf)
      {  if (user_profile.audio && !user_profile.recv_only && user_profile.send_file==null && !user_profile.send_tone) local.media.AudioInput.initAudioLine();
         if (user_profile.audio && !user_profile.send_only && user_profile.recv_file==null) local.media.AudioOutput.initAudioLine();
      }
      */
      
      // ################# patch to make rat working.. #################
      // in case of rat, do not load and play audio clips
      /* I don't need this 
      if (!user_profile.use_rat)
      {  try
         {  String jar_file=user_profile.ua_jar;
            clip_on=new AudioClipPlayer(Archive.getAudioInputStream(Archive.getJarURL(jar_file,CLIP_ON)),null);
            clip_off=new AudioClipPlayer(Archive.getAudioInputStream(Archive.getJarURL(jar_file,CLIP_OFF)),null);
            clip_ring=new AudioClipPlayer(Archive.getAudioInputStream(Archive.getJarURL(jar_file,CLIP_RING)),null);
         }
         catch (Exception e)
         {  log.error("error",e);
         }
         //clip_ring=new AudioClipPlayer(CLIP_RING,null);
         //clip_on=new AudioClipPlayer(CLIP_ON,null);
         //clip_off=new AudioClipPlayer(CLIP_OFF,null);
      }
      */
      
      resetLocalSession();
      
   }
   
   public String getCapabilities()
   {
	   //called by sipprovider when a SIP OPTIONS message received
	   
	   //build the caps like an invite
	   String optBody="";
	   
	   SessionDescriptor sdp=setupMedia(new SessionDescriptor(user_profile.username,sip_provider.getViaAddress()),"audio",this.rtpPort,user_profile.audio_avp,user_profile.audio_codec,user_profile.audio_sample_rate);

       if (user_profile.audio)
          optBody=sdp.toString();
       
       // I didn't add the video caps
	   return optBody;
   }
   

  
   /** Makes a new call (acting as UAC). */
   public void call(String target_url)
   {  
	  changeStatus(UA_OUTGOING_CALL);
      
	  if (call!=null)
	  {
		  // need to remove Invite listener
	      call.removeInviteListener();
	  }
	  
	  // 2008/11/30 allow override of usr,pass and realm if specified in target url
	  String tmpUsr=user_profile.username;
	  String tmpPass=user_profile.passwd;
	  String tmpRealm=user_profile.realm;
	  String tmpFrom=user_profile.from_url;

	  if (target_url.indexOf(";usr=")>=0)
		  tmpUsr=target_url.replaceAll("(?i).*;usr=([^;]+).*","$1");
	  if (target_url.indexOf(";pwd=")>=0)
		  tmpPass=target_url.replaceAll("(?i).*;pwd=([^;]+).*","$1");
	  if (target_url.indexOf(";realm=")>=0)
		  tmpRealm=target_url.replaceAll("(?i).*;realm=([^;]+).*","$1");
	  if (target_url.indexOf(";from=")>=0)
		  tmpFrom=target_url.replaceAll("(?i).*;from=([^;]+).*","$1");

	  target_url=target_url.replaceAll(";.*","");
	
	  call=new ExtendedCall(sip_provider,tmpFrom,user_profile.contact_url,tmpUsr,tmpRealm,tmpPass,this);      
     
	  // in case of incomplete url (e.g. only 'user' is present), try to complete it
      target_url=sip_provider.completeNameAddress(target_url).toString();
      log.info("\r\nAttempting call From:"+user_profile.from_url+" To:"+target_url);
      if (user_profile.no_offer) 
    	  call.call(target_url);
      else
      {
    	  resetLocalSession();
    	  call.call(target_url,local_session);
      }
   }   


   /** Waits for an incoming call (acting as UAS). */
   public synchronized void listen()
   {
	  if (call==null || call.callUsed())
	  {	  
		  if (call!=null)
		  {	  
			  int waitcnt=0;
			  while (this.call_state!=UA_IDLE && waitcnt++<11)
			  {	  
				  // give previous call a chance to close or timeout
				  try {Thread.sleep(250);} catch(Exception e){}
			  }
		  }

		  onRemoteSipHold=false;
		  clearDtmfBuffer();
		  
		  changeStatus(UA_IDLE);
		  call=new ExtendedCall(sip_provider,user_profile.from_url,user_profile.contact_url,user_profile.username,user_profile.realm,user_profile.passwd,this);      
		  call.listen();
		  
		  log.info("WAITING FOR INCOMING CALL");
	  }
   } 
   
   /** Closes an ongoing, incoming, or pending call */
   public void hangup()
   {  
	  hangup(403);
   } 

   /** Closes an ongoing, incoming, or pending call */
   public void hangup(int status)
   {  
	  if (clip_ring!=null) clip_ring.stop();
	  
	  if (this.call_state!=this.UA_IDLE)
		  closeMediaApplication();
      
	  if (call!=null)
      {
    	  log.debug("HANGUP");
    	  call.hangup(status);
      }
      // should not do this - changeStatus(UA_IDLE);
   } 

   /** Accepts a call */
   public void accept()
   {  if (clip_ring!=null) clip_ring.stop();
      if (call!=null)
    	  call.accept(local_session);
   }   

   /** Accepts a call */
   public void sendEarlyMedia()
   {  if (clip_ring!=null) clip_ring.stop();
      if (call!=null)
    	  call.sendEarlyMedia(local_session.replaceAll("a=sendrecv", "a=sendonly"));
   }   

   /** Redirects an incoming call */
   public void redirect(String redirection)
   {  if (clip_ring!=null) clip_ring.stop();
      if (call!=null) call.redirect(redirection);
   }   


   /** Launches the Media Application (currently, the RAT audio tool) */
   protected void launchMediaApplication()
   {
      // exit if the Media Application is already running  
      if (audio_app!=null || video_app!=null)
      {  log.error("media application is already running");
         return;
      }
      
      SessionDescriptor local_sdp=new SessionDescriptor(call.getLocalSessionDescriptor());
      //String local_media_address=(new Parser(local_sdp.getConnection().toString())).skipString().skipString().getString();
      int local_audio_port=0;
      int local_video_port=0;
      // parse local sdp
      for (Enumeration e=local_sdp.getMediaDescriptors().elements(); e.hasMoreElements(); )
      {  MediaField media=((MediaDescriptor)e.nextElement()).getMedia();
         if (media.getMedia().equals("audio")) 
            local_audio_port=media.getPort();
         if (media.getMedia().equals("video")) 
            local_video_port=media.getPort();
      }
      
      // parse remote sdp
      SessionDescriptor remote_sdp=new SessionDescriptor(call.getRemoteSessionDescriptor());
      String remote_media_address=(new Parser(remote_sdp.getConnection().toString())).skipString().skipString().getString();
      int remote_audio_port=0;              
      int remote_video_port=0;              
      for (Enumeration e=remote_sdp.getMediaDescriptors().elements(); e.hasMoreElements(); )
      {  MediaField media=((MediaDescriptor)e.nextElement()).getMedia();
         if (media.getMedia().equals("audio")) 
            remote_audio_port=media.getPort();
         if (media.getMedia().equals("video")) 
            remote_video_port=media.getPort();
      }

      // select the media direction (send_only, recv_ony, fullduplex)
      int dir=0;
      if (user_profile.recv_only) dir=-1;
      else
      if (user_profile.send_only) dir=1;
      
      if (user_profile.audio && local_audio_port!=0 && remote_audio_port!=0)
      {  // create an audio_app and start it
         if (user_profile.use_rat)
         {  audio_app=new RATLauncher(user_profile.bin_rat,local_audio_port,remote_media_address,remote_audio_port);
         }
         else 
         if (user_profile.use_jmf)
         {  // try to use JMF audio app
            try
            {  Class myclass=Class.forName("local.ua.JMFAudioLauncher");
               Class[] parameter_types={ java.lang.Integer.TYPE, Class.forName("java.lang.String"), java.lang.Integer.TYPE, java.lang.Integer.TYPE};
               Object[] parameters={ new Integer(local_audio_port), remote_media_address, new Integer(remote_audio_port), new Integer(dir)};
               java.lang.reflect.Constructor constructor=myclass.getConstructor(parameter_types);
               audio_app=(MediaLauncher)constructor.newInstance(parameters);
            }
            catch (Exception e)
            {  log.error("error",e);
               log.error(" trying to create the JMFAudioLauncher");
            }
         }
         // else
         if (audio_app==null)
         {  // for testing..
            String audio_in=null;
            if (user_profile.send_tone) audio_in=JAudioLauncher.TONE;
            else if (user_profile.send_file!=null) audio_in=user_profile.send_file;
            String audio_out=null;
            if (user_profile.recv_file!=null) audio_out=user_profile.recv_file;        
            //audio_app=new JAudioLauncher(local_audio_port,remote_media_address,remote_audio_port,dir,log);
            audio_app=new JAudioLauncher(local_audio_port,remote_media_address,remote_audio_port,dir,audio_in,audio_out,user_profile.audio_sample_rate[0],user_profile.audio_sample_size[0],user_profile.audio_frame_size[0]);
         }
         audio_app.startMedia();
      }
      if (user_profile.video && local_video_port!=0 && remote_video_port!=0)
      {  // create a video_app and start it
         if (user_profile.use_vic)
         {  video_app=new VICLauncher(user_profile.bin_vic,local_video_port,remote_media_address,remote_video_port);
         }
         else 
         if (user_profile.use_jmf)
         {  // try to use JMF video app
            try
            {  Class myclass=Class.forName("local.ua.JMFVideoLauncher");
               Class[] parameter_types={ java.lang.Integer.TYPE, Class.forName("java.lang.String"), java.lang.Integer.TYPE, java.lang.Integer.TYPE};
               Object[] parameters={ new Integer(local_video_port), remote_media_address, new Integer(remote_video_port), new Integer(dir)};
               java.lang.reflect.Constructor constructor=myclass.getConstructor(parameter_types);
               video_app=(MediaLauncher)constructor.newInstance(parameters);
            }
            catch (Exception e)
            {  log.error("error",e);
               log.error(" trying to create the JMFVideoLauncher");
            }
         }
         // else
         if (video_app==null)
         {  log.error("No external video application nor JMF has been provided: Video not started");
            return;
         }
         video_app.startMedia();
      }
   }
 
   
   /** Close the Media Application  */
   protected void closeMediaApplication()
   {  if (audio_app!=null)
      {  audio_app.stopMedia();
         audio_app=null;
      }
      if (video_app!=null)
      {  video_app.stopMedia();
         video_app=null;
      }
   }


   // ********************** Call callback functions **********************
   
   /** Callback function called when arriving a new INVITE method (incoming call) */
   public void onCallIncoming(Call call, NameAddress callee, NameAddress caller, String remote_sdp, Message invite)
   {  log.debug("onCallIncoming()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.debug("onCallIncoming INCOMING SIP CALL\r\nFrom: "+caller+" To: "+callee);
      //log.debug("inside UserAgent.onCallIncoming(): sdp=\n"+sdp);
      changeStatus(UA_INCOMING_CALL);
      call.ring();
      if (remote_sdp!=null)
      {  // Create the new SDP
         resetLocalSession();
         SessionDescriptor new_sdp=selectCodec(local_session,remote_sdp,false);
         if (new_sdp==null)
         {	 
        	 hangup();
        	 if (listener!=null) listener.onUaCallFailed(this);
        	 return;
         }
         
         String pubIp=sip_provider.getPublicIP();
         if (pubIp!=null && !pubIp.equals(sip_provider.getViaAddress()) && !pubIp.equals(invite.getRemoteAddress()))
         {
        	 // if we know our public ip 
        	 //    and our public ip is not our via ip 
        	 //       and the remote IP is not our public ip
        	 //          and the remote connection IP is an external IP
        	 //    respond with our public IP in the connection field
        	 SessionDescriptor remoteDesc=new SessionDescriptor(remote_sdp);
        	 ConnectionField remoteConnIp=remoteDesc.getConnection();
        	 if (remoteConnIp!=null)
        	 {	 
        		 String remoteConnAddr=remoteConnIp.getAddress();
        		 if (remoteConnAddr!=null )
        		 {
        			 InetAddress remConnInetAddr=null;
        			 try
        			 {
        				 remConnInetAddr=InetAddress.getByName(remoteConnAddr);
        			 }
        			 catch(Exception e)
        			 {log.debug("Error on address:"+remoteConnAddr,e);};
        			 
            		 if (remConnInetAddr!=null && !sip_provider.isIPLocal(remConnInetAddr))
            		 {	 
        			   // set our connection field to respond with our publicip
        			   ConnectionField new_cf=new ConnectionField("IP4",pubIp);
        			   new_sdp.setConnection(new_cf);
        			   log.debug("RTP connection field changed to public ip");
            		 }
        		 }
        	 }
         }

         local_session=new_sdp.toString();
         
      }
      // play "ring" sound
      if (clip_ring!=null) clip_ring.loop();
      if (listener!=null) listener.onUaCallIncoming(this,callee,caller,invite);
   }  


  
   
   /** Callback function called when arriving a new Re-INVITE method (re-inviting/call modify) */
   public void onCallModifying(Call call, String remote_sdp, Message invite)
   {  log.debug("onCallModifying()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      
      if (!onRemoteSipHold && remote_sdp!=null && (remote_sdp.indexOf("a=sendonly")>=0 || remote_sdp.indexOf("c=IN IP4 0.0.0.0")>=0))
      {	  
    	  log.info("SIP Call placed on hold");
    	  onRemoteSipHold=true;
      }	  
      else if (onRemoteSipHold)
      {	  
    	  log.info("SIP Call resuming");
    	  onRemoteSipHold=false;
      }
      
      String local_session;
      if (remote_sdp!=null && remote_sdp.length()>0)
      {  
         SessionDescriptor new_sdp=selectCodec(call.getLocalSessionDescriptor(),remote_sdp,false);
         if (new_sdp==null)
         {
        	 hangup();
        	 if (listener!=null) listener.onUaCallFailed(this);
        	 return;
         }
        	 
         local_session=new_sdp.toString();
         
      }
      else 
    	  local_session=call.getLocalSessionDescriptor();
      
      // accept immediatly
      call.accept(local_session);
      
   }


   /** Callback function that may be overloaded (extended). Called when arriving a 180 Ringing */
   public void onCallRinging(Call call, Message resp)
   {  log.debug("onCallRinging()");
      if (call!=this.call && call!=call_transfer) {  log.debug("NOT the current call");  return;  }
      log.info("SIP RINGING");
      // play "on" sound
      if (clip_on!=null) clip_on.replay();
      if (listener!=null) listener.onUaCallRinging(this);
   }


   /** Callback function called when arriving a 2xx (call accepted by remote side) */
   public void onCallAccepted(Call call, String remote_sdp, Message resp)
   {  log.debug("onCallAccepted()");
      if (call!=this.call && call!=call_transfer) {  log.debug("NOT the current call");  return;  }
      log.info("SIP ACCEPTED/CALL");
      changeStatus(UA_ONCALL);
      if (user_profile.no_offer)
      {  // Create the new SDP
         
         SessionDescriptor new_sdp=selectCodec(local_session,remote_sdp,true);
         if (new_sdp==null)
         {
        	 hangup();
        	 if (listener!=null) listener.onUaCallFailed(this);
        	 return;
         }
         
         local_session=new_sdp.toString();
         
         // answer with the local sdp
         call.ackWithAnswer(local_session);
      }
      else
      {
          SessionDescriptor new_sdp=selectCodec(local_session,remote_sdp,true);
          if (new_sdp==null)
          {
         	 hangup();
         	 if (listener!=null) listener.onUaCallFailed(this);
         	 return;
          }
          
          local_session=new_sdp.toString();
          call.setLocalSessionDescriptor(local_session); // need to update call session after select
      }
      
      // play "on" sound
      if (clip_on!=null) clip_on.replay();
      if (listener!=null) listener.onUaCallAccepted(this);

      launchMediaApplication();
      
      if (call==call_transfer)
      {  StatusLine status_line=resp.getStatusLine();
         int code=status_line.getCode();
         String reason=status_line.getReason();
         this.call.notify(code,reason);
      }
   }


   /** Callback function called when arriving an ACK method (call confirmed by local side) */
   public void onCallConfirmed(Call call, String sdp, Message ack)
   {  log.debug("onCallConfirmed()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.debug("CONFIRMED/CALL");
      changeStatus(UA_ONCALL);
      // play "on" sound
      if (clip_on!=null) clip_on.replay();
      launchMediaApplication();
      if (listener!=null) listener.onUaCallConfirmed(this);
      if (user_profile.hangup_time>0) this.automaticHangup(user_profile.hangup_time); 
   }


   /** Callback function called when arriving a 2xx (re-invite/modify accepted) */
   public void onCallReInviteAccepted(Call call, String sdp, Message resp)
   {  log.debug("onCallReInviteAccepted()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("RE-INVITE-ACCEPTED/CALL");
   }


   /** Callback function called when arriving a 4xx (re-invite/modify failure) */
   public void onCallReInviteRefused(Call call, String reason, Message resp)
   {  log.debug("onCallReInviteRefused()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("RE-INVITE-REFUSED ("+reason+")/CALL");
      if (listener!=null) listener.onUaCallFailed(this);
   }


   /** Callback function called when arriving a 4xx (call failure) */
   public void onCallRefused(Call call, String reason, Message resp)
   {  log.debug("onCallRefused()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      int code=-1;
      try
      {
       code=resp.getStatusLine().getCode();
      }
      catch(Exception e)
      {}
      log.info("REFUSED ("+code+":"+reason+")");
      changeStatus(UA_IDLE);
      if (call==call_transfer)
      {  
         this.call.notify(code,reason);
         call_transfer=null;
      }
      // play "off" sound
      if (clip_off!=null) clip_off.replay();
      if (listener!=null) listener.onUaCallFailed(this);
   }


   /** Callback function called when arriving a 3xx (call redirection) */
   public void onCallRedirection(Call call, String reason, Vector contact_list, Message resp)
   {  log.debug("onCallRedirection()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("REDIRECTION ("+reason+")");
      call.call(((String)contact_list.elementAt(0))); 
   }


   /** Callback function that may be overloaded (extended). Called when arriving a CANCEL request */
   public void onCallCanceling(Call call, Message cancel)
   {  log.debug("onCallCanceling()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("CANCEL");
      changeStatus(UA_IDLE);
      // stop ringing
      if (clip_ring!=null) clip_ring.stop();
      // play "off" sound
      if (clip_off!=null) clip_off.replay();
      if (listener!=null) listener.onUaCallCancelled(this);
   }


   /** Callback function called when arriving a BYE request */
   public void onCallClosing(Call call, Message bye)
   {  log.debug("onCallClosing()");
      if (call!=this.call && call!=call_transfer) {  log.debug("onCallClosing - NOT the current call");  return;  }
      if (call!=call_transfer && call_transfer!=null)
      {  log.info("CLOSE PREVIOUS CALL");
         this.call=call_transfer;
         call_transfer=null;
         return;
      }
      // else
      log.debug("CLOSING");
      closeMediaApplication();
      // play "off" sound
      if (clip_off!=null) clip_off.replay();
      /* should not do this - call is not officially closed 
      if (listener!=null) listener.onUaCallClosed(this);
      changeStatus(UA_IDLE);
      */
   }


   /** Callback function called when arriving a response after a BYE request (call closed) */
   public void onCallClosed(Call call,Message msg)
   {  log.debug("onCallClosed()");
      if (call!=this.call) {  log.debug("onCallClosed - NOT the current call");  return;  }
      log.debug("CLOSE/OK");
      changeStatus(UA_IDLE); // need to do this now since listen may be called and may lock wait there
      if (listener!=null) listener.onUaCallClosed(this);
   }

   /** Callback function called when the invite expires */
   public void onCallTimeout(Call call)
   {  log.debug("onCallTimeout()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.error("NOT FOUND/TIMEOUT");
      changeStatus(UA_IDLE);
      if (call==call_transfer)
      {  int code=408;
         String reason="Request Timeout";
         this.call.notify(code,reason);
         call_transfer=null;
      }
      // play "off" sound
      if (clip_off!=null) clip_off.replay();
      if (listener!=null) listener.onUaCallFailed(this);
   }



   // ****************** ExtendedCall callback functions ******************

   /** Callback function called when arriving a new REFER method (transfer request) */
   public void onCallTransfer(ExtendedCall call, NameAddress refer_to, NameAddress refered_by, Message refer)
   {  log.debug("onCallTransfer()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("Transfer to "+refer_to.toString());
      call.acceptTransfer();
      call_transfer=new ExtendedCall(sip_provider,user_profile.from_url,user_profile.contact_url,this);
      call_transfer.call(refer_to.toString(),local_session);
   }

   /** Callback function called when a call transfer is accepted. */
   public void onCallTransferAccepted(ExtendedCall call, Message resp)
   {  log.debug("onCallTransferAccepted()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("Transfer accepted");
   }

   /** Callback function called when a call transfer is refused. */
   public void onCallTransferRefused(ExtendedCall call, String reason, Message resp)
   {  log.debug("onCallTransferRefused()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("Transfer refused");
   }

   /** Callback function called when a call transfer is successfully completed */
   public void onCallTransferSuccess(ExtendedCall call, Message notify)
   {  log.debug("onCallTransferSuccess()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("Transfer successed");
      call.hangup();
      if (listener!=null) listener.onUaCallTrasferred(this);
   }

   /** Callback function called when a call transfer is NOT sucessfully completed */
   public void onCallTransferFailure(ExtendedCall call, String reason, Message notify)
   {  log.debug("onCallTransferFailure()");
      if (call!=this.call) {  log.debug("NOT the current call");  return;  }
      log.info("Transfer failed");
   }


   // ************************* Schedule events ***********************

   /** Schedules a re-inviting event after <i>delay_time</i> secs. */
   void reInvite(final String contact_url, final int delay_time)
   {  SessionDescriptor sdp=new SessionDescriptor(local_session);
      final SessionDescriptor new_sdp=new SessionDescriptor(sdp.getOrigin(),sdp.getSessionName(),new ConnectionField("IP4","0.0.0.0"),new TimeField());
      new_sdp.addMediaDescriptors(sdp.getMediaDescriptors());
      (new Thread(this.getClass().getName()) {  public void run() {  runReInvite(contact_url,new_sdp.toString(),delay_time);  }  }).start();
   }
    
   /** Re-invite. */
   private void runReInvite(String contact, String body, int delay_time)
   {  try
      {  if (delay_time>0) Thread.sleep(delay_time*1000);
         log.info("RE-INVITING/MODIFING");
         if (call!=null && call.isOnCall())
         {  log.info("REFER/TRANSFER");
            call.modify(contact,body);
         }
      }
      catch (Exception e) { e.printStackTrace(); }
   }


   /** Schedules a call-transfer event after <i>delay_time</i> secs. */
   void callTransfer(final String transfer_to, final int delay_time)
   {  (new Thread(this.getClass().getName()) {  public void run() {  runCallTransfer(transfer_to,delay_time);  }  }).start();
   }

   /** Call-transfer. */
   private void runCallTransfer(String transfer_to, int delay_time)
   {  try
      {  if (delay_time>0) Thread.sleep(delay_time*1000);
         if (call!=null && call.isOnCall())
         {  log.info("REFER/TRANSFER");
            call.transfer(transfer_to);
         }
      }
      catch (Exception e) { e.printStackTrace(); }
   }


   /** Schedules an automatic answer event after <i>delay_time</i> secs. */
   void automaticAccept(final int delay_time)
   {  (new Thread(this.getClass().getName()) {  public void run() {  runAutomaticAccept(delay_time);  }  }).start();
   }

   /** Automatic answer. */
   private void runAutomaticAccept(int delay_time)
   {  try
      {  if (delay_time>0) Thread.sleep(delay_time*1000);
         if (call!=null)
         {  log.info("AUTOMATIC-ANSWER");
            accept();
         }
      }
      catch (Exception e) { e.printStackTrace(); }
   }


   /** Schedules an automatic hangup event after <i>delay_time</i> secs. */
   void automaticHangup(final int delay_time)
   {  (new Thread(this.getClass().getName()) {  public void run() {  runAutomaticHangup(delay_time);  }  }).start();
   }

   /** Automatic hangup. */
   private void runAutomaticHangup(int delay_time)
   {  try
      {  if (delay_time>0) Thread.sleep(delay_time*1000);
         if (call!=null && call.isOnCall())
         {  log.info("AUTOMATIC-HANGUP");
            hangup();
            listen();
         }
      }
      catch (Exception e) { e.printStackTrace(); }
   }

 
   public void onDtmfReceived(Call call,int digit)
   {
      if (call!=this.call) {  log.debug("NOT the current call");  return;}

      processDtmf(digit);
   }   

   private void processDtmf(int digit)
   {
	      if (digit<10)
	    	  this.dtmfBuffer+=String.valueOf(digit);
	      else if (digit==10)
	    	  this.dtmfBuffer+="*";
	      else if (digit==11)
	    	  this.dtmfBuffer+="#";
	      else if (digit>11 && digit<16)
	    	  this.dtmfBuffer+=String.valueOf((char)(digit+53));
	      
	      if (listener!=null) listener.onDtmfReceived(this, digit,false);
   }
   
   
   public String getDtmfBuffer()
   {
	   return this.dtmfBuffer;
   }
   
   public void clearDtmfBuffer()
   {
	   this.dtmfBuffer="";
   }

   public void chopDtmfBuffer(int bytes)
   {
	   if (bytes<this.dtmfBuffer.length())
		   this.dtmfBuffer=this.dtmfBuffer.substring(bytes);
	   else
		   this.dtmfBuffer="";
   }

   private Pattern sgPat=Pattern.compile("Signal\\s*=\\s*([0-9A-D\\x2a#])",Pattern.CASE_INSENSITIVE);
   public void onInfoMsg(Call call, Message msg)
   {
	   if (call!=this.call) {  log.debug("NOT the current call");  return;}
	   
	    
	    if (msg.getContentTypeHeader().getContentType().toLowerCase().contains("dtmf"))
	    {
		    char sig=0;
	    	if (msg.getBody().length()>7)
		    {
	    	    Matcher sgMat=sgPat.matcher(msg.getBody());
		    	if (sgMat.find() && sgMat.groupCount()>0)
		    	    sig=sgMat.group(1).toUpperCase().charAt(0);
		    }
	    	else
	    		sig=msg.getBody().toUpperCase().charAt(0);
		    
 	 	    int digit;
    		if (sig=='*')
    			digit=10;
    		else if (sig=='#')
    			digit=11;
    		else if (sig>='A' && sig<='D')
    			digit=(sig-53);
    		else
    		    digit=sig-48;
    		
			processDtmf(digit);
	    }
   }
   
}
