/*
 * Copyright (C) 2005 Luca Veltri - University of Parma - Italy
 * 
 * This file is part of MjSip (http://www.mjsip.org)
 * 
 * MjSip is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * MjSip is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with MjSip; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Author(s):
 * Luca Veltri (luca.veltri@unipr.it)
 * 
 * Modified Greg Dorfuss mhspot.com 2008/09/10 
 * Handle multiple codec selections
 * 2008/10/21 patch to ignore optional channels
 * 2009/02/10 fix case sensitivity issue with dynamic codecs
 *            fix compatibility issue with rtpmap not being sent in sdp for standard codecs
 */

package org.zoolu.sip.call;


import org.zoolu.sdp.*;
import java.util.Enumeration;
import java.util.Vector;


/** Class SdpTools collects some static methods for managing SDP materials.
  */
public class SdpTools
{
   /** Costructs a new SessionDescriptor from a given SessionDescriptor
     * with olny media types and attribute values specified by a MediaDescriptor Vector.
     * <p> If no attribute is specified for a particular media, all present attributes are kept.
     * <br>If no attribute is present for a selected media, the media is kept (regardless any sepcified attributes).
     * @param sdp the given SessionDescriptor
     * @param m_descs Vector of MediaDescriptor with the selecting media types and attributes
     * @return this SessionDescriptor */
   public static SessionDescriptor sdpMediaProduct(SessionDescriptor sdp, Vector m_descs)
   {  Vector new_media=new Vector();
      if (m_descs!=null)
      {  for (Enumeration e=m_descs.elements(); e.hasMoreElements(); )
         {  MediaDescriptor spec_md=(MediaDescriptor)e.nextElement();
            //System.out.print("DEBUG: SDP: sdp_select: "+spec_md.toString());
            MediaDescriptor prev_md=sdp.getMediaDescriptor(spec_md.getMedia().getMedia());
            //System.out.print("DEBUG: SDP: sdp_origin: "+prev_md.toString());
            if (prev_md!=null)
            {  Vector spec_attributes=spec_md.getAttributes();
               Vector prev_attributes=prev_md.getAttributes();
               if (spec_attributes.size()==0 || prev_attributes.size()==0)
               {  new_media.addElement(prev_md);
               }
               else
               {  Vector new_attributes=new Vector();
                  for (Enumeration i=spec_attributes.elements(); i.hasMoreElements(); )
                  {  AttributeField spec_attr=(AttributeField)i.nextElement();
                     String spec_name=spec_attr.getAttributeName();
                     String spec_value=spec_attr.getAttributeValue();
                     for (Enumeration k=prev_attributes.elements(); k.hasMoreElements(); )
                     {  AttributeField prev_attr=(AttributeField)k.nextElement();
                        String prev_name=prev_attr.getAttributeName();
                        String prev_value=prev_attr.getAttributeValue();
                        if (prev_name.equals(spec_name) && prev_value.equalsIgnoreCase(spec_value))
                        {  new_attributes.addElement(prev_attr);
                           break;
                        }
                     }
                  }
                  if (new_attributes.size()>0) new_media.addElement(new MediaDescriptor(prev_md.getMedia(),prev_md.getConnection(),new_attributes));
               }
            }
         }
      }
      SessionDescriptor new_sdp=new SessionDescriptor(sdp);
      new_sdp.removeMediaDescriptors();
      new_sdp.addMediaDescriptors(new_media);
      return new_sdp;
   }
   
   /** Costructs a new SessionDescriptor from a given SessionDescriptor
     * with olny the first specified media attribute.
   /** Keeps only the fisrt attribute of the specified type for each media.
     * <p> If no attribute is present for a media, the media is dropped.
     * @param sdp the given SessionDescriptor
     * @param a_name the attribute name
     * @return this SessionDescriptor */
  
   public static SessionDescriptor sdpAttirbuteSelection(SessionDescriptor sdp, String a_name)
   {  Vector new_media=new Vector();
      for (Enumeration e=sdp.getMediaDescriptors().elements(); e.hasMoreElements(); )
      {  
    	 MediaDescriptor md=(MediaDescriptor)e.nextElement();
         AttributeField attr=md.getAttribute(a_name);
         if (attr!=null)
         { 
        	 new_media.addElement(new MediaDescriptor(md.getMedia(),md.getConnection(),attr));
         }
      }
      SessionDescriptor new_sdp=new SessionDescriptor(sdp);
      new_sdp.removeMediaDescriptors();
      new_sdp.addMediaDescriptors(new_media);
      return new_sdp;
   }
  
  
   
   /*
    * do media selection and remove others, first descriptor take priority over second
    * works with multiple medias
    */
   public static SessionDescriptor selectMedia(SessionDescriptor sdp, SessionDescriptor priSdp, SessionDescriptor secSdp)
   {
	      Vector priDescs=priSdp.getMediaDescriptors();
	      SessionDescriptor new_sdp=new SessionDescriptor(sdp);
	      new_sdp.removeMediaDescriptors();
	      
	      for (Enumeration e=priDescs.elements(); e.hasMoreElements(); )
	      {  
	    	  MediaDescriptor pri_md=(MediaDescriptor)e.nextElement();
	    	  String priMediaType=pri_md.getMedia().getMedia();
	    	  
	    	  //System.out.println("pr_mt="+priMediaType);
	    	  
	    	  MediaDescriptor new_md=sdp.getMediaDescriptor(priMediaType);
	    	  
	    	  MediaDescriptor sec_md=secSdp.getMediaDescriptor(priMediaType);
	    	  if (sec_md!=null)
	    	  {
	    		     MediaDescriptor new_cur_media =new MediaDescriptor(new MediaField(sdp.getMediaDescriptor(priMediaType).getMedia()),new_md.getConnection(),(AttributeField)null);
	    		     new_cur_media.getMedia().removeAllFormats();
	    		     
	    		     Vector priFormats=pri_md.getMedia().getFormatList();
	    		     Vector secFormats=sec_md.getMedia().getFormatList();
	    	    	 boolean matched=false;
	    	    	 for (int ca=0;ca<priFormats.size();ca++)
	    	    	 {		 
	    	    		 	 String priFormat=(String)priFormats.get(ca);
	    	    		 	 int numPriFormat=Integer.parseInt(priFormat);
	    	    		 	 AttributeField priAttr=getRtpMapForFormat(pri_md,priFormat);
	    	    		 	 if (priAttr!=null)
	    	    		 	 {		 
		    		    		 //System.out.println("priFormat="+priFormat.toString()+" priAttr="+priAttr.toString());
		    		    		 
		    		        	 for (int sa=0;sa<secFormats.size();sa++)
		    		        	 {	 
		    		        		   String secFormat=(String)secFormats.get(sa);
		  	    	    		 	   int numSecFormat=Integer.parseInt(secFormat);
		  	    	    		 	   AttributeField secAttr=getRtpMapForFormat(sec_md,secFormat);
		  	    	    		 	   if (secAttr!=null)
		  	    	    		 	   {	   
					    		    	   //System.out.println("secFormat="+secFormat.toString()+" secAttr="+secAttr.toString());
		  	    	    		 		   if (numPriFormat<96 || numPriFormat>127) //iana.org type
		  	    	    		 		   {	   
		  	    	    		 			   if (numSecFormat==numPriFormat && !matched)
		  	    	    		 			   {
		  	    	    		 				   	new_cur_media.getMedia().appendFormat(secFormat);
		  	    	    		 					new_cur_media.addAttribute(secAttr);
				    			        	   		//System.out.println("newDesc="+new_cur_media.toString());
		  	    	    		 				    matched=true;
				    			        	   		break;
		  	    	    		 			   }
		  	    	    		 		   }
		  	    	    		 		   else // dynamic payload test
		  	    	    		 		   {
		  	    	    		 			  String[] priCodecInfo=priAttr.getAttributeValues();
		  	    	    		 			  String priCodecFix=priCodecInfo[1].replaceAll("/1$","");
		  	    	    		 			  String[] secCodecInfo=secAttr.getAttributeValues();
		  	    	    		 			  String secCodecFix=secCodecInfo[1].replaceAll("/1$","");
		  	    	    		 			  if (secCodecFix.toUpperCase().equals(priCodecFix.toUpperCase()))
		  	    	    		 			  {	  
		  	    	    		 				  if (secCodecInfo[1].toLowerCase().contains("telephone-event"))
		  	    	    		 				  {
		  	    	    		 					   new_cur_media.getMedia().appendFormat(secFormat);
		  	    	    		 					   new_cur_media.addAttribute(secAttr);
		  	    	    		 					   new_cur_media.addAttribute(new AttributeField("fmtp",secFormat+" 0-15"));
					    			        	   	   
		  	    	    		 					   //System.out.println("newDesc="+new_cur_media.toString());
					    			        	   	  break;
		  	    	    		 				  }
			  	    	    		 			  else if (!matched)
			  	    	    		 			  {	  
					    			        		   // we are going to use the secondaries codec info
			  	    	    		 				   new_cur_media.getMedia().appendFormat(secFormat);
			  	    	    		 				   new_cur_media.addAttribute(secAttr);
			  	    	    		 				   //System.out.println("newDesc="+new_cur_media.toString());
				  	    	    		 			   matched=true;
					    			        	   	   break;
			  	    	    		 			  }
		  	    	    		 			  }
		  	    	    		 		   }
		  	    	    		 		   
		  	    	    		 	   }
		    		        	 }
	    	    		 	 }

	    	    	 }
	    		  
	   		      //System.out.println("## All newDesc=\r\n"+new_cur_media);
	   		      
	   		      // add it now
			      if (new_cur_media.getMedia().getFormatList().size()>0)
			    	  new_sdp.addMediaDescriptor(new_cur_media);
	    	  }
	    	  
	    	  
	      }
	      return new_sdp;
   }
   
   private static AttributeField getRtpMapForFormat(MediaDescriptor md, String format)
   {
	   Vector rtpmap=md.getAttributes("rtpmap");
	   for (int m=0;m<rtpmap.size();m++)
	   {
		   AttributeField cf=(AttributeField)rtpmap.get(m);
		   if (cf.getAttributeValues()[0].equals(format))
			   return cf;
	   }
	   
	   if (format.equals("0"))
		   return new AttributeField("rtpmap","0 PCMU/8000");
	   else if (format.equals("8"))
		   return new AttributeField("rtpmap","8 PCMA/8000");
	   else if (format.equals("3"))
		   return new AttributeField("rtpmap","3 GSM/8000");
	   
	   return null;
   }
   
   
/*  
   public static void main( String [] args) 
   {
	   boolean remoteHasPriority=false;
	   
	   String remoteMsg="v=0\r\n";
	   remoteMsg+="o=3711428102 1 1 IN IP4 174.21.183.55\r\n";
	   remoteMsg+="s=Session SIP/SDP\r\n";
	   remoteMsg+="c=IN IP4 174.21.183.55\r\n";
	   remoteMsg+="t=0 0\r\n";
	   remoteMsg+="m=audio 49152 RTP/AVP 111 103 0 8 96\r\n";
	   remoteMsg+="a=rtpmap:111 speex/16000/1\r\n";
	   //remoteMsg+="a=rtpmap:0 PCMU/8000\r\n";
	   //remoteMsg+="a=rtpmap:3 GSM/8000\r\n";
	   //remoteMsg+="a=rtpmap:103 iLBC/8000\r\n";
	   //remoteMsg+="a=rtpmap:8 PCMA/8000\r\n";
	   remoteMsg+="a=rtpmap:96 telephone-event/8000\r\n";
	   remoteMsg+="a=fmtp:96 0-15\r\n";
	   //remoteMsg+="m=video 63101 RTP/AVP 195 201 203\r\n";
	   //remoteMsg+="a=rtpmap:195 MPEG2/80000\r\n";
	   //remoteMsg+="a=rtpmap:201 MPEG2/90000\r\n";
	   //remoteMsg+="a=rtpmap:203 MPEG3/120000\r\n";

	   String localSdp="v=0\r\n";
	   localSdp+="o=3711428102 1 1 IN IP4 192.168.0.4\r\n";
	   localSdp+="s=Session SIP/SDP\r\n";
	   localSdp+="c=IN IP4 192.168.0.4\r\n";
	   localSdp+="t=0 0\r\n";
	   localSdp+="m=audio 63200 RTP/AVP 3 8 0 101\r\n";
	   //localSdp+="a=rtpmap:3 GSM/8000\r\n";
	   //localSdp+="a=rtpmap:99 iLBC/8000\r\n";
	   //localSdp+="a=rtpmap:8 PCMA/8000\r\n";
	   //localSdp+="a=rtpmap:0 PCMU/8000\r\n";
	   localSdp+="a=rtpmap:111 speex/16000\r\n";
	   localSdp+="a=rtpmap:101 telephone-event/8000\r\n";
	   localSdp+="a=fmtp:101 0-15\r\n";
	   //localSdp+="m=video 13101 RTP/AVP 203 201 195\r\n";
	   //localSdp+="a=rtpmap:201 MPEG2/90000\r\n";
	   //localSdp+="a=rtpmap:203 MPEG3/120000\r\n";
	   //localSdp+="a=rtpmap:195 MPEG2/80000\r\n";
	   
	   SessionDescriptor local_sdp=new SessionDescriptor(localSdp);
	   
	   
	   //AttributeField newAttr=new AttributeField("rtpmap","15 ILBC");
	   //local_sdp.getMediaDescriptor("audio").addAttribute(newAttr);
	   //local_sdp.getMediaDescriptor("audio").getMedia().appendFormat("15");
	   //local_sdp.getMediaDescriptor("audio").getMedia().removeFormat("3");
	   //local_sdp.getMediaDescriptor("audio").removeAttributeField("rtpmap","3 GSM");
	   
	   System.out.println("## local:"+local_sdp.toString());

	   
	   SessionDescriptor remote_sdp=new SessionDescriptor(remoteMsg);

	   System.out.println("## remote:"+remote_sdp.toString());
	   
       SessionDescriptor new_sdp=new SessionDescriptor(remote_sdp.getOrigin(),remote_sdp.getSessionName(),local_sdp.getConnection(),local_sdp.getTime());
       new_sdp.addMediaDescriptors(local_sdp.getMediaDescriptors());
       
       if (remoteHasPriority)
    	   new_sdp=SdpTools.selectMedia(new_sdp,remote_sdp,local_sdp);
       else 
    	   new_sdp=SdpTools.selectMedia(new_sdp,local_sdp,remote_sdp);
       
	   System.out.println("\r\nafter select:"+new_sdp.toString());
   }
*/ 
}
