/*
 * Copyright (C) 2009 Greg Dorfuss - MHSpot LLC
 * This file is part of MjSip (http://www.mjsip.org)
 * 
 * MjSip is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * MjSip is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with MjSip; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Author(s):
 * Greg Dorfuss 2009/04/27
 */

package org.zoolu.sip.provider;

import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.SocketTimeoutException;
import java.net.BindException;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

public class StunClient extends Thread
{
		private int intervalMinutes=0;
		private static Logger log=null;
		private static boolean bigEndian=true;
		private static SipProvider sip_provider=null;
		private String localIp;
		private int localPort;
		private boolean stopNow=false;
		private String[] stunServers=null;
		
		 public static void main( String [] args) 
		 {
			 PropertyConfigurator.configure("log.properties");
			 StunClient sc=new StunClient(1,"192.168.0.4",0,"stunserver.org:3478,stun01.sipphone.com:3478",null);
		 }
		 
		StunClient( int argIntervalMinutes,String argLocalIp,int argLocalPort,String arg_stun_server,SipProvider arg_sip_provider) 
		{
			this.setName(this.getClass().getName()+".T"+this.getName().replaceAll("Thread-", ""));
			intervalMinutes=argIntervalMinutes;
			localIp=argLocalIp;
			localPort=argLocalPort;
			stunServers=arg_stun_server.split(",");
			sip_provider=arg_sip_provider;
			log=Logger.getLogger(this.getName());
			start();
		}

		public void halt()
		{
			stopNow=true;
		}
		
		public void run() 
		{
			try
			{
				while (!stopNow)
				{
					for (int s=0;s<stunServers.length;s++)
					{	
						log.debug("STUN: Server: "+stunServers[s]);
						String[] stunParms=stunServers[s].split(":");
						String stun_server=stunParms[0].trim();
						int stun_port;
						if (stunParms.length>1)
							stun_port=Integer.parseInt(stunParms[1].trim());
						else
							stun_port=3478;

						String pubIp=getPubIp(localIp,localPort,stun_server,stun_port);
						if (pubIp!=null && pubIp.length()>0)
						{	
							if (sip_provider!=null)
								sip_provider.setPublicIP(pubIp);
							break;
						}
					}

					if (intervalMinutes<=0)
						break; // one time deal
					
					int cnt=0;
					while (!interrupted() && cnt<intervalMinutes)
					{	
						if (stopNow)
							return;
						
						sleep(60*1000);
						cnt++;
					}
				}
			}
			catch(InterruptedException e)
			{}
		}
		
		private String getPubIp(String iaddress,int localPort,String stunServer,int stun_port)  
		{
			String retvar="";
			int timeSinceFirstTransmission = 0;
			int timeout = 300;
			DatagramSocket stunSocket = null;
			InetAddress stunInet=null;
			try
			{
			  stunInet=InetAddress.getByName(stunServer);
			}
			catch(java.net.UnknownHostException eh)
			{
				log.info("STUN: Error - Unknown Host: "+stunServer);
				return "";
			}
			catch(Exception e)
			{
				log.info("STUN: Error",e);
				return "";
			}
			
			while (true) 
			{
				try {
					stunSocket = new DatagramSocket(new InetSocketAddress(iaddress, localPort));
					stunSocket.setReuseAddress(true);
					stunSocket.connect(stunInet, stun_port);
					stunSocket.setSoTimeout(timeout);
					
					int reqlen=28;
					byte[] msgHdr =new byte[reqlen];
					// binding req
					msgHdr[1]=1;
					msgHdr[3]=0x08;
					
					// make trans
					byte[] transId=genTransId();
					for (int x=0;x<transId.length;x++)
						msgHdr[x+4]=transId[x];
					
					// chgreq
					byte[] chgreq={0x00, 0x03,0x00,0x04,0x00,0x00,0x00,0x00};
					for (int x=0;x<chgreq.length;x++)
						msgHdr[x+20]=chgreq[x];
					
					//log.debug("STUN: Bind request msgsize="+msgHdr.length);
					
					DatagramPacket send = new DatagramPacket(msgHdr, msgHdr.length);
					stunSocket.send(send);
					log.debug("STUN: Bind request sent");
				
					byte[] rx = new byte[200];
					int wait=0;
					boolean gotResponse=false;
					while (wait++<10) 
					{
						DatagramPacket receive = new DatagramPacket(rx, 200);
						stunSocket.receive(receive);
						if (getInt(rx,0)==0x0101 && testTransID(rx,transId))
						{
							gotResponse=true;
							log.debug("STUN: Bind response received");
							break;
						}
					}
					
					
					stunSocket.close();
					
					if (gotResponse) 
					   retvar=extractIP(rx);
					
					break;
					
				} 
				catch (SocketTimeoutException ste) 
				{
					if (timeSinceFirstTransmission < 7900) 
					{
						log.debug("STUN: timeout while receiving the response.");
						timeSinceFirstTransmission += timeout;
						int timeoutAddValue = (timeSinceFirstTransmission * 2);
						if (timeoutAddValue > 1600) timeoutAddValue = 1600;
						timeout = timeoutAddValue;
					} 
					else 
					{
						log.error("STUN: No Response.");
						break;
					}
					
					try	{if (stunSocket!=null) stunSocket.close();} catch(Exception e){}
				}
				catch (BindException be)
				{
					if (be.getMessage().contains("Address already in use"))
					{
						if (timeSinceFirstTransmission < 7900) 
						{
							try {sleep(timeout);}catch (Exception e){}
							timeSinceFirstTransmission += timeout;
							int timeoutAddValue = (timeSinceFirstTransmission * 2);
							if (timeoutAddValue > 1600) timeoutAddValue = 1600;
							timeout = timeoutAddValue;
							log.debug("error:",be);
						} 
						else 
						{
							log.error("STUN: No Response.");
							break;
						}
					}
					else	
						log.error("error:",be);
				}
				catch(Exception e)
				{
					log.error("error:",e);
				}
			}
			return retvar;
		}	 
	 
		
		private boolean testTransID(byte[] id,byte[] idHdrTransId) 
		{
			for (int x=0;x<16;x++)
			{
				if (id[x+4]!=idHdrTransId[x])
					return false;
			}
			return true;
		}

		private byte[] genTransId()
		{
			byte[] id=new byte[16];
			for (int x=0;x<16;x++)
				id[x]=(byte)(Math.random()*255);
			return id;	
		}
		
		private String extractIP(byte[] msg)
		{
			String retvar=null;
			
			int offset=20;
			int msgLen=getInt(msg,offset) & 0xffff;
			
			while (msgLen>0)
			{	
				int type = getInt(msg,offset) & 0xffff;
				int attribLen = getInt(msg,offset+2) & 0xffff;

				if (type==0x0001)
				{	
					int family=msg[offset+5];
					if (family==0x01)
					{	
						int port=getInt(msg,offset+6) & 0xffff;
						int firstOctet = msg[offset+8] & 0xff;
						int secondOctet = msg[offset+9] & 0xff;
						int thirdOctet = msg[offset+10] & 0xff;
						int fourthOctet = msg[offset+11] & 0xff;
						retvar=firstOctet + "." + secondOctet + "." + thirdOctet + "." + fourthOctet; // + ":" + port;
						log.debug("addr="+retvar+":"+port);
					}
					else
						log.debug("STUN: Unsupported Address family");
					
					break;
				}

				msgLen-=attribLen;
				offset+=attribLen;
			}
			return retvar;
		}
	 
		private static int getInt(byte buffer[], int byteOffset)
		{
		    return bigEndian ? buffer[byteOffset] << 8 | buffer[byteOffset + 1] & 0xff : buffer[byteOffset + 1] << 8 | buffer[byteOffset] & 0xff;
		}
}
