/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Based on mjsip 1.6 software and skype4java
 * 
 * Author(s):
 * Greg Dorfuss
 */

package local.ua;

import java.io.BufferedReader;
import java.io.FileReader;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import java.util.*;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

/* Sip Skype Authorizations
 */
public class AuthMap
{
	
	private Vector<DialOutRuleSet> skypeOutRules=null;
	private Vector<DialOutRuleSet> sipOutRules=null;
	private Vector sipSkypeAuthMap=null;
	private Hashtable skypeSipAuthMap=null;
	private Logger log = null;
	private String RulePairSplitDelim=":";
	
	public static void main(String[] args)
	{
	    PropertyConfigurator.configure("log.properties"); // needed only first time

	    AuthMap test=new AuthMap("SipToSkypeAuth.props","SkypeToSipAuth.props","SkypeOutDialingRules.props","SipOutDialingRules.props");
		test.test();
	}
	
	public AuthMap(String argSipToSkypeFile,String argSkypeToSipFile,String argSkypeOutDialingRules,String argSipOutDialingRules)
	{
		log = Logger.getLogger(this.getClass());
		sipSkypeAuthMap=loadSipSkypeAuthMap(argSipToSkypeFile);
		sipOutRules=loadSipOutRules(argSipOutDialingRules);
		skypeSipAuthMap=loadSkypeSipAuthMap(argSkypeToSipFile);
		skypeOutRules=loadSkypeOutRules(argSkypeOutDialingRules);
	}

   private void test()
   {
		log = Logger.getLogger(this.getClass().getName());
		
		sipSkypeAuthMap=loadSipSkypeAuthMap("SipToSkypeAuth.props");
		String dest=getSkypeDest("sip:<15615555553@sipphone.com>", "192.168.1.220", "sip:<11@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");
		
		dest=getSkypeDest("sip:<15615555554@sipphone.com>", "192.168.1.220", "sip:<11@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");
		
		dest=getSkypeDest("sip:<15615555555@sipphone.com>", "192.168.1.220", "sip:<11@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");

		dest=getSkypeDest("sip:<15615555556@sipphone.com>", "192.168.1.220", "sip:<11@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");

		dest=getSkypeDest("sip:<1561555321@sipphone.com>", "192.168.1.220", "sip:<11@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");
   
		dest=getSkypeDest("sip:<1561555321@sipphone.com>", "192.168.2.220", "sip:<11@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");

		dest=getSkypeDest("sip:<15614443333@sip.vonics.net>", "192.168.1.220", "sip:<13@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");
   

		dest=getSkypeDest("sip:<15614443333@sip.vonics.net>", "192.168.1.220", "sip:<01234567@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");
		
		dest=getSkypeDest("sip:<15614443333@sip.vonics.net>", "192.168.1.220", "sip:<05611234567@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");
		
		dest=getSkypeDest("sip:<15614443333@sip.vonics.net>", "192.168.1.220", "sip:<015611234567@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");
		
		dest=getSkypeDest("sip:<15614443333@sip.vonics.net>", "192.168.1.220", "sip:<0244445555@192.168.0.4>");
		log.info("Dest="+dest);
		log.info("FilteredDest="+skypeOutFilter(dest)+"\n");

   
		dest="55";
		log.info("Dest="+dest);
		log.info("FilteredSIPDest="+sipOutFilter(dest)+"\n");
   

		dest="4325555";
		log.info("Dest="+dest);
		log.info("FilteredSIPDest="+sipOutFilter(dest)+"\n");
   
   }
	
	
   private Vector<AMap> loadSipSkypeAuthMap(String argMapFile)
   {
	   Vector<AMap> map=new Vector<AMap>();
	   String curLine;
	   
	   FileReader in;
       
	   try 
	   {
	        in = new FileReader(argMapFile);
	        BufferedReader br=new BufferedReader(in);
	        
	        while ( (curLine=br.readLine())!=null)
	        {
	        	if (curLine.trim().length()>0 && !curLine.startsWith("#"))
	        	{
	        		String[] parms=curLine.split(",",4);
	        		
	        		try
	        		{
	        			map.add(new AMap(parms[0].toLowerCase().replaceAll("\\x2a.*$","").trim(),parms[1].toLowerCase().replaceAll("\\x2a.*$","").trim(),parms[2].toLowerCase().replaceAll("\\x2a.*$","").trim(),parms[3].replaceAll("//.*$","").trim()));
	        		}
	        		catch(Exception e)
	        		{
	        			log.error("loadSipSkypeAuthMap: Rule: \""+curLine+"\" invalid - ",e);
	        		}
	        	}
	        }
	        
	        br.close();
	        in.close();
	        
	        if (map.size()<1)
	        	throw new Exception("No SipToSkype Authorizations Found.");
	        
	   }
	   catch(Exception e)
	   {
		  log.fatal("Error with file:"+argMapFile,e);
	      System.exit(SkypeUA.ExitCode.INITFAIL.code());	  
	   }
	   
	   return map;
   }
   
   public String getSkypeDest(String callerSipCID,String sipCIP,String sipDest)
   {
	   Pattern realmPat=Pattern.compile("@([^>:]+)\\b",Pattern.CASE_INSENSITIVE);
	   Pattern sipidPat=Pattern.compile("sip:<?([^@<]+)@",Pattern.CASE_INSENSITIVE);
	   Matcher cuidMat=sipidPat.matcher(callerSipCID);
	   String callUID=null;
	   if (!cuidMat.find())
		   return null;
	   callUID=cuidMat.group(1);

	   Matcher realmMat=realmPat.matcher(callerSipCID);
	   String callRealm=null;
	   if (!realmMat.find())
		   return null;
	   callRealm=realmMat.group(1);
	   
	   log.debug("map SipCallerID="+callUID+" realm:"+callRealm+" IP:"+sipCIP+" Dest:"+sipDest);
	   
	   String dest=null;
	   
	   // go through maps looking for a match
	   Iterator mapIt=sipSkypeAuthMap.iterator();
	   while (mapIt.hasNext())
	   {
		   AMap thisMap=(AMap) mapIt.next();

		   //printLog("test="+thisMap.callerUserId+":"+thisMap.realm+":"+thisMap.ip);

		   boolean sipCIPisLocalNet=false;
		   if (sipCIP.startsWith("10.") || sipCIP.startsWith("127.") || sipCIP.startsWith("192.168.") || sipCIP.startsWith("169.254."))
			   sipCIPisLocalNet=true;
		   else if (sipCIP.startsWith("172."))
		   {
			   String[] tmp=sipCIP.split("\\.");
			   int tmp2=Integer.parseInt(tmp[1]);
			   if (tmp2>=16 && tmp2<=31)
				   sipCIPisLocalNet=true;
		   }


		   if ( 
				(thisMap.callerUserId.length()==0 || thisMap.callerUserId.equals(callUID)) &&
		        (thisMap.realm.length()==0 || thisMap.realm.equals(callRealm)) &&
		        (thisMap.ip.length()==0 || sipCIP.startsWith(thisMap.ip) || (thisMap.ip.equalsIgnoreCase("localnet") && sipCIPisLocalNet))
		   	  )
		   {
			   log.debug("match="+thisMap.callerUserId+":"+thisMap.realm+":"+thisMap.ip);
			   
			   dest=thisMap.dest;
			   if (dest.toLowerCase().equals("deny"))
			   {	   
				   log.debug("Blocked Caller");
				   dest=null;
			   }
			   else if (dest.toLowerCase().equals("calleeid"))
			   {
				   Matcher duidMat=sipidPat.matcher(sipDest); // extract target sip dest id
				   if (!duidMat.find())
				   {	
					   log.debug("No Destination");
					   dest=null;
				   }
				   else
					   dest=duidMat.group(1);
			   }
			   else if (dest.length()==0)
			   {
				   log.debug("No Destination");
				   dest=null;
			   }   
			   return dest;
		   }
	   }
	   // no match
	   log.debug("UnAuthorized Caller");
	   return null;
   }

   
 
   private Hashtable<String,String> loadSkypeSipAuthMap(String argMapFile)
   {
	   Hashtable<String,String> map=new Hashtable<String,String>();
	   String curLine;
	   
	   FileReader in;
       
	   try 
	   {
	        in = new FileReader(argMapFile);
	        BufferedReader br=new BufferedReader(in);
	        
	        while ( (curLine=br.readLine())!=null)
	        {
	        	if (curLine.trim().length()>0 && !curLine.startsWith("#"))
	        	{
	        		String[] parms=curLine.split(",");
	        		// key is skypeuserid
	        		String key=parms[0].toLowerCase().trim();
	        		map.put(key, parms[1].replaceAll("//.*$","").trim());
	        	}
	        }
	        
	        br.close();
	        in.close();
	        
	        if (map.size()<1)
	        	throw new Exception("No SkypeToSip Authorizations Found.");
	        
	   }
	   catch(Exception e)
	   {
		  log.fatal("Error with file:"+argMapFile,e);
	      System.exit(SkypeUA.ExitCode.INITFAIL.code());	  
	   }
	   
	   return map;
   }
   
   public String getSipDest(String callerSkypeId)
   {
	   log.debug("mapSkypeCaller="+callerSkypeId);
	   
	   //key is skypeuserid
	   
	   String dest=null;
	   String keys[]=new String[2];
	   int s=0;
	   keys[s++]=callerSkypeId;
	   keys[s++]="*";
	   
	   // look for a match to one of the keys
	   for (int k=0;k<keys.length;k++)
	   {	   
		   if (skypeSipAuthMap.containsKey(keys[k]))
		   {
			   //printLog("match="+keys[k]);
			   
			   dest=skypeSipAuthMap.get(keys[k]).toString();
			   if (dest.toLowerCase().equals("deny"))
			   {	   
				  log.info("Blocked Caller");
			      return null;
			   }
			   return dest;
		   }
	   }
	   // no match
	   log.info("UnAuthorized Caller");
	   return null;
   }
   
   private Vector<DialOutRuleSet> loadSkypeOutRules(String argRulesFile)
   {
	    
	   Vector<DialOutRuleSet> retvar=new Vector<DialOutRuleSet>();
	   String curLine;
	   FileReader in;
       
	   try 
	   {
	        in = new FileReader(argRulesFile);
	        BufferedReader br=new BufferedReader(in);
	        
	        while ( (curLine=br.readLine())!=null)
	        {
	        	if (curLine.trim().length()>0 && !curLine.startsWith("#"))
	        	{
	 			   String rulePair[]=curLine.split(RulePairSplitDelim);
	     		   try
	    		   {
	     			  DialOutRuleSet rs=new DialOutRuleSet(Pattern.compile(rulePair[0].trim()),rulePair[1].trim());
	     			   retvar.add(rs);
	    		   }
				   catch (Exception e)
				   {
					   log.error("loadSkypeOutRules: Rule: \""+curLine+"\" invalid - ",e);
				   }
	        	}
	        }
	        
	        br.close();
	        in.close();
	   }
	   catch(Exception e)
	   {
		  log.fatal("Error with file:"+argRulesFile,e);
	      System.exit(SkypeUA.ExitCode.INITFAIL.code());	  
	   }
	   
	   
	   return retvar;
   }
   
   public String skypeOutFilter(String dest)
   {
	   // return orig if nothing matched
	   for (int r=0;r<this.skypeOutRules.size();r++)
	   {
		   DialOutRuleSet rs=this.skypeOutRules.get(r);
		   Matcher mat=rs.pattern.matcher(dest);
		   if (mat.find())
		   {
			   dest=mat.replaceAll(rs.replacement);
			   break;
		   }
	   }
	   return dest;
   }
    
   
   private Vector<DialOutRuleSet> loadSipOutRules(String argRulesFile)
   {
	    
	   Vector<DialOutRuleSet> retvar=new Vector<DialOutRuleSet>();
	   String curLine;
	   FileReader in;
       
	   try 
	   {
	        in = new FileReader(argRulesFile);
	        BufferedReader br=new BufferedReader(in);
	        
	        while ( (curLine=br.readLine())!=null)
	        {
	        	if (curLine.trim().length()>0 && !curLine.startsWith("#"))
	        	{
	 			   String rulePair[]=parmSplitter(curLine);
	     		   try
	    		   {
	     			  DialOutRuleSet rs=new DialOutRuleSet(Pattern.compile(rulePair[0].trim()),rulePair[1].trim());
	     			   retvar.add(rs);
	    		   }
				   catch (Exception e)
				   {
					   log.error("loadSipOutRules: Rule: \""+curLine+"\" invalid - ",e);
				   }
	        	}
	        }
	        
	        br.close();
	        in.close();
	   }
	   catch(Exception e)
	   {
		  log.fatal("Error with file:"+argRulesFile,e);
	      System.exit(SkypeUA.ExitCode.INITFAIL.code());	  
	   }
	   
	   
	   return retvar;
   }
   
   private String [] parmSplitter(String curLine)
   {
	   int pos=curLine.indexOf(RulePairSplitDelim);
	   
	   String[] retvar={"",""};
	   
	   retvar[0]=curLine.substring(0,pos);
	   retvar[1]=curLine.substring(pos+1);
	   return retvar;
   }
   
   
   public String sipOutFilter(String dest)
   {
	   // return orig if nothing matched
	   for (int r=0;r<this.sipOutRules.size();r++)
	   {
		   DialOutRuleSet rs=this.sipOutRules.get(r);
		   Matcher mat=rs.pattern.matcher(dest);
		   if (mat.find())
		   {
			   dest=mat.replaceAll(rs.replacement);
			   break;
		   }
	   }
	   return dest;
   }   
   
   
}

class AMap
{
	   String callerUserId=null;
	   String realm=null;
	   String ip=null;
	   String dest=null;
	   AMap(String argCallerUserId,String argRealm,String argIP,String argDest)
	   {
		   this.callerUserId=argCallerUserId;
		   this.realm=argRealm;
		   this.ip=argIP;
		   this.dest=argDest;
	   }
}

class DialOutRuleSet
{
	Pattern pattern=null;
	String replacement=null;

	DialOutRuleSet(Pattern argPat,String argReplace)
	{
		   this.pattern=argPat;
		   this.replacement=argReplace;
	}

}