/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Author(s):
 * Greg Dorfuss
 */

/* Skype resets counters at GMT 0.
 * 
 * Big question - if the call runs past the reset time (0 GMT) where does the call get counted -
 * I think it's split between the two days.
 */

package local.ua;

import com.skype.Call;
import java.io.FileWriter;
import java.io.FileReader;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.BufferedReader;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Date;
import java.util.Enumeration;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;
import java.util.Hashtable;

public class CallHistoryHandler
{
	private String skypeUserid;
	private Logger log = null;
	private FileWriter writer=null;
	private String curLogName="";
	private String callLogPath=null;
	private String[] tollFreePrefixList=null;
	
	private int pstnMinutesToday=0;
	private Hashtable<String,Integer> pstnQualifiedCallHash=null;
	
	private Hashtable<Call.Type,Hashtable<String,CallHistoryEntry>> memLog=null;
	
	private String lockObj=new String();
	
	CallHistoryHandler(String argSkypeUserid,String argCallLogPath,String argTollFreeNumberList)
	{
		if (argTollFreeNumberList.length()>0)
			tollFreePrefixList=argTollFreeNumberList.replaceAll(" ", "").split(",");
		callLogPath=argCallLogPath;
		log = Logger.getLogger(this.getClass());
		skypeUserid=argSkypeUserid.toLowerCase().replaceAll("[^a-z0-9_.]", "");
	}

	public String getResetTime()
	{
		SimpleDateFormat format = new SimpleDateFormat("HH:mm:ss z");
		Calendar mCal = Calendar.getInstance();
		mCal.set(Calendar.HOUR_OF_DAY, 0);
		mCal.set(Calendar.MINUTE,0);
		mCal.set(Calendar.SECOND, 0);
		mCal.set(Calendar.MILLISECOND, 0);
		mCal.add(Calendar.MILLISECOND,mCal.getTimeZone().getRawOffset());
		return format.format(mCal.getTime());
	}
	
	public int getPstnTodayCountQualified()
	{
	   synchronized (lockObj)
	   {
		   if (!switchCallLog(null)) // make sure we look at today's data
			   return 0;
		   else
			   return this.pstnQualifiedCallHash.size();
	   }
	}

	public int getPstnTodayTimeQualified()
	{
	   synchronized (lockObj)
	   {
		   if (!switchCallLog(null)) // make sure we look at today's data
			   return 0;
		   else
			   return this.pstnMinutesToday;

	   }
	}
	
	public boolean isTollFreeNumber(String callTo)
	{
		if (tollFreePrefixList==null)
			return false;
		
		String tmp=callTo.replaceAll(" ", "").replaceAll("^[+]","").replaceAll("^00", "");
		for (int x=0;x<tollFreePrefixList.length;x++)
		{
			if (tmp.startsWith(tollFreePrefixList[x]))
				return true;
		}
		return false;
	}
		
	public boolean addCall(CallHistoryEntry entry)
	{
		if (entry.durationSeconds==0)
			return false;
		
		// determine if need to split into two logs due to midnight GMT
		
		// calc this entries reset time
		Calendar resetTime = Calendar.getInstance();
		resetTime.setTime(entry.startTime); // based on entry start date
		resetTime.set(Calendar.HOUR_OF_DAY, 0);
		resetTime.set(Calendar.MINUTE,0);
		resetTime.set(Calendar.SECOND, 0);
		resetTime.set(Calendar.MILLISECOND, 0);
		resetTime.add(Calendar.DATE, 1);
		resetTime.add(Calendar.MILLISECOND,resetTime.getTimeZone().getRawOffset());

		Calendar callStartTime = Calendar.getInstance();
		callStartTime.setTime(entry.startTime);
		
		if (callStartTime.getTimeInMillis()<resetTime.getTimeInMillis() && callStartTime.getTimeInMillis()+(entry.durationSeconds*1000)>=resetTime.getTimeInMillis())
		{
			//split call
			CallHistoryEntry entry2=new CallHistoryEntry(entry);
			// fix prior day entry
			entry.durationSeconds=(int)(resetTime.getTimeInMillis()-callStartTime.getTimeInMillis())/1000;
			entry.callCost="Rollover";
			// fix current day entry
			entry2.durationSeconds-=entry.durationSeconds;
			entry2.startTime=resetTime.getTime();
			addCallEntry(entry);
			addCallEntry(entry2);
			return true;
		}
		else
			return addCallEntry(entry);
	}
	
	
	private boolean addCallEntry(CallHistoryEntry entry)
	{

	   synchronized (lockObj)
	   {
			
			if (!switchCallLog(entry.startTime)) // make sure we are looking at the correct log
				return false;
			
			if (!memLog.containsKey(entry.callType))
				   memLog.put(entry.callType, new Hashtable<String,CallHistoryEntry>());
			
			Hashtable<String,CallHistoryEntry> typeLog=memLog.get(entry.callType);
			if (!typeLog.containsKey(entry.callId)) // ignore dupe call ids
			{
				try
				{
			      writer.write(entry+"\r\n");
			      writer.flush();
				}
				catch(Exception e)
				{
					log.error("addCall error",e);
					return false;
				}

				typeLog.put(entry.callId,entry); 
				
				if (entry.callType==Call.Type.OUTGOING_PSTN && isCallQualified(entry.callType, entry.callTo,entry.callCost))
				{
					if (!pstnQualifiedCallHash.containsKey(entry.callTo))
						this.pstnQualifiedCallHash.put(entry.callTo,1);
					this.pstnMinutesToday+=(int)((entry.durationSeconds+59)/60);
				}
					
				return true;
			}
			else
				return false;
	   }
	}
	
	private boolean switchCallLog(Date relevantDate)
	{
		SimpleDateFormat format = new SimpleDateFormat("yyyyMMdd");  
		Calendar mCal = Calendar.getInstance();
		if (relevantDate!=null)
			mCal.setTime( relevantDate);
		mCal.add(Calendar.MILLISECOND,-mCal.getTimeZone().getRawOffset()); // convert to GMT
		String newLogName=callLogPath+"callLog_"+skypeUserid+"_"+format.format(mCal.getTime())+".log";
		if (!newLogName.equals(curLogName) || writer==null)
		{
			log.debug("Call Log Switch: "+newLogName);
			
			try
			{
				if (writer!=null)
					writer.close();
				loadCallHistory(newLogName);
				writer=new FileWriter (new File(newLogName),true);
			}
			catch(Exception e)
			{
				log.error("switchCallLog error",e);
				curLogName="";
				return false;
			}
			curLogName=newLogName;
		}
		return true;
	}
	
	private boolean loadCallHistory(String logFileName)
	{
		boolean retvar=false;
		FileReader reader=null;
		BufferedReader bufReader=null;
		memLog=new Hashtable<Call.Type,Hashtable<String,CallHistoryEntry>>();
		
		try
		{
		  reader=new FileReader(new File(logFileName));
		}
		catch(FileNotFoundException e)
		{
		}
		catch(Exception e)
		{
			log.error("loadCallHistory error",e);
		}
		
		if (reader!=null)
		{	
			bufReader = new BufferedReader(reader);
			
			String data; 
			int line=0;
			try
			{
				while((data = bufReader.readLine()) != null) 
				{ 
					line++;
					CallHistoryEntry entry=null;
					try
					{
					   entry=new CallHistoryEntry(data);
					}
					catch(Exception e) // ignore corrupt entries
					{
					  log.warn("Corrupt call history entry skipped at line:"+line+" - "+e.getLocalizedMessage());	
					}
					
					if (entry!=null)
					{	
						if (!memLog.containsKey(entry.callType))
						   memLog.put(entry.callType, new Hashtable<String,CallHistoryEntry>());
						  memLog.get(entry.callType).put(entry.callId,entry);
					}
				} 
				bufReader.close();
				reader.close();
				retvar=true;
			}
			catch(Exception e)
			{
				log.error("loadCallHistory error",e);
			}
		}
		
		pstnQualifiedCallHash=hashQualifiedCalls(Call.Type.OUTGOING_PSTN);
		pstnMinutesToday=calcCallTimeQualified(Call.Type.OUTGOING_PSTN);
		
		return retvar;
	}
	
	private Hashtable<String,Integer> hashQualifiedCalls(Call.Type type)
	{
		Hashtable<String,CallHistoryEntry> v=memLog.get(type);

		Hashtable<String,Integer> um=new Hashtable<String,Integer>();
		if (v!=null)
		{	
			Enumeration en=v.elements();
			while (en.hasMoreElements())
			{
				CallHistoryEntry entry=(CallHistoryEntry) en.nextElement();
				if (isCallQualified(entry.callType,entry.callTo,entry.callCost))
				{	
				   if (!um.containsKey(entry.callTo))
					   um.put(entry.callTo,0);
				}
			}
		}
		return um;
	}

	private int calcCallTimeQualified(Call.Type type)
	{
		Hashtable<String,CallHistoryEntry> v=memLog.get(type);
		if (v==null)
			return 0;
		int totalMinutes=0;
		Enumeration en=v.elements();
		while (en.hasMoreElements())
		{
			CallHistoryEntry entry=(CallHistoryEntry) en.nextElement();
			if (isCallQualified(entry.callType,entry.callTo,entry.callCost))
				totalMinutes=totalMinutes+(int)((entry.durationSeconds+59)/60);
		}
		
		return totalMinutes;
	}

	private boolean isCallQualified(Call.Type type,String callTo,String callCost)
	{
		if (type!=Call.Type.OUTGOING_PSTN)
			return false;
		if (callCost.replaceAll("[^1-9]", "").length()>0) // paid calls don't count
			return false;
		// test if a toll free number - they don't count
		if (isTollFreeNumber(callTo))
			return false;

		return true; // passed all our tests
	}
	
/*	
	public static void main(String[] args) throws Exception
	{
	    PropertyConfigurator.configure("log.properties"); // needed only first time
		CallHistoryHandler test=new CallHistoryHandler("testid","log/","1800,1888,1866,1877");
		
		System.out.println("Reset time="+test.getResetTime());
		
		SimpleDateFormat formatter=new SimpleDateFormat("yyyyMMdd HH:mm:ss");
		if (false)
		{
			// make up some call data
			for (int x=0;x<10;x++)
				test.addCall(new CallHistoryEntry(Call.Type.OUTGOING_P2P, String.valueOf((int)(Math.random()*65535)), "sip:4353243244:5060", "+1800334234", formatter.parse("20090523 14:00:00"), (x+1)*60, "FREE"));
			for (int x=0;x<10;x++)
				test.addCall(new CallHistoryEntry(Call.Type.OUTGOING_PSTN, String.valueOf((int)(Math.random()*65535)), "sip:4353243244:5060", "+1800344234", formatter.parse("20090523 14:00:00"), (x+1)*60, "FREE"));
			for (int x=0;x<10;x++)
				test.addCall(new CallHistoryEntry(Call.Type.OUTGOING_PSTN, String.valueOf((int)(Math.random()*65535)), "sip:4353243244:5060", "+180031423"+x, formatter.parse("20090524 14:00:00"), (x+1)*60, "FREE"));
		}

		// split test
		test.addCall(new CallHistoryEntry(Call.Type.OUTGOING_PSTN,String.valueOf((int)(Math.random()*65535)), "sip:4353243244:5060", "+15551234567", formatter.parse("20090526 18:50:00"),60*65, "FREE"));		
		
		// paid call test
		test.addCall(new CallHistoryEntry(Call.Type.OUTGOING_PSTN,String.valueOf((int)(Math.random()*65535)), "sip:4353243244:5060", "+15551234568", formatter.parse("20090526 20:30:00"),20*65, "1.00 EUR"));		

		System.out.println("pstnQualifiedCallcnt="+test.getPstnTodayCountQualified());
		System.out.println("pstnCalltime="+test.getPstnTodayTimeQualified());
		
	}
*/
}
