/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Based on mjsip 1.6 software and skype4java
 * 
 * Author(s):
 * Greg Dorfuss
 */

package local.ua;


import org.zoolu.sip.address.NameAddress;
import org.zoolu.sip.address.SipURL;
import org.zoolu.sip.provider.SipProvider;
import org.zoolu.sip.message.Message;

import com.skype.Call;
import com.skype.CallStatusChangedListener;
import com.skype.Skype;
import com.skype.SkypeException;
import com.skype.Chat;
import com.skype.Friend;
import com.skype.connector.Connector;
import java.text.DecimalFormat;
import java.util.Date;

import org.apache.log4j.Logger;

/* Skype SIP user agent (UA).
 */
public class SSCallChannel extends Thread implements UserAgentListener, RegisterAgentListener, AuthTimerInterface, CallStatusChangedListener,SipCallRingTimerInterface, CallTimerInterface
{           

   /** User Agent */
   protected SkypeUserAgent ua;
   
   private SkypeProfile skype_profile;
   
   private Call currentSkypeCall=null;
   private boolean skypeAudioRedirected=false;
   
   private ControllerChannelInterface controller=null;
   
   private String authPin="";
   private int authFailCnt=0;
   private AuthState callAuthState=AuthState.IDLE;
   private enum AuthState {IDLE,SIP_WAITFORPIN, SIP_WAITFORDESTINATION,SKYPE_WAITFORPIN,SKYPE_WAITFORDESTINATION};
   private AuthTimer authTimer=null;

   private CallTimer callTimer=null;
   
   /** UserAgentProfile */
   private UserAgentProfile user_profile;
         
   private Logger log=null;
   
   private CallHistoryEntry callEntry=new CallHistoryEntry();;
   
   private int hangupStatus=403;
   
   private String curSipCaller="";
   private String curSipCallee="";
   private String curSipCallerIP="";
   private boolean doPinCalleeDial=false;
   
   private SipCommandRunner sipCmdRunner=null;
   private SkypeCommandRunner skypeCmdRunner=null;
   private SipCallRingTimer sipCallRingTimer=null;
   
   private boolean locked=false;
   
   private double glbSkypeCallRate=0;
   private int glbSkypeCallPrecision=0;
   private String glbSkypeCallCurrency="";
   private String glbSkypeCallid=null;
   private String glbPossibleRunaway=null;

   
   private long callStart=0;
   private long lastSipCallAttempt=0;
   private String joinSkypeUserId=null;
   
   /** Costructs a CallChannel */
   public SSCallChannel(SipProvider sip_provider, UserAgentProfile user_profile,SkypeProfile skype_profile,int argRtpPort,int argSkypePort,ControllerChannelInterface argCI,int chanId) throws Exception
   {  
	  this.setName(this.getClass().getName()+".#C"+chanId);
	  this.log = Logger.getLogger(this.getName());
	  this.controller=argCI;
	  this.skype_profile=skype_profile;
      this.user_profile=user_profile;
      this.ua=new SkypeUserAgent(sip_provider,user_profile,this,skype_profile,argRtpPort,argSkypePort,chanId);      
      this.hangupStatus=skype_profile.baseFailureResponse;
      start();
   }
   
   /** Makes a new sip call */
   public void call(String target_url)
   {  
      callEntry.callTo=target_url.toString();

	  lock();
	  ua.hangup();
      log.info("UAC: CALLING "+target_url.replaceAll(";.*",""));
      if (!ua.user_profile.audio && !ua.user_profile.video) log.info("ONLY SIGNALING, NO MEDIA");
      ua.call(target_url);       
   } 
         
         
   /** Receives incoming calls (auto accept) */
   public void listen()
   {  
	  logCall(); 
	  curSipCaller="";
	  curSipCallee="";
	  curSipCallerIP="";
	  doPinCalleeDial=false;
	  joinSkypeUserId=null;
	  
	  clearCallTimer();
	  
	  clearAuthTimer();

	  clearSipCommandRunner();

	  clearSkypeCommandRunner();
	  
	  clearSipCallRingTimer();
	  
	  callAuthState=AuthState.IDLE;
	  
      ua.hangup(hangupStatus);
      
      if (currentSkypeCall!=null && skype_profile.skypeInboundSipDestUnavailableAction.equals("ring"))
      {	  
    	 try
    	 {
    		 Call.Type ct=currentSkypeCall.getType();
	         if (currentSkypeCall.getStatus()==Call.Status.RINGING && (ct==Call.Type.INCOMING_P2P || ct==Call.Type.INCOMING_PSTN))
	         {	 
	    	  log.info("Allowing Skype Client to ring.");
			  currentSkypeCall.removeCallStatusChangedListener(this);
	    	  currentSkypeCall=null;
	         }
    	 }
    	 catch(Exception e)
    	 {log.debug("error",e);}
      }
    	  
   	  cancelSkypeCall();
      
      hangupStatus=skype_profile.baseFailureResponse;

      ua.listen(); 

      unLock();
      
      if (!ua.user_profile.audio && !ua.user_profile.video) log.info("ONLY SIGNALING, NO MEDIA");       
   } 


   /** Starts the UA */
   public void run()
   {
      try
      {  // Set the re-invite
         if (user_profile.re_invite_time>0)
         {  ua.reInvite(user_profile.contact_url,user_profile.re_invite_time);
         }

         // Set the transfer (REFER)
         if (user_profile.transfer_to!=null && user_profile.transfer_time>0)
         {  ua.callTransfer(user_profile.transfer_to,user_profile.transfer_time);
         }

          // UAS
          if (user_profile.accept_time>=0) 
           	log.info("UAS: AUTO ACCEPT MODE");
          listen();
         
      }
      catch (Exception e)  {  log.fatal("error",e); System.exit(SkypeUA.ExitCode.FATALERROR.code());  }

   
   }

   // ******************* UserAgent callback functions ******************

   /** When a new sip call is incoming */
   public void onUaCallIncoming(UserAgent tua, NameAddress callee, NameAddress caller,Message invite)
   {  
	     lock();
	     log.info("incoming sip call from "+caller.toString()+" callee="+callee.toString());
	     lastSipCallAttempt=System.currentTimeMillis();
	     callEntry.callFrom=caller.toString();
    	 
    	 //SessionDescriptor remote_sdp=new SessionDescriptor(tua.call.getRemoteSessionDescriptor());
    	 //curSipCallerIP=(new Parser(remote_sdp.getConnection().toString())).skipString().skipString().getString();
    	 curSipCallerIP=invite.getRemoteAddress();
         curSipCaller=caller.toString();
         curSipCallee=callee.toString();
	     handleSipCall(caller.toString(),curSipCallerIP,callee.toString());
   }
   
   /** When an inciming call has been confirmed */
   public void onUaCallConfirmed(UserAgent ua)
   {
	   		if (ua.onRemoteSipHold)
	   		{	
				log.info("Call Holding");
	   			return;
	   		}
	   		
	   		if (this.ua.skypeHoldingLocal && !this.ua.onRemoteSipHold)
	   		{	
	   			resumeSkypeCall();
				log.info("Call Resuming");
	   			return;
	   		}	
	   		
			log.info("CallConfirmed Active");
			
			if (sipCmdRunner!=null)
				sipCmdRunner.start();
			
			startCallTime();
   }
   
   /** When an outgoing sip call is remotly ringing */
   public void onUaCallRinging(UserAgent tua)
   {  
	   log.debug("onUaCallRinging");
   }

   /** When an outgoing sip call has been accepted */
   public void onUaCallAccepted(UserAgent tua)
   {
	   log.debug("onUaCallAccepted");
	   if (joinSkypeUserId!=null)
	   {
		 if (this.makeSkypeCall(joinSkypeUserId))
		 {	 
			 this.ua.queueSipClip(skype_profile.dialingFile);
		 }
		 else
		 {	 
			 log.error("Skype called Failed.");
			 this.ua.queueSipClip(skype_profile.invalidDestFile); 
			 this.listen();
		 }
	   }
	   else
	   {   
	     if (skypeAnswer())
	    	 startCallTime();
	     else
			 this.listen();
	   }
   }
   
   /** When a call has been trasferred */
   public void onUaCallTrasferred(UserAgent tua)
   {  
   }

   /** When an incoming sip call has been cancelled */
   public void onUaCallCancelled(UserAgent tua)
   {  
	   log.debug("onUaCallCancelled");
	   listen();
   }

   /** When an ougoing sip call has been refused or timeout */
   public void onUaCallFailed(UserAgent tua)
   {  
	   log.debug("onUaCallFailed");
   	   listen();
   }

   /** When a sip call has been locally or remotely closed */
   public void onUaCallClosed(UserAgent tua)
   {  
	   log.debug("onUaCallClosed");
       listen();     
   }
   
   // dmf received from sip
   public void onDtmfReceived(UserAgent tua,int digit,boolean isSkype)
   {
	   //log.debug("onDtmfReceived digit:"+digit+" sipDtmfBuffer="+ua.getDtmfBuffer()+" skypeDtmfBuf:"+ua.getSkypeDtmfBuffer());
	   if (!isSkype)
	   {
		   if (callAuthState==AuthState.SIP_WAITFORPIN)
		   {
	
			   if (tua.getDtmfBuffer().startsWith(authPin))
			  {
				  // got auth pin, send dest prompt and wait for dest entry
				  authFailCnt=0;
				  clearAuthTimer();
				  
				  if (doPinCalleeDial)
				  {
					  callAuthState=AuthState.IDLE;
					  String skypeDest=curSipCallee.replaceAll("(?i).*sip:<?([^@<]+)@.*", "$1");
					  if (directSkypeDial(skypeDest))
						  this.ua.queueSipClip(skype_profile.dialingFile); // would be nicer if we sent ring to sip device

				  }
				  else
				  {	  
					  callAuthState=AuthState.SIP_WAITFORDESTINATION;
					  log.info("Pin Authorized");
					  tua.chopDtmfBuffer(authPin.length());
					  this.ua.queueSipClip(skype_profile.destinationFile);
					  startAuthTimer(skype_profile.destinationTimeout);
				  }
			  }
		   }
		   else if (callAuthState==AuthState.SIP_WAITFORDESTINATION)
		   {
			   if (tua.getDtmfBuffer().indexOf("#")>=0)
			   {
				   // got end of dest entry, send dialing sound, make the skype call
				   authFailCnt=0;
				   clearAuthTimer();
				   callAuthState=AuthState.IDLE;
				   String skypeDest=ua.getDtmfBuffer().replaceAll("#.*$", "").trim();
				   this.ua.clearDtmfBuffer();
				   if (skypeDest.length()>0)
				   {	   
					   // make the skype Call
					   log.info("Dialing Skype Destination:"+skypeDest);
					   if (!makeSkypeCall(skypeDest))
					   {
						   log.info("Skype Call Failed");
						   callAuthState=AuthState.SIP_WAITFORDESTINATION;
						   this.ua.queueSipClip(skype_profile.invalidDestFile); 
						   this.ua.queueSipClip(skype_profile.destinationFile);
						   startAuthTimer(skype_profile.destinationTimeout);
					   }
					   else
						   this.ua.queueSipClip(skype_profile.dialingFile); // would be nicer if we sent ring to sip device
							   
				   }
				   else
				   {
					   callAuthState=AuthState.SIP_WAITFORDESTINATION;
					   this.ua.queueSipClip(skype_profile.destinationFile);
					   startAuthTimer(skype_profile.destinationTimeout);
				   }
			   }
		   }
		   else if (currentSkypeCall!=null && !this.ua.skypeHoldingLocal)
		   {
			   if (this.skype_profile.sendSipDtmfToSkype)
			   {	   
				   try
				   {
					   if (currentSkypeCall.getStatus()==Call.Status.INPROGRESS)
					   {	   
						   int fixDigit=digit;
						   if (digit==10)
							   fixDigit=Call.DTMF.TYPE_ASTERISK.ordinal();
						   else if (digit==11)
								fixDigit=Call.DTMF.TYPE_SHARP.ordinal();
						
						   if (fixDigit<=11)
						   {	   
						     currentSkypeCall.send(Call.DTMF.values()[fixDigit]);
						     log.debug("DTMF:"+digit+" sent to skype");
						   }
					   }   
				   }
				   catch (Exception e)
				   {
					   log.error("Error sending dtmf: "+digit+" to skype: ",e);
				   }
			   }
		   }
		   else if (callAuthState==AuthState.IDLE)
		   {
			   log.debug("Auth State not handled:"+callAuthState);
		   }
	   }
	   else
	   {	   
		   if (callAuthState==AuthState.SKYPE_WAITFORPIN)
		   {
			   
			  if (this.ua.getSkypeDtmfBuffer().startsWith(authPin))
			  {
				  // got auth pin, send dest prompt and wait for dest entry
				  authFailCnt=0;
				  clearAuthTimer();
				  callAuthState=AuthState.SKYPE_WAITFORDESTINATION;
				  log.info("Pin Authorized");
				  this.ua.chopSkypeDtmfBuffer(authPin.length());
				  this.ua.queueSkypeClip(skype_profile.skypeDestinationFile);
				  startAuthTimer(skype_profile.destinationTimeout);
			  }
		   }
		   else if (callAuthState==AuthState.SKYPE_WAITFORDESTINATION)
		   {
			   if (this.ua.getSkypeDtmfBuffer().indexOf("#")>=0)
			   {
				   // got end of dest entry, send dialing sound, make the skype call
				   authFailCnt=0;
				   clearAuthTimer();
				   callAuthState=AuthState.IDLE;
				   String sipDest=ua.getSkypeDtmfBuffer().replaceAll("#.*$", "").trim();
				   sipDest=this.controller.getAuthMap().sipOutFilter(sipDest);
				   this.ua.clearSkypeDtmfBuffer();
				   if (sipDest.length()>0)
				   {
					   log.info("Dialing Sip Destination:"+sipDest);
					   this.ua.queueSkypeClip(skype_profile.skypeDialingFile); // would be nicer if we sent ring to skype
					   this.waitForSkypeClipsComplete();

					   // set the skype id
					   try
					   {
						 NameAddress tmpfrom;
						 String skypeid=currentSkypeCall.getPartnerId();	
						 if (skype_profile.replaceFromWithSkypeId)
						 {
							 SipURL tmpurl=new SipURL(ua.user_profile.contact_url);
					    	 tmpfrom=new NameAddress(skypeid,new SipURL(skypeid,tmpurl.getHost(),tmpurl.getPort()));
						 }
						 else
							 tmpfrom=new NameAddress(skypeid,new SipURL(ua.user_profile.contact_url));
					     
					     ua.user_profile.from_url=tmpfrom.toString();
					   }
					   catch (Exception e)
					   {
					     log.error("Error",e);
					     listen();
					     return;
					   }

					   // make the sip Call
					   this.call(sipDest);
				   }
				   else
				   {
					   callAuthState=AuthState.SKYPE_WAITFORDESTINATION;
					   this.ua.queueSkypeClip(skype_profile.skypeDestinationFile);
					   startAuthTimer(skype_profile.destinationTimeout);
				   }
			   }
		   }
		   else if (this.ua.call_state!=UserAgent.UA_IDLE && !this.ua.onLocalSipHold)
		   {
			   if (this.skype_profile.sendSkypeDtmfToSip)
			   {	
					   this.ua.queueSipDtmfDigits(this.ua.dtmfConvertToString(digit));
					   log.debug("DTMF:"+digit+" sent to sip");
			   }
		   }
		   else if (callAuthState==AuthState.IDLE)
		   {
			   log.debug("Auth State not handled:"+callAuthState);
		   }	   
	   }
   }
   // AuthTimerInterface callback
   public void onAuthTimeout()
   {
	   authTimer=null;
	   if (callAuthState!=AuthState.IDLE)
	   {	 
		   authFailCnt++;
		   if (callAuthState==AuthState.SIP_WAITFORPIN)
		   {
			   log.info("Pin timeout");
			   
			   if (this.ua.getDtmfBuffer().length()>0 && authFailCnt<skype_profile.pinRetryLimit) // they tried, so restart
			   {
				   this.ua.queueSipClip(skype_profile.invalidPinFile); 
				   this.ua.clearDtmfBuffer();
				   this.ua.queueSipClip(skype_profile.pinFile);
				   startAuthTimer(skype_profile.pinTimeout);
			   }
			   else
			   {	   
				   this.ua.queueSipClip(skype_profile.invalidPinFile);
				   waitForSipClipsComplete();
				   listen();
			   }
		   }
		   else if (callAuthState==AuthState.SIP_WAITFORDESTINATION)
		   {
			   log.info("Destination timeout");
			   if (this.ua.getDtmfBuffer().length()>0 && authFailCnt<skype_profile.destRetryLimit) // they tried, so restart
			   {
				   this.ua.queueSipClip(skype_profile.invalidDestFile); 
				   this.ua.clearDtmfBuffer();
				   this.ua.queueSipClip(skype_profile.destinationFile);
				   startAuthTimer(skype_profile.destinationTimeout);
			   }
			   else
			   {	   
				   this.ua.queueSipClip(skype_profile.invalidDestFile); 
				   waitForSipClipsComplete();
				   listen();
			   }

		   }
		   else if (callAuthState==AuthState.SKYPE_WAITFORPIN)
		   {
			   log.info("Pin timeout");
			   
			   if (this.ua.getSkypeDtmfBuffer().length()>0 && authFailCnt<skype_profile.pinRetryLimit) // they tried, so restart
			   {
				   this.ua.queueSkypeClip(skype_profile.skypeInvalidPinFile); 
				   this.ua.clearSkypeDtmfBuffer();
				   this.ua.queueSkypeClip(skype_profile.skypePinFile);
				   startAuthTimer(skype_profile.pinTimeout);
			   }
			   else
			   {	   
				   this.ua.queueSkypeClip(skype_profile.skypeInvalidPinFile);
				   waitForSkypeClipsComplete();
				   cancelSkypeCall();
				   unLock();
			   }
		   }
		   else if (callAuthState==AuthState.SKYPE_WAITFORDESTINATION)
		   {
			   log.info("Destination timeout");
			   if (this.ua.getSkypeDtmfBuffer().length()>0 && authFailCnt<skype_profile.destRetryLimit) // they tried, so restart
			   {
				   this.ua.queueSkypeClip(skype_profile.skypeInvalidDestFile); 
				   this.ua.clearSkypeDtmfBuffer();
				   this.ua.queueSkypeClip(skype_profile.skypeDestinationFile);
				   startAuthTimer(skype_profile.destinationTimeout);
			   }
			   else
			   {	   
				   this.ua.queueSkypeClip(skype_profile.skypeInvalidDestFile); // hangup after
				   waitForSkypeClipsComplete();
				   cancelSkypeCall();
				   unLock();
			   }	   

		   }
		   else
		   {
			   cancelSkypeCall();
			   listen();
		   }
	   }
   }

   public void startSipAuthSequence(String argPin)
   {
	    authPin=argPin;
	  
	    //log.info("Pin="+authPin);
		   
	    if (authPin.trim().length()==0)
		   this.ua.queueSipClip(skype_profile.destinationFile);
	    else
		   this.ua.queueSipClip(skype_profile.pinFile);
	   
		authFailCnt=0;
		if (authPin.trim().length()==0)
	    {	   
		   callAuthState=AuthState.SIP_WAITFORDESTINATION;
		   authTimer=new AuthTimer(skype_profile.destinationTimeout,this);
		   authTimer.start();
	    }
	    else
	    {
		   callAuthState=AuthState.SIP_WAITFORPIN;
		   authTimer=new AuthTimer(skype_profile.pinTimeout,this);
		   authTimer.start();
	    }
   }
   
   public void startSkypeAuthSequence(String argPin)
   {
	    authPin=argPin;
	  
	    //log.info("Pin="+authPin);
		   
	    if (authPin.trim().length()==0)
		   this.ua.queueSkypeClip(skype_profile.skypeDestinationFile);
	    else
		   this.ua.queueSkypeClip(skype_profile.skypePinFile);
	   
		authFailCnt=0;
	    if (authPin.trim().length()==0)
	    {	   
		   callAuthState=AuthState.SKYPE_WAITFORDESTINATION;
		   authTimer=new AuthTimer(skype_profile.destinationTimeout,this);
		   authTimer.start();
	    }
	    else
	    {
		   callAuthState=AuthState.SKYPE_WAITFORPIN;
		   authTimer=new AuthTimer(skype_profile.pinTimeout,this);
		   authTimer.start();
	    }
   }

   private void clearAuthTimer()
   {
	   if (authTimer!=null)
		   authTimer.stopTimer();
	   authTimer=null;
   }
   
   private void startAuthTimer(int timeout)
   {
		  authTimer=new AuthTimer(timeout,this);
		  authTimer.start();
   }
   
   public void clearSipCommandRunner()
   {
	   if (sipCmdRunner!=null)
		   sipCmdRunner.stopCommands();
	   sipCmdRunner=null;
   }

   public void clearSkypeCommandRunner()
   {
	   if (skypeCmdRunner!=null)
		   skypeCmdRunner.stopCommands();
	   skypeCmdRunner=null;
   }

   // **************** RegisterAgent callback functions *****************

   /** When a UA has been successfully (un)registered. */
   public void onUaRegistrationSuccess(RegisterAgent ra, NameAddress target, NameAddress contact, String result)
   {  log.info("Registration success: "+result);
   }

   /** When a UA failed on (un)registering. */
   public void onUaRegistrationFailure(RegisterAgent ra, NameAddress target, NameAddress contact, String result)
   {  log.info("Registration failure: "+result);
   }
   

	private void handleSipCall(String callerSipCID,String sipCIP, String argdestination)
	{
		// argdestination = <sip:66@192.168.0.4:5070>
		String destination=this.controller.getAuthMap().getSkypeDest(callerSipCID, sipCIP, argdestination);
		
		log.debug("handleSipCall - authMap:"+destination);

		if (destination==null || destination.length()==0)
		{
			  log.info("handleSipCall - rejected call");
	          listen();
	          return;
		}
		else if (destination.indexOf(":")<0)
		{
			directSkypeDial(destination);
		}
		else
		{
			log.info("handleSipCall - Command List:"+destination);
		    if (ua.call_state==UserAgent.UA_INCOMING_CALL)
			   ua.accept();
			
			sipCmdRunner=new SipCommandRunner(destination,this);
		}
		
	}
	
	private boolean directSkypeDial(String destination)
	{
		log.info("Skype Dial:"+destination);
		if (!makeSkypeCall(destination))
		{
	          listen();
	          return false;
		}		
		return true;
	}
	
	/* when skype call status changes */
	public void statusChanged(Call.Status status) throws SkypeException
	{
		if (status==Call.Status.FINISHED || status==Call.Status.CANCELLED || status==Call.Status.MISSED || status==Call.Status.REFUSED || status==Call.Status.FAILED || status==Call.Status.UNPLACED || status==Call.Status.BUSY)
		{
			// tear down the call
			log.info("skypeCallStatus - Complete: "+status);
			if (currentSkypeCall!=null)
			{	
				currentSkypeCall.removeCallStatusChangedListener(this);
				cancelSkypeCall(); // unplaced status sometimes leaves skype on a call - so squash it
			}
			this.ua.stopMedia();

			hangupStatus=skype_profile.baseFailureResponse;
			if (status==Call.Status.REFUSED)
				hangupStatus=skype_profile.skypeRefusedResponse;
			else if (status==Call.Status.FAILED) // when invalid user, no skype credit or something
				hangupStatus=skype_profile.skypeFailedResponse;
			else if (status==Call.Status.UNPLACED)
				hangupStatus=skype_profile.skypeUnPlacedResponse;
			else if (status==Call.Status.BUSY)
				hangupStatus=skype_profile.skypeBusyResponse;
			
			listen();
			
		}
		else if (skype_profile.handleEarlyMedia && status==Call.Status.EARLYMEDIA)
		{
			// skypeout earlymedia handling
			try
			{
				// start skype media and accept the sip call
				log.info("skypeCallStatus - "+status);
				if (ua.call_state==UserAgent.UA_INCOMING_CALL)
				{
					if (skype_profile.sendSkypeEarlyMediaOverSipSessionProgress)
						ua.sendEarlyMedia();
					else	
						ua.accept();
				}	
				this.redirSkypeAudio();
				this.ua.startSkypeMedia(skype_profile.sendSkypeEarlyMediaOverSipSessionProgress);  
			}
			catch(Exception e)
			{
				
				log.error("skypeCallStatus: error",e);
				cancelSkypeCall();
				listen();
			}
			   
		}
		else if (status==Call.Status.INPROGRESS)
		{
			try
			{
				callEntry.callType=currentSkypeCall.getType();
				if (ua.call_state==UserAgent.UA_OUTGOING_CALL && (callEntry.callType==Call.Type.INCOMING_P2P || callEntry.callType==Call.Type.INCOMING_PSTN))
				{ 
						log.info("Incoming Skype Call Manually Answered.");
						currentSkypeCall.removeCallStatusChangedListener(this);
						currentSkypeCall=null;
						listen(); // this will stop the sip outbound call
						return;
				}
			
				// start skype media and accept the sip call
				log.info("skypeCallStatus - "+status);
		        getSkypeCallRate();
				if (ua.call_state==UserAgent.UA_INCOMING_CALL)
				   ua.accept();
				
				this.redirSkypeAudio();
				this.ua.startSkypeMedia(false);
				startCallTime();
				
				this.ua.skypeHoldingLocal=false;
				this.ua.skypeHoldingRemote=false;
			}
			catch(Exception e)
			{
				log.error("error",e);
				cancelSkypeCall();
				listen();
			}
			   
		}
		else if (status==Call.Status.LOCALHOLD)
		{
			log.info("skypeCallStatus - "+status);
			this.ua.skypeHoldingLocal=true;
		}
		else if (status==Call.Status.REMOTEHOLD)
		{
			log.info("skypeCallStatus - "+status);
			this.ua.skypeHoldingRemote=true;
		}
		else if (status==Call.Status.ROUTING || status==Call.Status.RINGING)
		{
			log.info("skypeCallStatus - "+status);
			// let's redir the audio asap
		    redirSkypeAudio();
		}	
		else
		{	
			log.info("skypeCallStatus - "+status);
		}
		
	}
	
	private boolean makeSkypeCall(String dest)
	{
		boolean retvar=false;
		boolean retry=true;
		
		if (!skype_profile.skypeClientSupportsMultiCalls)
		{	
			try
			{
				controller.holdOtherSkypeCalls(null);
			}
			catch (Exception e)
			{
				log.error("makeSkypeCall: holdOtherSkypeCalls failed. ",e);
			    return false;
			}
		}		
		
		String filteredDest=this.controller.getAuthMap().skypeOutFilter(dest);
		if (!filteredDest.equals(dest))
		{
			dest=filteredDest;
			log.info("Actual number dialed:"+dest);
		}
		
		if (dest.length()<1)
		{
			log.error("makeSkypeCall: invalid destination");
			return false;
		}
		
		if (isOverUsageLimit(dest))
		{
			this.hangupStatus=skype_profile.overUsageLimitSipResponse;
			log.info("Call rejected - Usage Limit Reached");
			this.listen();
			return false;
		}
		
		
		while (retry)
		{	
			try
			{
				Skype.getVersion(); // test connection now
				
				if (skype_profile.sendSkypeIM)
					sendSkypeIM(dest);
				
				setCurrentSkypeCall(Skype.call(dest));
		        callEntry.callTo=dest;
		        retvar=true;
		        break;
			}
			catch(com.skype.NotAttachedException e)
			{
				retry=this.controller.lostSkypeConnection();
				if (!retry) 
				{	
					this.listen();
			   	    log.error("Reconnect to Skype client failed.");
					this.controller.restart();
				}
			}
			catch(SkypeException e)
			{
				  retry=false;
				  if (e.getLocalizedMessage()==null)
				  {
					  log.error("Error: makeSkypeCall: skypeDest="+dest,e);
				  }
				  else if (e.getLocalizedMessage().contains("Unrecognised identity"))
				  {	  
					  this.hangupStatus=skype_profile.skypeFailedResponse;
					  log.error("Error: makeSkypeCall: skypeDest="+dest+" "+e.getLocalizedMessage());
				  }
				  else if (e.getLocalizedMessage().contains("Cannot call yourself"))
				  {
					  log.error("Error: makeSkypeCall: skypeDest="+dest+" "+e.getLocalizedMessage());
				  }
				  else if (e.getLocalizedMessage().contains("command execution failed"))
				  {
					    log.error("Error: makeSkypeCall: skypeDest="+dest+" "+e.getLocalizedMessage());
					    retry=this.controller.lostSkypeConnection();
						if (!retry) 
						{	
							this.listen();
					   	    log.error("Reconnect to Skype client failed.");
							this.controller.restart();
						}
				  }
				  else
					  log.error("Error: makeSkypeCall: skypeDest="+dest+" "+e.getLocalizedMessage(),e);
				  break;
			}
		}
		return retvar;
	}
	
	private void sendSkypeIM(String dest)
	{
		// skypeout starts with + or 00
		if (dest.startsWith("+") || dest.startsWith("00"))
			return;
		
		// ignore skype out calls
		int alphalen=dest.replaceAll("[^\\p{Alpha}]", "").length();
		int numlen=dest.replaceAll("[^0-9]", "").length();

		if (alphalen>0 || numlen<7)
		{
			if (alphalen==0) // get userid of speed dial
				dest=getUserFromSpeedDial(dest);

			// skype user must be a letter
			char firstChar=dest.toLowerCase().charAt(0);
			if (firstChar<'a' || firstChar>'z')
				return;

			log.info("Sending Skype IM to: "+dest);
			String msg=skype_profile.skypeImMessage;
			// replace [callerid] with sip callers ID
			msg=msg.replaceAll("\\x5bcallerid\\x5d",curSipCaller.replaceAll("(^\"|\" .*$)", ""));
			
			log.info("IM: "+msg);
			try
			{
				Chat chat = Skype.chat(dest);
				chat.send(msg);  
				
			}
			catch(SkypeException e)
			{
		        if (e.getLocalizedMessage()==null)
					log.error("Error: sendSkypeIM: skypeDest="+dest,e);
		        else	
		        	log.error("Error: sendSkypeIM: skypeDest="+dest+" "+e.getLocalizedMessage(),e);
			}
			
			try	{Thread.sleep(this.skype_profile.sendSkypeImDelay*1000);}catch(Exception e)	{}
		}
	}
	
	private String getUserFromSpeedDial(String dest)
	{
		// get userid of speed dial, if not matched returns original data
		try
		{
			Friend[] fr = Skype.getContactList().getAllFriends();
			for (int s=0;s<fr.length;s++)
			{	
				if (fr[s].getSpeedDial().toString().equals(dest))
				{
					dest=fr[s].getId();
					break;
				}
			}
		}
		catch(SkypeException e)
		{
			if (e.getLocalizedMessage()==null)
				log.error("Error: getUserFromSpeedDial: skypeDest="+dest,e);
			else	
				log.error("Error: getUserFromSpeedDial: skypeDest="+dest+" "+e.getLocalizedMessage(),e);
		}
		return dest;
	}

	public void cancelSkypeCall()
	{
   	    skypeAudioRedirected=false;

   	    Call curCall=currentSkypeCall;
		if (curCall!=null)
		{	
		    try
		    {
		      glbPossibleRunaway=curCall.getId();
		    }
		    catch(Exception e2)
		    {}

		    try
			{
		    	curCall.cancel();
			}
			catch(Throwable e)
			{
			    String msg=e.getLocalizedMessage();
			    if (msg==null || msg.indexOf("Cannot hangup inactive call")<0)
			    	log.debug("cancelSkypeCall:",e);
			}
		}
	    currentSkypeCall=null;
	}


	
	public boolean skypeAnswer()
	{
		boolean ansStatus=true;
		
		log.debug("skypeAnswer");

		
		if (!skype_profile.skypeClientSupportsMultiCalls)
		{	
			if (currentSkypeCall==null)
			{
				log.debug("No skype call to answer");
			    this.cancelSkypeCall();
				return false;
			}

			try
			{
				controller.holdOtherSkypeCalls(this.currentSkypeCall);
			}
			catch (Exception e)
			{
				log.error("skypeAnswer: holdOtherSkypeCalls failed. ",e);
			    this.cancelSkypeCall();
			    ansStatus=false;
			    return false;
			}
		}
		
		if (currentSkypeCall==null)
		{
			log.debug("No skype call to answer");
		    this.cancelSkypeCall();
			return false;
		}
			
		Call.Status curStat=null;
		try
		{
			curStat=currentSkypeCall.getStatus();
			if (curStat==Call.Status.INPROGRESS)
				return true;
			else if (curStat==Call.Status.MISSED || curStat==Call.Status.CANCELLED 
					|| curStat==Call.Status.FAILED || curStat==Call.Status.FINISHED 
					|| curStat==Call.Status.BUSY || curStat==Call.Status.REFUSED 
					|| curStat==Call.Status.UNPLACED)
			{	
				ansStatus=false;
			}
			else
			{	
				currentSkypeCall.answer();
				this.redirSkypeAudio();
				this.ua.startSkypeMedia(false);
				callEntry.callFrom=currentSkypeCall.getPartnerId();;
			}
		}
		catch(Exception e)
		{
			if (currentSkypeCall!=null) // currentSkypeCall can be null if skype caller hungup before call gets answered
			{
			    log.error("skypeAnswer: CallStatus="+curStat,e);
			}
			else
				log.debug("No skype call to answer");
		    this.cancelSkypeCall();
		    ansStatus=false;
		}
		
		return ansStatus;
	}
	
	public void waitForSkypeClipsComplete()
	{
		try
		{
			while (currentSkypeCall.getStatus()==Call.Status.INPROGRESS && !this.ua.getSkypeClipQueueComplete())
			{	
				Thread.sleep(500);
			}
		}
		catch(Exception e)
		{
			log.error("waitForSkypeClipsComplete:",e);
		}
	   try{Thread.sleep(500);}catch(Exception e){}
   }
   
	public void waitForSipClipsComplete()
	{
			while (ua.call_state==UserAgent.UA_ONCALL && !this.ua.getSipClipQueueComplete())
			{	
				try
				{
				Thread.sleep(500);
				}
				catch(Exception e)
				{
				}
			}
			try{Thread.sleep(500);}catch(Exception e){}		
	}
	
	public boolean isIdle()
	{
		if (ua.call_state==UserAgent.UA_IDLE && this.currentSkypeCall==null && !this.locked)
			return true;
		else
			return false;
	}
	
	public boolean answerSkypeCall(String sipDest)
	{
		if (skypeAnswer())
		{	
			this.skypeCmdRunner=new SkypeCommandRunner(sipDest,this);
			this.skypeCmdRunner.start();
			return true;
		}
		else
			return false;
	}
	
	public Call getCurrentSkypeCall()
	{
		return this.currentSkypeCall;
	}
	public void setCurrentSkypeCall(Call call)
	{
		this.currentSkypeCall=call;
	}
	
	public String getSipDest() throws SkypeException
	{
		return this.controller.getAuthMap().getSipDest(this.currentSkypeCall.getPartnerId());
	}

	public void lock()
	{
		this.locked=true;
	}
	public void unLock()
	{
		this.locked=false;
	}
	
	public void skypeCallMaked()
	{
		   this.currentSkypeCall.addCallStatusChangedListener(this);
		   //	let's redir the audio asap
		   redirSkypeAudio();
	}
	
	public void redirSkypeAudio()
	{
		if (currentSkypeCall!=null && !skypeAudioRedirected)
		{	
		   try
		   {
		    String skypeCallid=currentSkypeCall.getId();
	        Connector.getInstance().execute("ALTER CALL "+skypeCallid+" SET_OUTPUT PORT=\""+ua.skypeOutPort+"\""); // this is the received audio from remote skpe user
    	    Connector.getInstance().execute("ALTER CALL "+skypeCallid+" SET_INPUT PORT=\""+ua.skypeInPort+"\""); // this is the audio to be sent to the remote skype user
     	    log.debug("###  redirSkypeAudio success");
			skypeAudioRedirected=true;
		   }
		   catch(Exception e)
		   {
				log.debug("redirSkypeAudio: error",e);
		   }
		}
	}
	
	

	public void resumeSkypeCall()
	{
		try
		{
			currentSkypeCall.resume();
			//this.ua.skypeHoldingLocal=false;
		}
		catch (Exception e)
		{
			log.error("Error resuming skype call: ",e);
			listen();
		}
	}

	public void holdSkypeCall() throws Exception
	{
		currentSkypeCall.hold();
	}

	public void sendSkypeDtmfDigits(String digits)
	{
		for (int p=0;p<digits.length();p++)
		{
		   char digit=digits.charAt(p);
		   int sDig=-1;
		   if (digit=='*')
	        	sDig=Call.DTMF.TYPE_ASTERISK.ordinal();
	        else if (digit=='#')
	        	sDig=Call.DTMF.TYPE_SHARP.ordinal();
	        else if (digit>='0' && digit<='9')
	        	sDig=(digit-48);
		   
		   try
		   {
			   currentSkypeCall.send(Call.DTMF.values()[sDig]);
			   log.debug("DTMF:"+digit+" sent to skype");
			   Thread.sleep(800);
		   }
		   catch (Exception e)
		   {
			   log.error("Error sending dtmf: "+digit+" to skype: ",e);
		   }
		}
	}
	
	private synchronized void startCallTime()
	{
		if (callStart>0)
			return;
		
		if (currentSkypeCall==null)
			return;
		
		try
		{
		  if (currentSkypeCall.getStatus()!=Call.Status.INPROGRESS)
			 return;
		}
		catch (SkypeException e)
		{
			log.error("startCallTime:",e);
			return;
		}
		
		if (this.ua.call_state!=UserAgent.UA_ONCALL)
			return;
		
		// to get here, not already here before and skype and sip call in progress
		callStart=System.currentTimeMillis();
		callEntry.startTime=new Date();
		try
		{
		  //already set callEntry.callType=currentSkypeCall.getType();
		  //callEntry.callId=currentSkypeCall.getId(); // seems their id's can change for the same call???
	      callEntry.callId=Long.toString((long)(currentSkypeCall.getStartTime().getTime()/1000));

		}
		catch(Exception e)
		{log.error("startCallTime error",e);}
		
		clearSipCommandRunner();
		clearSkypeCommandRunner();
		startCallTimer();
	}
	
	private void logCall()
	{
		if (callStart>0)
		{	
			  callEntry.durationSeconds=(System.currentTimeMillis()-callStart)/1000;
	    	  callStart=0;
	    	  long minutes=callEntry.durationSeconds/60;
	    	  String seconds=Long.toString(callEntry.durationSeconds%60+100).substring(1);
	    	  
	    	  callEntry.callCost=getSkypeCallCost();
	    	  
	    	  double curBal=util.getSkypeCreditBalance();
	    	  
	    	  log.info(callEntry.callType+" From: "+callEntry.callFrom+" To: "+callEntry.callTo+" CallTime: "+minutes+":"+seconds+" Cost: "+callEntry.callCost+" "+util.formatSkypeCreditBalance(curBal));
	    	  this.controller.getCallHistoryHandler().addCall(callEntry);
	    	  
	    	  if (callEntry.callType==Call.Type.OUTGOING_PSTN)
	    		  this.controller.showCallHist();
	    	  
	    	  
	    	  callEntry=new CallHistoryEntry();
	    	  
	    	  this.controller.updateSkypeCreditBalance(curBal);
		}  
	}
	
	void halt()
	{
		this.ua.halt();
		this.interrupt();
	}
	
	public long getLastSipCallTime()
	{
		return lastSipCallAttempt;
	}
	
	public void setPinCalleeDial()
	{
		doPinCalleeDial=true;
	}

	public boolean doSkypeSIPJoin(String skypeUserId)
	{
	    String sipDest=this.controller.getAuthMap().getSipDest(skypeUserId);
		if (sipDest==null || sipDest.length()==0)
		{
			  log.info("doSkypeSIPJoin - rejected call - no SIP Destination");
			  this.listen();
			  return false;
		}
		else
		{
			joinSkypeUserId=skypeUserId;
			if (sipDest.toLowerCase().indexOf("sip:")>=0)
			{
			     sipDest=this.controller.getSipUserContact(sipDest);
			     callEntry.callFrom=sipDest;
			     
			     NameAddress tmpfrom=new NameAddress(this.ua.user_profile.contact_url);
			     tmpfrom.setDisplayName(skypeUserId);
			     
			     this.ua.user_profile.from_url=tmpfrom.toString();

		    	 log.info("doSkypeSIPJoin - Direct SIP Dial to:"+sipDest.replaceAll(";.*","")+" from:"+this.ua.user_profile.from_url);
		    	 this.call(sipDest);
		    	 sipCallRingTimer=new SipCallRingTimer(20,this);
			}
			else
			{
				log.info("doSkypeSIPJoin - Command List:"+sipDest);
				this.answerSkypeCall(sipDest);
			}
			return true;
		}	
	}
	
	public void onSipCallRingTimeout()
	{
		// when called - cancels the sip call if not connected yet
		if (this.ua.call_state!=UserAgent.UA_ONCALL)
		{
			log.info("SIP destination did not answer. Call Cancelled.");
			this.listen();
		}
		clearSipCallRingTimer();
	}
	
   private void clearSipCallRingTimer()
   {
		   if (sipCallRingTimer!=null)
			   sipCallRingTimer.stopTimer();
		   sipCallRingTimer=null;
   }

	private void getSkypeCallRate()
	{
	    glbSkypeCallRate=0;
	    glbSkypeCallPrecision=0;
	    glbSkypeCallCurrency="";
	    glbSkypeCallid=null;

	    if (currentSkypeCall==null)
	    	return;

		try
		{
		    glbSkypeCallid=currentSkypeCall.getId();
		    
			if (callEntry.callType==Call.Type.OUTGOING_PSTN)
			{
				glbSkypeCallRate=Long.parseLong(util.parseSkypeResponse("GET CALL "+glbSkypeCallid+" RATE","CALL "+glbSkypeCallid+" RATE"));
				if (glbSkypeCallRate==0)
					log.info("Call Rate: Free");
				else
				{	
					glbSkypeCallPrecision=Integer.parseInt(util.parseSkypeResponse("GET CALL "+glbSkypeCallid+" RATE_PRECISION","CALL "+glbSkypeCallid+" RATE_PRECISION"));
					if (glbSkypeCallPrecision==0)
						glbSkypeCallPrecision=1;
					glbSkypeCallRate=glbSkypeCallRate / Math.pow(10, glbSkypeCallPrecision);
					glbSkypeCallCurrency=util.parseSkypeResponse("GET CALL "+glbSkypeCallid+" RATE_CURRENCY","CALL "+glbSkypeCallid+" RATE_CURRENCY");
					log.info("Call Rate: "+util.formatAmount(glbSkypeCallRate,glbSkypeCallPrecision)+" "+glbSkypeCallCurrency);
				}	
			}
		}
		catch(Throwable e)
		{
			log.debug("getSkypeCallRate",e);
		}	
	}
	
	private String getSkypeCallCost()
	{
		if (glbSkypeCallid==null)
			return "";
		
		try
		{
			if (callEntry.callType==Call.Type.OUTGOING_PSTN)
			{
				callEntry.durationSeconds=Integer.parseInt(util.parseSkypeResponse("GET CALL "+glbSkypeCallid+" DURATION","CALL "+glbSkypeCallid+" DURATION")); // user their offical time
				if (glbSkypeCallRate>0)
				{	
					long durMin=(callEntry.durationSeconds+59)/60;
					double callCost = (durMin * glbSkypeCallRate)+skype_profile.connectionFee;
					return util.formatAmount(callCost,2)+" "+glbSkypeCallCurrency;
				}
				else
					return "FREE";
			}
		}
		catch(Throwable e)
		{
			log.debug("appendSkypeCallCost:",e);
		}	
		return "FREE";
	}   
   

   public void onCallTimeoutWarning()
   {
	  log.info("Call Time Limit Warning.");
	  if (skype_profile.overLimitWarningFile.length()>0)
	  {	  
		 ua.queueSipClip(skype_profile.overLimitWarningFile);
	     ua.queueSkypeClip(skype_profile.overLimitWarningFile);
	  }
   }

   public void onCallTimeout()
   {
	  log.info("Call Time Limit reached - call Terminated.");
	  listen();
   }

   private void clearCallTimer()
   {
	   if (callTimer!=null)
		   callTimer.stopTimer();
	   callTimer=null;
   }
   
   private void startCallTimer()
   {
	   int cutOff=calcMaxCallTime();
	   if (cutOff>0)
	   {
	     log.info("This call is limited to: "+cutOff+" minutes");
	     callTimer=new CallTimer(cutOff,skype_profile.warnMinutesBeforeCutoff,this);
	     callTimer.start();
	   }
	   else
		 callTimer=null;
   }
	
   private int calcMaxCallTime()
   {
	   //  calc cutoff time section     
	   int cutOff=skype_profile.maxCallTimeLimitMinutes;
	   if (this.callEntry.callType==Call.Type.OUTGOING_PSTN)
	   {
	       if (skype_profile.MaxPstnCallTimeLimitMinutes>0 && (skype_profile.MaxPstnCallTimeLimitMinutes<cutOff || cutOff==0))
	           cutOff=skype_profile.MaxPstnCallTimeLimitMinutes;

	       if (skype_profile.dailyPstnLimitMinutes>0)
	       {
	           int pstnRemaining=skype_profile.dailyPstnLimitMinutes-this.controller.getCallHistoryHandler().getPstnTodayTimeQualified();
	           if (pstnRemaining<=0)
	               log.error("remaining time lte 0???");
	           if (pstnRemaining<cutOff || cutOff==0)
	              cutOff=pstnRemaining;
	       }   
	   }

	   return cutOff;
   }
   
	private boolean isOverUsageLimit(String filteredDest)
	{
		String tmp=filteredDest.replaceAll(" ", "").replaceAll("^[+]","").replaceAll("^00", "");
		if (tmp.length()<1)
			return false;
		if (Character.isDigit(tmp.charAt(0))) // PSTN call test
		{	
			   CallHistoryHandler chh=this.controller.getCallHistoryHandler();
			   if (chh.isTollFreeNumber(tmp))
				   return false;
			   // call count limit check
			   if (skype_profile.dailyPstnUniqueNumberLimit>0 && chh.getPstnTodayCountQualified()>=skype_profile.dailyPstnUniqueNumberLimit)
			        return true;

			   // test time usage    
		       if (skype_profile.dailyPstnLimitMinutes>0)
		       {
			           int pstnRemaining=skype_profile.dailyPstnLimitMinutes-chh.getPstnTodayTimeQualified();
			           if (pstnRemaining<=0 || pstnRemaining<skype_profile.refuseNewPstnCallsWhenRemainingMinutesUnder)
			               return true;
			   }
		}	
		return false;
	}

	public String getPossibleRunawaySkypeCallid()
	{
		return glbPossibleRunaway;
	}
   
   
}


