/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * 
 * A lot of globals to avoid memory allocation delays
 * 
 * Author(s):
 * Some Goertzel Portions taken  from
http://en.wikipedia.org/wiki/Goertzel_algorithm
 * Greg Dorfuss
 */


package local.ua;

import java.io.File;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;

import local.media.PipedInputStreamSizable;
import java.io.IOException;
import javax.sound.sampled.*;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

public class SSInBandDtmfDetector extends Thread
{
	private Logger log=null;
    private static enum States {SILENCE,ACTIVE,FALSEHIT};
	
    private float SAMPLING_RATE = 16000.0F; 
    private int BYTESPERDATASAMPLE=2;
    private int GOERTZEL_N;
    private int NUM_FREQS;
    private int STREAM_BYTES; 
    private int HIT_THRESHHOLD;
    private int NOHIT_THRESHHOLD;
    private int HIT_THRESHHOLDms;
    private int NOHIT_THRESHHOLDms;
    
    private float curValues[];
    private int activeDetect;
    private long activeHitCnt;
    private long silenceHitCnt;
    private States detectState;
    private float freqCoeffValueArray[];
    private float Q0;
    private float Q1[];
    private float Q2[];
    private boolean running=true;
    private PipedInputStream dataIs=new PipedInputStreamSizable(65536);
    private SSInBandDtmfInterface dtmfInt=null;
    private float curSample;
    private int activeDTMF;

    private static int dtmf_row_col_map[][] = {
		    {1, 2, 3, 12},
	        {4, 5, 6, 13},
	        {7, 8, 9, 14},
	        {10, 0, 11, 15}};
    
    private int detectRow;
    private int detectCol;
    private int peak_count;
    private int max_index;
    private double  maxval;
    private double tFactor;
    private int  iloop;





    public SSInBandDtmfDetector(SSInBandDtmfInterface argUa,int argSampleRate,int argBytesPerSample,int argHitThresholdms,int argSilenceThresholdms) 
    {
 
    	this.setName(this.getClass().getName()+".T"+this.getName().replaceAll("Thread-", ""));
		log = Logger.getLogger(this.getName());
    	this.dtmfInt=argUa;
    	SAMPLING_RATE=argSampleRate;
    	BYTESPERDATASAMPLE=argBytesPerSample;
    	
    	HIT_THRESHHOLDms=argHitThresholdms;
    	NOHIT_THRESHHOLDms=argSilenceThresholdms;
    	
    	init();
    	start();
    }

    
    
    private void init() 
    {
    	 double msPerRound=5.5;
    	 int gMagic=(int)(msPerRound*BYTESPERDATASAMPLE*8);
		 GOERTZEL_N=Math.round(SAMPLING_RATE/gMagic);
		 
		 STREAM_BYTES=GOERTZEL_N*BYTESPERDATASAMPLE;

		 // set minimum for hit and minimum between hits

		 HIT_THRESHHOLD=(int) (HIT_THRESHHOLDms/msPerRound);
		 NOHIT_THRESHHOLD=(int) (NOHIT_THRESHHOLDms/msPerRound);
		 if (NOHIT_THRESHHOLD<1)
			 NOHIT_THRESHHOLD=1;

		 float frequencyArray[] = {697.0F, 770.0F, 852.0F, 941.0F, 1209.0F, 1336.0F, 1477.0F, 1633.0F};
		 
		 NUM_FREQS=frequencyArray.length;

		 freqCoeffValueArray = new float[NUM_FREQS];
		 Q1= new float[NUM_FREQS];
		 Q2 = new float[NUM_FREQS];
		 curValues=new float[NUM_FREQS];
		 
		 for(int n = 0; n < NUM_FREQS; n++)
		 {
			 freqCoeffValueArray[n] =(float) (2.0 * java.lang.Math.cos(2.0 * java.lang.Math.PI * frequencyArray[n] /SAMPLING_RATE));
		 }
        
 		silenceHitCnt=1;
		detectState=States.SILENCE;
		activeDetect=-1;

    }

    public void stopDetecter()
    {
    	running=false;
    }
    
    public PipedInputStream getPipedInputStream()
    {
    	return this.dataIs;
    }
    
    
    public void run()
    {
    	
    	byte data[] = new byte[STREAM_BYTES];

        while (!Thread.interrupted() && running)
		{
		    try
		    {
		       if (dataIs.available()>=STREAM_BYTES)
		       {	   
				        dataIs.read(data,0,STREAM_BYTES);

				    
				        // process the samples
				    	
				    	if (BYTESPERDATASAMPLE==2)
				    	{	
					        for (int si = 0; si < STREAM_BYTES; si+=2) 
					        {
					        	curSample=data[si + 1] << 8 | data[si] & 0xff; // 2 bytes per sample
					            
					        	//log.warn("samp="+curSample);
					        	
					            for (int qi = 0; qi < NUM_FREQS; qi++) 
					            {
					                Q0 = freqCoeffValueArray[qi] * Q1[qi] - Q2[qi] + curSample;
					                Q2[qi] = Q1[qi];
					                Q1[qi] = Q0;
					            }
					        }
				    	}
				    	else
				    	{	
					        for (int si = 0; si < STREAM_BYTES; si++) 
					        {
					        	curSample=data[si];
					            
					            for (int qi = 0; qi < NUM_FREQS; qi++) 
					            {
					                Q0 = freqCoeffValueArray[qi] * Q1[qi] - Q2[qi] + curSample;
					                Q2[qi] = Q1[qi];
					                Q1[qi] = Q0;
					            }
					        }
				    	}
					
				        for (int si = 0; si < NUM_FREQS; si++) 
				        {
				        	curValues[si]=((Q1[si] * Q1[si]) + (Q2[si] * Q2[si]) - (freqCoeffValueArray[si] * Q1[si] * Q2[si]));
				            
				            // wipe it as we go
				            Q1[si]=0;
				            Q2[si]=0;
				            
				        }
				 
				        activeDTMF = doDTMFDetection();
				        
				        //if (activeDTMF>=0)
				        //	log.warn("ab="+activeDTMF);
				        
				        if (activeDTMF<0)
				        {	
				        	if (detectState!=States.SILENCE)
				        	{	
				        		if (activeDetect>=0 && activeHitCnt>=HIT_THRESHHOLD && silenceHitCnt>=NOHIT_THRESHHOLD)
				        		{	
					        		if (dtmfInt!=null)
					        			dtmfInt.gotDtmfDigit(activeDetect);
						        	else	
						        		log.warn(" got:"+activeDetect);
				
					        		activeDetect=-1;
				        		}
				
				        		silenceHitCnt=1;
				        		detectState=States.SILENCE;
				        	}
				        	else
				        	{
				        		silenceHitCnt++;
				        	}
				        }
				        else 
				        {
				        	if (detectState!=States.ACTIVE)
				        	{	
				        		if (silenceHitCnt>=NOHIT_THRESHHOLD)
				        		{	
					        		activeHitCnt=1;
					        		detectState=States.ACTIVE;
					        		activeDetect=activeDTMF;
				        		}
				        		else
				        		{	
				            		detectState=States.FALSEHIT;
				            		activeDetect=-1;
				        		}
				        	}
				        	else if (activeDTMF!=activeDetect)
				        	{
				        		detectState=States.FALSEHIT;
				        		activeDetect=-1;
				        	}
				        	else
				        	{
				        		activeHitCnt++;
				        	}
				        }	
		       }
		       else
			    	try {sleep(10);} catch(Exception se){}
		    }
		    catch(IOException ioe)
		    {
		    	try {sleep(10);} catch(Exception se){}
		    }
		}
       
    }

    
    
    private int doDTMFDetection()
    {
      
      /* Find the largest in the row group. */
      detectRow = 0;
      maxval = 0.0;
      for ( iloop=0; iloop<4; iloop++ )
      {
        if ( curValues[iloop] > maxval )
        {
          maxval = curValues[iloop];
          detectRow = iloop;
        }
      }
      
      /* Check row for minimum energy */
      if ( curValues[detectRow] < 4.0e5 )   /* 2.0e5 ... 1.0e8 no change */
      {
        /* energy not high enough */
      	return -1;
      }
     
      /* Find the largest in the column group. */
      detectCol = 4;
      maxval = 0.0;
      for ( iloop=4; iloop<8; iloop++ )
      {
        if ( curValues[iloop] > maxval )
        {
          maxval = curValues[iloop];
          detectCol = iloop;
        }
      }
     
      /* Check col for minimum energy */
      if ( curValues[detectCol] < 4.0e5 )
      {
        /* energy not high enough */
    	return -1;
      }

     
        /* Twist check
         * CEPT => twist < 6dB
         * AT&T => forward twist < 4dB and reverse twist < 8dB
         *  -ndB < 10 log10( v1 / v2 ), where v1 < v2
         *  -4dB < 10 log10( v1 / v2 )
         *  -0.4  < log10( v1 / v2 )
         *  0.398 < v1 / v2
         *  0.398 * v2 < v1
         */
        if ( curValues[detectCol] > curValues[detectRow] )
        {
          /* Normal twist */
          max_index = detectCol;
          if ( curValues[detectRow] < (curValues[detectCol] * 0.398) )    /* twist > 4dB, error */
             return -1;
        }
        else /* if ( r[row] > r[col] ) */
        {
          /* Reverse twist */
          max_index = detectRow;
          if ( curValues[detectCol] < (curValues[detectRow] * 0.158) )    /* twist > 8db, error */
        	  return -1;
        }
     
        /* Signal to noise test
         * AT&T states that the noise must be 16dB down from the signal.
         * Here we count the number of signals above the threshold and
         * there ought to be only two.
         */
        if ( curValues[max_index] > 1.0e9 )
        	tFactor = curValues[max_index] * 0.158;
        else
        	tFactor = curValues[max_index] * 0.010;
     
        peak_count = 0;
        for ( iloop=0; iloop<8; iloop++ )
        {
          if ( curValues[iloop] > tFactor )
            peak_count++;
        }
        if ( peak_count > 2 )
        	return -1;
     
        return dtmf_row_col_map[detectRow][detectCol-4];
       
    }
    
    
    
    /** for testing. */
	public static void main(String[] args) throws Exception 
	{         
	       	   //String clipFile="testingclips/dtmf-1.wav";
    	       //String clipFile="testingclips/DTMF_tones_8k_16bit.wav";
	       	   //String clipFile="testingclips/DTMF_tones_16k_16bit.wav";
				String clipFile="testingclips/skypetest.wav";
	       	   
	       	   int ReadSize=5000;
	       	   
	       	   AudioInputStream audStream=null;
			   try
			   {
				    AudioFormat af=AudioSystem.getAudioFileFormat(new File(clipFile)).getFormat();
		       	    SSInBandDtmfDetector detector=new  SSInBandDtmfDetector(null,(int)af.getSampleRate(),af.getSampleSizeInBits()/8,20,6);
		       	    PipedOutputStream dtmfDectorStream=new PipedOutputStream(detector.getPipedInputStream());
		       	    
		       	    PropertyConfigurator.configure("log.properties");
		       	     
					System.out.println("clipFormat="+af.toString());
					
					byte data[] = new byte[ReadSize];
					
					audStream=AudioSystem.getAudioInputStream(new File(clipFile));

					 System.out.println("File:"+clipFile+"Opened");
			     
				     while (audStream.available()>=ReadSize)
				     {	   
		
				    	    audStream.read(data,0,ReadSize);
				    	    dtmfDectorStream.write(data,0,ReadSize);
				    	    try {sleep(1);} catch(Exception se){}
				     }
				     dtmfDectorStream.flush();
			   }
			   catch (Exception e)
			   {
				   e.printStackTrace();
				   System.err.println("Error - Msg:"+e.getLocalizedMessage());
			   }
	       	   
	       	   
			   System.exit(0);
	   }    
    
}




