/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Based on mjsip 1.6 software and skype4java
 * 
 * Author(s):
 * Greg Dorfuss
 */

package local.ua;

import java.io.IOException;
import java.io.PipedOutputStream;
import java.net.DatagramSocket;
import java.net.DatagramPacket;
import java.net.SocketException;
import java.net.SocketTimeoutException;


import org.apache.log4j.Logger;

import local.ua.sscodecs.SSCodec;

/*
 * receive audio from remote sip and send to local skype
 */

public class SkypeRtpReceiver extends Thread implements SSInBandDtmfInterface
{

	private enum enumRX {SUCCESS,IOERROR,TIMEOUT,FAIL};
	
	private boolean doSkypeSend=false;
    private boolean stopnow=false;
    private Logger log = null;
    
    /** Size of the read buffer */
    private static final int RTP_BUFFER_SIZE=512;
    
    private SSCodec ssCodec=null;

    /** Maximum blocking time, spent waiting for reading new bytes [milliseconds] */
    private static final int SO_TIMEOUT=100;

    /** The RtpSocket */
    private DatagramSocket rtp_socket=null;

    private SkypeUserAgent ua=null;
    
    private boolean doRtpRedirect=false;
    
    SkypeAudioSenderServer skypeSender=null;
    
    private PipedOutputStream skypeOutStream = null;
    
    private int codec_payload_type;
    
    private int redirectCntReceived=0;
    
    private boolean startMediaCalled=false;
    
    private boolean inbandDtmfActive=false;
    
	public SkypeRtpReceiver(SkypeUserAgent argUa) throws Exception
	{
		this.setName(this.getClass().getName()+".T"+this.getName().replaceAll("Thread-", ""));
		log = Logger.getLogger(this.getName());
		this.ua=argUa;
	}

	/** Inits the RtpStreamReceiver */
	public void initRtp(SSCodec argCodec,DatagramSocket socket) throws Exception
	{
	  initCodec(argCodec);
	  rtp_socket=socket;
	}
	
	public void initCodec(SSCodec argCodec)
	{
		  this.ssCodec=argCodec;
		  codec_payload_type=this.ssCodec.getPayloadType();
		  
	}
	
	public synchronized void startMedia()
	{
		startMediaCalled=true;
		this.notify();
	}

	public void startSkypeMedia() throws Exception
	{
		doSkypeSend=true;
		if (!this.ua.skype_profile.inbandFullTimeDtmfDetection)
			inbandDtmfActive=false;
	}

	
	public void stopMedia()
	{
		stopnow=true;
		doSkypeSend=false;
		skypeSender.stopMedia();

	}
	
	public void gotDtmfDigit(int digit)
	{
		this.ua.onDtmfReceived(this.ua.call, digit);
	}
	
	void halt()
	{
		interrupt();
	}
	
	public void interrupt()
	{
		stopnow=true;
		super.interrupt();
	}
	
	
	  public void run() 
	  {
		  try
		  {
			  skypeSender=new SkypeAudioSenderServer(ua);
			  skypeSender.start();

			  skypeOutStream = new PipedOutputStream(skypeSender.getPipedInPutStream());
			  
			  while (true)
			  {	 
				  inbandDtmfActive=this.ua.skype_profile.enableSIPInbandDtmfDetector;
				  synchronized(this) 
	              {
					  while (!startMediaCalled)
						  this.wait();
	              }
				  startMediaCalled=false;
			      rtpReceiver();
			  }
		  }
		  catch (InterruptedException ie)
		  {
		  	log.debug("interrupted");
		  	skypeSender.halt();
		  }
		  catch (Exception e)
		  {
			  log.error("error",e);
		  }
      }
	
	  
	public void resetRTPAllowRedirect()
	{
		if (this.ua.skype_profile.enableSendRTPtoReceivedAddress)
		{	
			if (!doRtpRedirect)
			{	
				log.info("RtpSender Address unLocked.");
				doRtpRedirect=true;
			}
			redirectCntReceived=0;
		}
			
	} 
	
	private void rtpReceiver() throws IOException
	{
				if (ua.skype_profile.audioPriorityIncrease>0)
				{
					try
					{
					this.setPriority(Thread.NORM_PRIORITY+ua.skype_profile.audioPriorityIncrease);
					}
					catch (Exception e)
					{
						log.warn("setPriority Error. ",e);
					}
				}
  	
				if (this.ua.skype_profile.enableSendRTPtoReceivedAddress)
					doRtpRedirect=true;
			
				byte[] buffer=new byte[RTP_BUFFER_SIZE];
			    RtpPacket rtp_packet=new RtpPacket(buffer,0);
		      	DatagramPacket rtpDatagram=new DatagramPacket(buffer,0,buffer.length);
	
		      	stopnow=false;
		      	
				byte[] outpcm=new byte[RTP_BUFFER_SIZE*4];
				int outpcmlen=0;
			    long rtpTimeoutCnt=0;
			    long ioErrorCnt=0;
			    long ukErrorCnt=0;
			    int timeoutCnt=0;
			    int payLoadType=0;
			    int maxTimeoutCnt=ua.skype_profile.noRtpReceivedAutoHangupSeconds*1000/(SO_TIMEOUT+10); // timeout till timeout auto hangup
			    boolean dtmfStartEvent=false;
			    enumRX rxStatus;
			    long rtpPacketCnt=0;
			    long unKnownPayloadCnt=0;
			    
			    redirectCntReceived=0;
			    
			    
	            SSInBandDtmfDetector dtmfDetecter=null;
	            PipedOutputStream sipDtmfDectorStream=null;
	            if (ua.skype_profile.enableSIPInbandDtmfDetector)
	            {	
	              dtmfDetecter=new SSInBandDtmfDetector(this,16000,2,ua.skype_profile.SipDtmfDetectorHitThreshold,ua.skype_profile.SipDtmfDetectorSilenceThreshold);
	              sipDtmfDectorStream=new PipedOutputStream(dtmfDetecter.getPipedInputStream());
	            }

	            
			    /*
			    boolean writeUlawFile=false;
			    boolean writePcmFile=false;
	            UlawWriter fileUlawOut=null;
	            WavWriter filePcmOut=null;
	            
	            if (writePcmFile)
	            {	
	            	filePcmOut=new WavWriter("testingclips/test_pcm_in.wav");
	            }
	            if (writeUlawFile)
	            {	
	            	fileUlawOut=new UlawWriter("testingclips/test_ulaw_in.au");
	            }
	            */

	            
			    log.debug("+++ sipAudioReceiver - sip Port:"+rtp_socket.getLocalPort());
				
		    	 //Skype API is 16 Khz and 16 Bits per sample PCM
		    	  //The 16-bit samples are stored as 2's-complement signed integers, ranging from -32768 to 32767.

			    this.skypeSender.stopSkypeClips();
			    this.skypeSender.setSipConnected();
			    
			    clearRTPPackets();
			    
				while (!stopnow)
				{
						      try
						      {
						    	  rtp_socket.receive(rtpDatagram);
							      rtp_packet.packet_len=rtpDatagram.getLength();
							      rxStatus=enumRX.SUCCESS;
						      }
						      catch (SocketTimeoutException e)
						      {
						    	  rxStatus=enumRX.TIMEOUT;
						      }
						      catch (IOException e)
						      {
						    	  rxStatus=enumRX.IOERROR;
						      }
						      catch (Exception e)
						      {
						    	  log.error("receive RTP error",e);
						    	  rxStatus=enumRX.FAIL;
						      }
						 	
							if (rxStatus==enumRX.SUCCESS)
							{			
									timeoutCnt=0;
								
									payLoadType=rtp_packet.getPayloadType();
									int hdrSize=rtp_packet.getHeaderLength();
									
									if (payLoadType==codec_payload_type)
									{	
											int availBytes=rtp_packet.getPayloadLength();
											rtpPacketCnt++;

											
											if (doRtpRedirect)
											{	
												redirectCntReceived++;
												this.ua.skypeRtpSender.redirectRTP(rtpDatagram.getAddress(),rtpDatagram.getPort());
												if (redirectCntReceived>=this.ua.skype_profile.lockRtpSendAddressAfterPackets)
												{
													log.info("RtpSender Address locked.");
													doRtpRedirect=false;
												}
											}
											
											
											outpcmlen=ssCodec.CodecToPcm(buffer, hdrSize, availBytes,outpcm);
											
											if (inbandDtmfActive)
								        		   sipDtmfDectorStream.write(outpcm,0,outpcmlen); // send wav data to DTMF detector
											
											if (doSkypeSend && !this.ua.skypeHoldingLocal && !this.ua.skypeHoldingRemote)
											{	
												skypeOutStream.write(outpcm,0, outpcmlen);
												this.skypeSender.sipSentData();
											}
											
											/*
											if (writePcmFile)
												filePcmOut.write(outpcm,outpcmpos);

											if (writeUlawFile)
												fileUlawOut.write(buffer,hdrSize,availBytes);
											*/
									}
									else if (payLoadType==this.ua.dtmf2833PayloadType_session)
									{
										//log.error("#### got dtmf2833 "+buffer[hdrSize]+":"+buffer[hdrSize+1]+":"+buffer[hdrSize+2]+":"+buffer[hdrSize+3]);
										//log.info("### "+rtp_packet.getDuration()+" "+rtp_packet.getSequenceNumber());
									    int eventFlag = (byte) (buffer[hdrSize+1] & 0x80);
									    
									    if (eventFlag==0)
									    	dtmfStartEvent=true;
									    else if (dtmfStartEvent)
									    {	
									    	dtmfStartEvent=false;
									    	int eventId = buffer[hdrSize];
									    	this.gotDtmfDigit(eventId);
									    }	
										
									}
									else
									{	
										unKnownPayloadCnt++;
									}	
								
							}
							else if (rxStatus==enumRX.FAIL)
							{
								ukErrorCnt++;
								// no need to wait here rtp_socket.receive waits ,try	{sleep(10);}	catch (Exception ew3){}
							}
							else if (rxStatus==enumRX.IOERROR)
							{
								ioErrorCnt++;
								// no need to wait here rtp_socket.receive waits , try	{sleep(10);}	catch (Exception ew3){}
							}
							else 
							{
								if (!this.ua.onRemoteSipHold && !stopnow && this.ua.call_state==UserAgent.UA_ONCALL)
								{	
									timeoutCnt++;
									rtpTimeoutCnt++;
								
									if (timeoutCnt>=maxTimeoutCnt)
									{
										log.info("rtp Timeout: hanging up");
										this.ua.hangup();
										stopnow=true;
									}
								}
							}
				}
		
				/*
				if (writePcmFile)
					filePcmOut.close();

				if (writeUlawFile)
					fileUlawOut.close();
				*/
				
				if (ua.skype_profile.enableSIPInbandDtmfDetector)
	        		   sipDtmfDectorStream.close();
				
				rtp_socket=null;
				
				if (ua.skype_profile.audioPriorityIncrease>0)
				{
					try
					{
						this.setPriority(Thread.NORM_PRIORITY);
					}
					catch (Exception e)
					{
						log.warn("resetPriority Error. ",e);
					}
				}

				
				skypeSender.stopMedia();
				
				String msg="RTPReceiver stats - packets:"+rtpPacketCnt;
				if (rtpTimeoutCnt>0)
					msg+=" timeOuts:"+rtpTimeoutCnt;
				if (ioErrorCnt>0)
					msg+=" IOErrors:"+ioErrorCnt;
				if (unKnownPayloadCnt>0)
				   msg+=" InvalidPayload Packets:"+unKnownPayloadCnt;
				if (ukErrorCnt>0)
				   msg+=" Unknown Errors:"+ukErrorCnt;
				
				log.info(msg);

	}

	
	private void clearRTPPackets() throws SocketException
	{
		// clear out rtp packets as fast as possible - force an error
      	DatagramPacket dgp=new DatagramPacket(new byte[1],1);
      	
      	rtp_socket.setSoTimeout(5);
      	int ioErrCnt=0;
      	while (true)
	    {	
			try
			{
				rtp_socket.receive(dgp);
			}
			catch (SocketTimeoutException ce)
			{
				break;
			}
			catch (IOException ce)
			{
				ioErrCnt++;
				if (ioErrCnt>100)
				{	
					log.warn("??? IO Errors during clearRTPPackets.");
					break;
				}
			}
		}
      	rtp_socket.setSoTimeout(SO_TIMEOUT);
	}

	
	
}
