/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Based on mjsip 1.6 software and skype4java
 * 
 * Author(s):
 * Greg Dorfuss
 * 
 * 2009/02/07 - changes to rfc2833 code for improved compatibility
 */

package local.ua;

import java.io.IOException;
import java.net.DatagramSocket;
import java.net.DatagramPacket;
import java.net.InetAddress;

import local.ua.sscodecs.FIRFilter;
import local.ua.sscodecs.FilterInterface;
import local.ua.sscodecs.FilterLowPassRc;
import local.ua.sscodecs.SSCodec;
import local.ua.sscodecs.SSCodecFactory;

import java.util.Vector;

import javax.sound.sampled.AudioFormat;
import javax.sound.sampled.AudioSystem;
import javax.sound.sampled.AudioInputStream;
import java.io.File;

import org.apache.log4j.Logger;
import java.util.concurrent.locks.LockSupport;
import local.ua.sscodecs.SSCodecInfo;

/*
 * Receive Skype Audio and sends it to remote destination via sip rtp
 */

public class SkypeRtpSender extends Thread
{
	private static final long nanosPerMs=1000000;
    private boolean skypeConnected=false;
    private boolean stopnow=false;
    
    private Logger log = null;
    
    private Vector<String> sipClipQueue=new Vector<String>();
    private AudioInputStream sipClipStream=null;
    private Vector<String> dtmfSipQueue=new Vector<String>();
    
    private SkypeUserAgent ua=null;
    
    private DatagramSocket rtp_socket=null;
    
    private int frame_size=-1;
    private long frame_time=-1;
    private long frame_time_nanos=-1;
    private SSCodec ssCodec=null;
    private SSCodec ssClipCodec=null;
    private int codec_payload_type=-1;
    private byte[] codecBuf=new byte[512];
    private RtpPacket rtp_packet=new RtpPacket(codecBuf,0);
    private byte[] blankbuf=new byte[512];
    private RtpPacket blankpacket=new RtpPacket(blankbuf,0);
    private byte[] dtmfbuf=new byte[512];
    private RtpPacket dtmfpacket=new RtpPacket(dtmfbuf,0);

    private int startPayloadPos=-1;
    private DatagramPacket sendDatagram=null;
    private int packetInteval=-1;
    private int packetCounter=0;

    private Object redirectLock=new Object();
    
    private static final int pcmBytesPerMs=32;

    private int dest_port=-1;
    private InetAddress dest_addr=null;
    
    private SkypeAudioReceiverServer skypeReceiver=null;

    private static final int PIPEBUFFER_SIZE=65536;
    private static final int maxBehindBytes=PIPEBUFFER_SIZE-5000;

    private PipedInputStreamPCM skypeStream=null;
    
    private boolean skypeSentData=false;
    private boolean startMediaCalled=false;
    
    //used by jitter buffer
	// total 130ms range
	private long jittTargetNanos=100*nanosPerMs;
	private long jittMaxBehindNanos=180*nanosPerMs;

  	private static long abs_maxdelay_nanos=0;
	private static long abs_mindelay_nanos=0;
	private static int pcmBytesPerRtpPacket=0;

	private enum FilterType {NONE,RC,FIR};
	private FilterInterface filter=null;
	private boolean useFilter=false;
	
	public SkypeRtpSender(SkypeUserAgent argUa) throws Exception
	{
		this.setName(this.getClass().getName()+".T"+this.getName().replaceAll("Thread-", ""));
		log = Logger.getLogger(this.getName());
		this.ua=argUa;
		
       if (this.ua.skype_profile.jitterLevel==1)
       {	   
			  jittTargetNanos=50*nanosPerMs;
			  jittMaxBehindNanos=100*nanosPerMs;
       }
       else if (this.ua.skype_profile.jitterLevel==2)
       {	   
			  jittTargetNanos=75*nanosPerMs;
			  jittMaxBehindNanos=150*nanosPerMs;
       }
       else if (this.ua.skype_profile.jitterLevel==3)
       {	   
			  jittTargetNanos=90*nanosPerMs;
			  jittMaxBehindNanos=180*nanosPerMs;
       }
       else if (this.ua.skype_profile.jitterLevel==4)
       {	   
			  jittTargetNanos=110*nanosPerMs;
			  jittMaxBehindNanos=220*nanosPerMs;
       }
       else
       {	   
			  jittTargetNanos=140*nanosPerMs;
			  jittMaxBehindNanos=280*nanosPerMs;
       }
       
	}
	
	public void queueSipDtmfDigits(String argDigits)
	{
		for (int d=0;d<argDigits.length();d++)
			dtmfSipQueue.add(argDigits.substring(d,d+1));
	}
	
	public void queueSipClip(String argClipFile)
	{
		log.debug("Play Clip:"+argClipFile);
		sipClipQueue.add(argClipFile);
	}
	public boolean areClipsComplete()
	{
		if (sipClipQueue.size()==0 && sipClipStream==null)
			return true;
		else
			return false;
	}
	
	/*
	 * stops any clips and clears the queue
	 */
	private void stopSipClips()
	{
		sipClipQueue.removeAllElements();
		if (sipClipStream!=null)
		{
			try
			{
				sipClipStream.close();
			}
			catch(Exception e)
			{
			}
			sipClipStream=null;
		}
		ssClipCodec=null;
	}
	
	/*
	 * prepares for sip sending
	 */
	public void initRtp(SSCodec argCodec,DatagramSocket src_socket, String dest_addr, int arg_dest_port) throws Exception
	   {
		  initCodec(argCodec);
		  this.dest_port=arg_dest_port;
	      this.dest_addr=InetAddress.getByName(dest_addr);
	      rtp_socket=src_socket; 
	   }          

	public void initCodec(SSCodec argCodec)
	{
		  ssCodec=argCodec;
	      frame_size=ssCodec.getFrameSize();
		  int frame_rate=ssCodec.getSampleRate()/frame_size;
	      long byte_rate=frame_rate*this.frame_size;
          frame_time=(frame_size*1000)/byte_rate; // milliseconds
          frame_time_nanos=frame_time*nanosPerMs; // nanoseconds

          pcmBytesPerRtpPacket=ssCodec.getPcmFrameSize();
          codec_payload_type=ssCodec.getPayloadType();
	      
          rtp_packet.setPayloadType(codec_payload_type);      
          startPayloadPos=rtp_packet.getHeaderLength();

          blankpacket.setPayloadType(codec_payload_type);      
	      blankpacket.setPayloadLength(ssCodec.getCodecBlankPacket(blankbuf,startPayloadPos));
	      blankpacket.setSscr(rtp_packet.getSscr());
	      
	      dtmfpacket.setPayloadType(this.ua.dtmf2833PayloadType_session);
          dtmfpacket.setSscr(rtp_packet.getSscr());
	      //dtmfpacket.setPayloadLength(frame_size);
          dtmfpacket.setPayloadLength(4);
	      
	      packetInteval=this.frame_size;
	      
	  	  abs_maxdelay_nanos=(frame_time+40)*nanosPerMs;
		  
			if (FilterType.valueOf(this.ua.skype_profile.filterParams[0].toUpperCase())==FilterType.RC)
				filter=new FilterLowPassRc(Float.parseFloat(this.ua.skype_profile.filterParams[1]),Float.parseFloat(this.ua.skype_profile.filterParams[2]),pcmBytesPerRtpPacket);
			else if (FilterType.valueOf(this.ua.skype_profile.filterParams[0].toUpperCase())==FilterType.FIR)
				filter=new FIRFilter(Integer.parseInt(this.ua.skype_profile.filterParams[1]),FIRFilter.WINDOW.valueOf(this.ua.skype_profile.filterParams[2].toUpperCase()),FIRFilter.FILTTYPE.valueOf(this.ua.skype_profile.filterParams[3].toUpperCase()),Float.parseFloat(this.ua.skype_profile.filterParams[4]),Float.parseFloat(this.ua.skype_profile.filterParams[5]),16000,pcmBytesPerRtpPacket);
			else
				filter=null;

			if (filter!=null)
				useFilter=true;
			else
				useFilter=false;

	}

	
	public synchronized void skypeSentData()
	{
		skypeSentData=true;
		this.notify();
	}
	
	
	public boolean redirectRTP(InetAddress arg_dest_addr, int arg_dest_port)
	{
		  if (!this.dest_addr.equals(arg_dest_addr) || arg_dest_port!=this.dest_port)
			return	redirectRTP(arg_dest_addr.getHostAddress(),arg_dest_port);
		  else
		    return false;
     }
				
	public boolean redirectRTP(String arg_dest_addr, int arg_dest_port)
	{
	  if (rtp_socket==null)
		  return false;
      try
      {  
    	 synchronized (redirectLock)
    	 {
    		 if (!this.dest_addr.getHostAddress().toString().equals(arg_dest_addr) || arg_dest_port!=this.dest_port)
    		 {	 
    			 log.info("updating RTP media destination to:"+arg_dest_addr+":"+arg_dest_port+" from:"+this.dest_addr.getHostAddress()+":"+this.dest_port);
    			 this.dest_port=arg_dest_port;
    			 this.dest_addr=InetAddress.getByName(arg_dest_addr);
    		 }
	    	 clearSkypeAudio();
    	 }
   	     return true;
      }
      catch (Exception e) {  log.error("redirectRTP:", e); }
      
	  return false;
	}

	
	private long rtpDelay_frame_time_passed=0;
	private long rtpDelay_frame_starttime=System.nanoTime();
	private boolean resetRtpDelay=true;
	private void doRtpDelay()
	{
		// track time that should have passed and adjust delay accordingly
		// we only use this for clips,dtmf and blank packets
		long curNanoTime=System.nanoTime();

		if (resetRtpDelay)
		{
			//log.info("################# reset ##############");
			resetRtpDelay=false;
			rtpDelay_frame_time_passed=0;
			rtpDelay_frame_starttime=curNanoTime;
		}
		
		long slpTime=this.frame_time_nanos-(curNanoTime-rtpDelay_frame_starttime-rtpDelay_frame_time_passed);
		if (slpTime>this.frame_time_nanos)
			LockSupport.parkNanos(this.frame_time_nanos);
		else if (slpTime>0)
			LockSupport.parkNanos(slpTime);
		else if (slpTime<-this.frame_time_nanos) // gone wacky
		{
			if (packetCounter>5)
				resetRtpDelay=true;
		}
		
		//log.info("sl:"+slpTime);
		
		rtpDelay_frame_time_passed+=this.frame_time_nanos;
	}
	
	/*
	 * starts the skype bridge receiver
	 */
	public void startSkypeMedia() throws Exception
	{
		stopSipClips();
		skypeConnected=true;
		resetRtpDelay=true;
	}
	
	/*
	 * starts the sipSender
	 */
	public synchronized void startMedia()
	{
		startMediaCalled=true;
		this.notify();
	}
	
	/*
	 * stops all skype to sip audio
	 */
	public void stopMedia()
	{
		skypeReceiver.stopMedia();
		stopnow=true;
	    skypeConnected=false;
	}
	
	void halt()
	{
		interrupt();
	}
	
	public void interrupt()
	{
		stopnow=true;
		super.interrupt();
	}
	
	
	  public void run()
	  {
		  try
		  {
			  skypeReceiver=new SkypeAudioReceiverServer(this.ua);
			  skypeReceiver.start();

			  skypeStream=new PipedInputStreamPCM(skypeReceiver.getPipedOutputStream(),PIPEBUFFER_SIZE);

			  while (true)
			  {	 
				  synchronized(this) 
	              {
					  while (!startMediaCalled)
						  this.wait();
	              }
				 startMediaCalled=false;
			     rtpSender();
			  }
		  }
		  catch (InterruptedException ie)
		  {
		  	log.debug("interrupted");
		  	skypeReceiver.halt();
		  }
		  catch (Exception e)
		  {
			  log.error("error",e);
		  }
      }
	
	/*
	 * main processing - sends clips and skype audio to sip dest via rtp
	 */
	private void rtpSender()
	{
				sendDatagram=new DatagramPacket(new byte[1],0,0,null,0); // fake setup
        
				if (ua.skype_profile.audioPriorityIncrease>0)
				{
					try
					{
					this.setPriority(Thread.NORM_PRIORITY+ua.skype_profile.audioPriorityIncrease);
					}
					catch (Exception e)
					{
						log.warn("setPriority Error. ",e);
					}
				}
				
				stopnow=false;

		        long time=0;
		    	byte[] inpcm=new byte[2048];

		        /*
		        boolean writePcmFile=false;
			    boolean writeUlawFile=false;

	            UlawWriter fileUlawOut=null;
	            WavWriter filePcmOut=null;
	            
	            if (writePcmFile)
	            	filePcmOut=new WavWriter("testingclips/test_pcm_out.wav");

	            if (writeUlawFile)
	            	fileUlawOut=new UlawWriter("testingclips/test_ulaw_out.au");
	            */
	            

				//	Skype API is 16 Khz and 16 Bits per sample PCM
		    	//The 16-bit samples are stored as 2's-complement signed integers, ranging from -32768 to 32767.
				
	            log.debug("+++ sipAudioSender Connected - skypeframesize:"+pcmBytesPerRtpPacket+" framesize:"+this.frame_size+" frame_time:"+frame_time);
				
	            log.info("RTP media target: "+this.dest_addr.getHostAddress()+":"+this.dest_port);
	            
				int availBytes=0;
				
				long choppedCnt=0;
				long underRuns=0;
				int bufferCeiling=0;
				long pcmPacketCounter=0;
		    	resetRtpDelay=true;

		    	
		    	// jitter buf vars
		    	long pcmBytesReceived=0;
				long pcmBytesSent=0;
				double jitDiffPcktCnt=0;
				long jitMkupNanos=0;
				long jitTimeAdj=0;
				long jitTotalTimeDiffNanos;
				long jitRtpDelay=0;
				int dataMissed=0;

				try	{skypeStream.clearBuffer();}catch(IOException e){}
				
				skypeReceiver.setSipSending(true,pcmBytesPerRtpPacket);
				
				while (!stopnow)
				{
						if (dtmfSipQueue.size()>0)
						{	

							char digit=getNextDtmfDigit();
					        log.debug("Sending digit:"+digit);
					        
					        if(this.ua.user_profile.dtmfINFOType.length()>0)
					        {  
					        	// send using INFO msgs
					        	String msg=null;
					        	if (this.ua.user_profile.dtmfINFOType.contains("-relay"))
					        		msg="Signal="+digit+"\r\nDuration=100\r\n";
					        	else	
					        		msg=digit+"\r\n";
					        	
					        	this.ua.call.sendInfoMsg(msg,this.ua.user_profile.dtmfINFOType);
					        } 
					        
					        if (this.ua.user_profile.dtmf2833PayloadType>0)
					        {
					        	// send using rfc2833
						        if (digit=='*')
						        	dtmfbuf[startPayloadPos]=10;
						        else if (digit=='#')
						        	dtmfbuf[startPayloadPos]=11;
						        else if (digit>='A' && digit<='D')
						        	dtmfbuf[startPayloadPos]=(byte)(digit-53);
						        else 
						        	dtmfbuf[startPayloadPos]=(byte)(digit-48);
						        	
					        
						        // notice we are bumping times/seqn just like audio packets
						    	resetRtpDelay=true;

				    	     	// send start event (marked) 3 times
				    	     	dtmfbuf[startPayloadPos+1]=0; // start event flag and volume
						        dtmfpacket.setDuration(0);
				            	dtmfpacket.setSequenceNumber(packetCounter);
				            	dtmfpacket.setTimestamp(time); // only start event one should get set
						        dtmfpacket.setMarker(true);
				            	doRtpDelay();
					            for (int r=0;r<3;r++)
					            	send(dtmfpacket);

				            	blankpacket.setSequenceNumber(packetCounter);
				            	blankpacket.setTimestamp(time);
					            send(blankpacket);

					            packetCounter++;
					            time+=packetInteval;
						            	
						        dtmfpacket.setMarker(false);
					            for (int r=1;r<4;r++)
					            {	
							        dtmfpacket.setDuration(packetInteval*r);
							        dtmfpacket.setSequenceNumber(packetCounter);
					            	// wrong dtmfpacket.setTimestamp(time);
							        doRtpDelay();
					            	send(dtmfpacket);
					            	
					            	blankpacket.setSequenceNumber(packetCounter);
					            	blankpacket.setTimestamp(time);
						            send(blankpacket);

						            packetCounter++;
						            time+=packetInteval;
					            }

					            // send end event packet 3 times
						        dtmfbuf[startPayloadPos+1]=-128; // end event flag
						        dtmfpacket.setDuration(packetInteval*4);
				            	dtmfpacket.setSequenceNumber(packetCounter);
				            	// wrong dtmfpacket.setTimestamp(time);
						        doRtpDelay();
						        for (int r=0;r<3;r++)
							       	send(dtmfpacket);

				            	blankpacket.setSequenceNumber(packetCounter);
				            	blankpacket.setTimestamp(time);
					            send(blankpacket);
						        
						        packetCounter++;
					            time+=packetInteval;						    	     
						    	     
					        }   

					       //resetRtpDelay=true;

				            // send 250 ms of blank packets
				            for (int r=0;r<250/frame_time;r++)
				            {
				            	blankpacket.setSequenceNumber(packetCounter++);
				            	blankpacket.setTimestamp(time+=packetInteval);
				            	doRtpDelay();
					            send(blankpacket);
				            }
				            
				            clearSkypeAudio();
						            
						}
						else if (sipClipStream!=null)
						{
							    
								//playing a clip
							    int clipRead=-1;
							    try
							    {
							    	clipRead=sipClipStream.read(inpcm,0,pcmBytesPerRtpPacket);
							    }
							    catch(Exception e)
							    {log.error("error",e);}
							    
								if (clipRead>=pcmBytesPerRtpPacket)
								{	
									if (useFilter)
										filter.filter(inpcm);
									
									rtp_packet.packet_len=startPayloadPos+ssClipCodec.PcmToCodec(inpcm, pcmBytesPerRtpPacket,codecBuf,startPayloadPos);
									rtp_packet.setSequenceNumber(packetCounter++);
						            rtp_packet.setTimestamp(time+=packetInteval);
					            	doRtpDelay();
						            send(rtp_packet);
								}
								else if (clipRead>0)
								{
									//not enough for a full sip frame, it's probably the end of the file - send a blank packet
										blankpacket.setSequenceNumber(packetCounter++);
										blankpacket.setTimestamp(time+=packetInteval);
						            	doRtpDelay();
							            send(blankpacket);
								}
								else if (clipRead==-1)
								{
									log.debug("--- clip Closed");
									try
								    {
										sipClipStream.close();
								    }
								    catch(Exception e)
								    {}	
								    sipClipStream=null;

								    if (sipClipQueue.size()==0 && skypeConnected)
										clearSkypeAudio();
								}
						}
						else if (sipClipQueue.size()>0)
						{
							sipClipStream=getNextClipStream();
							resetRtpDelay=true;
							
						}
						else if (skypeConnected)
						{
							
								if (this.ua.skype_profile.jitterLevel==0) // only if not using jitter
								{	
									synchronized (this)
									{
										try
										{
										  if (!skypeSentData) // incase we get notified while outside this block
											  wait(100); // wait for a notify or timeout
										  
										  skypeSentData=false;
										}
										catch(InterruptedException e)
										{
											log.error("Interrupted",e);
										}
									}
								}	
								
								try
								{
								    availBytes=skypeStream.available();
								}
								catch(IOException e)
								{
									log.error("pipe ioerror",e);
								}
									
								//log.info(availBytes);
								
								
								
								if (availBytes>=pcmBytesPerRtpPacket)
								{	
									dataMissed=0;
									
									if (availBytes>bufferCeiling)
										bufferCeiling=availBytes;
									
									if (availBytes>=maxBehindBytes)
									{
										try
										{
										  skypeStream.chopBuffer((int)(maxBehindBytes*.9));
						            	  availBytes=skypeStream.available();
										}
										catch (IOException e){}
										choppedCnt++;
									}
									
	
									
									if (availBytes>bufferCeiling)
										bufferCeiling=availBytes;
									
									if (availBytes>=maxBehindBytes)
									{
										try
										{
										  skypeStream.chopBuffer((int)(maxBehindBytes*.9));
						            	  availBytes=skypeStream.available();
										}
										catch (IOException e){}
										choppedCnt++;
									}
									
									
									if (this.ua.skype_profile.jitterLevel>0)
									{
										
										  jitTotalTimeDiffNanos=(availBytes/pcmBytesPerMs*nanosPerMs);
	
										  if (jitTotalTimeDiffNanos>jittMaxBehindNanos)
											  jitRtpDelay=abs_mindelay_nanos; // we are way too slow need to catch up
										  else if (jitTotalTimeDiffNanos<=jittTargetNanos)
										  {
											  // we are too fast
											  jitMkupNanos=jittTargetNanos-jitTotalTimeDiffNanos;
											  jitDiffPcktCnt=(double)(availBytes)/pcmBytesPerRtpPacket+1;
											  jitTimeAdj=(long)(jitMkupNanos/jitDiffPcktCnt);
											  jitRtpDelay=frame_time_nanos-jitTimeAdj;
											  if (jitRtpDelay>abs_maxdelay_nanos)
												  jitRtpDelay=abs_maxdelay_nanos;
										  }
										  else
										  {
											  // we are too slow
											  jitMkupNanos=jitTotalTimeDiffNanos-jittTargetNanos;
											  jitDiffPcktCnt=(double)(availBytes)/pcmBytesPerRtpPacket-1;
											  jitTimeAdj=(long)(jitMkupNanos/jitDiffPcktCnt);
											  jitRtpDelay=frame_time_nanos-jitTimeAdj;
											  if (jitRtpDelay<abs_mindelay_nanos)
												  jitRtpDelay=abs_mindelay_nanos;
										  }
										  
										  //log.info("timediff="+jitTotalTimeDiffNanos+" dl="+jitRtpDelay);
	
										
									}
									
									
									if (jitRtpDelay>=0)
									{
											do {	
												    pcmBytesReceived+=availBytes;
													pcmPacketCounter++;
	
													//if (pcmPacketCounter%10==0)
														//log.info("r "+(int)((availBytes-skypeFrameSize)/pcmBytesPerMs));
													
													try
													{
														skypeStream.read(inpcm,0,pcmBytesPerRtpPacket);
													}
													catch (IOException e)
													{
														log.fatal(e);
														stopnow=true;
														break;
													}
													
													/*
													if (writePcmFile)
										            	filePcmOut.write(inpcm,skypeFrameSize);
										            */
													
													if (useFilter)
														filter.filter(inpcm);
													
													rtp_packet.packet_len=startPayloadPos+ssCodec.PcmToCodec(inpcm, pcmBytesPerRtpPacket,codecBuf,startPayloadPos);
													
													/*
													if (writeUlawFile)
														fileUlawOut.write(outulaw,startPayloadPos,outulawpos-startPayloadPos);
													*/
										            rtp_packet.setSequenceNumber(packetCounter++);
										            rtp_packet.setTimestamp(time+=packetInteval);
										            
										            if (jitRtpDelay>0)
										            {	
											  		  	// LockSupport.parkNanos(jitRtpDelay);
											  		  	try{sleep(jitRtpDelay/1000000);}catch(Exception e){}
										            }
										            
										            send(rtp_packet);
										            pcmBytesSent+=pcmBytesPerRtpPacket;
										            availBytes-=pcmBytesPerRtpPacket;
										            
											} while (availBytes>=pcmBytesPerRtpPacket && (this.ua.skype_profile.jitterLevel==0 || jitRtpDelay==0));
									}
									else
										try{sleep(20);}catch(Exception e){}
									
									
								}	
								else
								{
									//waiting for more skype audio
									if (this.ua.skypeHoldingLocal || this.ua.skypeHoldingRemote )
									{
										// we are skype holding and no data - send blanks to sip device
										blankpacket.setSequenceNumber(packetCounter++);
										blankpacket.setTimestamp(time+=packetInteval);
										doRtpDelay();
							            send(blankpacket);
									}
									else
									{   
										if (packetCounter>5)
										{	
											//we have received at least 5 packets - it's a real underrun
											if (this.ua.skype_profile.jitterLevel>0)
											{
												dataMissed++;
												if (dataMissed>2)
												{	
												   underRuns++;
												   dataMissed=0;
												}
											}
											else
												underRuns++;
											//log.info("ru");
										}
									}
									
									if (this.ua.skype_profile.jitterLevel>0)
										try{sleep(20);}catch(Exception e){}
								}
						}

						else
						{
							// idle
							// we are connected to rtp so send blank rtp packets
							blankpacket.setSequenceNumber(packetCounter++);
							blankpacket.setTimestamp(time+=packetInteval);
			            	doRtpDelay();
				            send(blankpacket);
						}
				}
				
				// it's time to shutdown
				stopSipClips();
				
				skypeReceiver.setSipSending(false,pcmBytesPerRtpPacket);
				skypeReceiver.stopMedia();
				try{skypeStream.clearBuffer();}catch (IOException e){log.error("Error",e);}
				
				/*
				if (writePcmFile)
	            	filePcmOut.close();

	            if (writeUlawFile)
					fileUlawOut.close();
				*/

				rtp_socket=null;
				sendDatagram=null;
				filter=null;
				
				if (ua.skype_profile.audioPriorityIncrease>0)
				{
					try
					{
					this.setPriority(Thread.NORM_PRIORITY);
					}
					catch (Exception e)
					{
						log.warn("resetPriority Error. ",e);
					}
				}
				//bufferCeiling-=pcmBytesPerRtpPacket;
				int maxLatency=(bufferCeiling/pcmBytesPerMs);
				if (maxLatency<0)
					maxLatency=0;

				//pcmBytesReceived-=(pcmBytesPerRtpPacket*pcmPacketCounter);
				int avgLatency=0;
				if (pcmPacketCounter>0)
				  avgLatency=(int)(pcmBytesReceived/pcmPacketCounter/pcmBytesPerMs);
				
				String msg="RTPSender stats - packets:"+packetCounter;
				if (choppedCnt>0)
					msg+=" chopCnt:"+choppedCnt;
				if (underRuns>0)
					msg+=" underRuns:"+underRuns;
				msg+=" maxLatency:"+maxLatency+"ms avgLatency:"+avgLatency+"ms";
				log.info(msg);
	}
	
	/*
	 * gets the next clip from the clip queue
	 */
	private AudioInputStream getNextClipStream()
	{
	   AudioInputStream retvar=null;
	   while (sipClipQueue.size()>0 && retvar==null)
	   {	   
		   String clipFile=sipClipQueue.get(0);
		   sipClipQueue.remove(0);
		   try
		   {
			 //printLog("clipFormat="+AudioSystem.getAudioFileFormat(new File(clipFile)).toString());
			 
			 AudioFormat format=new AudioFormat(AudioFormat.Encoding.PCM_SIGNED, 16000.0F, 16, 1, 2, 16000.0F, false);
			 AudioInputStream srcStream=AudioSystem.getAudioInputStream(new File(clipFile));
		     retvar=AudioSystem.getAudioInputStream(format, srcStream);
		     log.debug("+++ clip "+clipFile+" Opened");

			 if (ssClipCodec==null)
			 {
			   // make a codec instance to play clips with no gain
			   SSCodecInfo codecInfo=ssCodec.getCodecInfo();
			   codecInfo.inGain=1;
			   codecInfo.outGain=1;
			   ssClipCodec=SSCodecFactory.getCodec(codecInfo);
			 }
		   }
		   catch (Exception e)
		   {
			   log.error("File "+clipFile+" Error - Msg:",e);
			   retvar=null;
		   }
	   }
	   return retvar;
	}
	
	/*
	 * gets the next dtmf digit from queue
	 */
	
	private char getNextDtmfDigit()
	{
		 if (dtmfSipQueue.size()>0)
		 {	 
		   String digit=dtmfSipQueue.get(0);
		   dtmfSipQueue.remove(0);
		   return digit.charAt(0);
		 }
		 else
			 return 0;
	}
	
	private void clearSkypeAudio()
	{
		try
		{
	          if (skypeConnected && skypeStream.available()>0)
	          	skypeStream.skip(skypeStream.available()); // throw away the skype audio received during this time period
		}
		catch (IOException e)
		{
			log.error("Error",e);
		}
	}

	private void send(RtpPacket rPacket)
	{  
		sendDatagram.setData(rPacket.packet,0,rPacket.packet_len);
		sendDatagram.setPort(this.dest_port);
		sendDatagram.setAddress(this.dest_addr);
		try
		{
	     rtp_socket.send(sendDatagram);
		}
		catch (Exception e)
		{
			String msg=e.getMessage();
			if (msg!=null && msg.indexOf("Socket is closed")>=0)
				stopnow=true;
			else
				log.error("Rtp Send Error",e);
		}
	}
	


}
