/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Based on mjsip 1.6 software and skype4java
 * 
 * Author(s):
 * Greg Dorfuss
 */

package local.ua;

import org.zoolu.sip.address.*;
import org.zoolu.sip.provider.OptionHandler;
import org.zoolu.sip.provider.SipStack;
import org.zoolu.sip.provider.SipProvider;

import com.skype.Call;
import com.skype.CallListener;
import com.skype.Skype;
import com.skype.SkypeException;
import com.skype.NotAttachedException;
import com.skype.connector.Connector;

import java.text.DecimalFormat;
import java.util.Calendar;
import java.util.Enumeration;
import java.util.Vector;
import java.util.concurrent.TimeoutException;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.io.File;
import java.lang.reflect.Method;

import local.server.ServerProfile;
import local.ua.sscodecs.SSCodecFactory;
import local.ua.sscodecs.SSCodec;
import local.ua.test.SkypeConnectorReliabilityTest;

import org.zoolu.sip.provider.Identifier;
import local.server.SSRegistrar;
import local.server.LocationService;
import com.skype.CommandFailedException;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

/* Skype SIP user agent (UA).
 */
public class SkypeUA implements RegisterAgentListener, CallListener,ShutdownTimerInterface, OptionHandler, ControllerChannelInterface,ConfigWatchTimerInterface,ConnectorWatchTimerInterface
{           
	public enum ExitCode 
	{
	    NORMAL (0),
	    INITFAIL (1),
	    FATALERROR (2),
	    STARTERROR(9),
	    SKYPE4JAVACONNECTERROR(7),
	    REINITERROR(11),
	    CONFIGERROR(16);

	    private final int code;
	    ExitCode(int code) 
	    {
	        this.code = code;
	    }
	    public int code()   { return code; }
	}
	
   private Connector _instance=null;
	
   private static final String stsVersion="v20090902";	
	
   private Vector<SSCallChannel> callChannels=null;
   
   private SkypeProfile skype_profile;
   
   private String skypeUserId=null;

   /** Register Agent */
   private RegisterAgent ra;
   
   /** UserAgentProfile */
   private UserAgentProfile user_profile;
         
   private Logger log=null;
   
   private CallHistoryHandler callHistoryHandler=null;
   
   private ShutdownTimer shutdownTimer=null;
   private int shutdownTimerIntervalSeconds=300; // check every 5 minutes
   private long shutdownTimerFiredCount=0;

   private ConfigWatchTimer configWatcher=null;
   private int configWatchTimerIntervalSeconds=60; // check every 1 minutes
   private long configWatchTimerFiredCount=0;
   
   private Vector<ConfigFileInfo> configFiles=null;
   private AuthMap authMaps=null;
   
   private SipProvider sip_provider=null;
   private String transferSkypeId=null;
   private String initFile=null;
   
   private SSRegistrar mjServer=null;
   private ServerProfile server_profile=null;
   
   private ConnectorWatchDogTimer connectorWatchDog=null;
   private boolean lowBalanceNotificationSent=false;
   
    /** The main method. */
	public static void main(String[] args) throws Exception 
	   {         
			  String file="siptosis.cfg";
		      if (args.length>0)
		    	  file=args[0];
	
		      
			  if (args.length==0)
			  {	  
			     BasicInstaller installer=new BasicInstaller(getVMBitSize());
				 installer=null;
			  }
		  
			  try 
			  {
			    	new SkypeUA(file);
		      }
		      catch (Exception e)  
		      {  
		    	  e.printStackTrace(); 
		    	  System.exit(SkypeUA.ExitCode.STARTERROR.code());  
		      }
	   }    
	   	
   
   /** Costructs a UA with a default media port */
   public SkypeUA(String file) throws Exception
   {
	   initUA(file);
	   
	   run();
   }

   public void initUA(String file) throws Exception
   {
	  initFile=file;
      sip_provider=new SipProvider(file);
      user_profile=new UserAgentProfile(file);
      skype_profile=new SkypeProfile(file);
      server_profile=new ServerProfile(file);

	  
      if (skype_profile.runConnectorReliabilityTest)
      {
    	  SkypeConnectorReliabilityTest srt=new SkypeConnectorReliabilityTest();
    	  System.exit(SkypeUA.ExitCode.NORMAL.code);
      }
      

      
	  log = Logger.getLogger(this.getClass().getName());
	  int bits=getVMBitSize();
	  log.info("Starting SipToSis "+stsVersion);
	  log.info("os="+System.getProperty("os.name")+" arch="+System.getProperty("os.arch")+" ver="+System.getProperty("os.version"));
	  log.info("javaVer="+System.getProperty("java.version")+" - "+System.getProperty("java.vendor")+" ("+bits+" bit)");
	  log.debug("javaLibPath="+System.getProperty("java.library.path"));
	  log.debug("javaClassPath="+System.getProperty("java.class.path"));

      if (bits==64 && System.getProperty("os.name").toLowerCase().contains("windows") && (System.getenv("JAVAEXEPATH")==null || System.getenv("JAVAEXEPATH").length()==0))
      {
    	  log.warn("************************************************************");
    	  log.warn("* WARNING: JAVAEXEPATH not set - Recommended for 64 bit VM *");
    	  log.warn("***********************************************************");
      }    
	  
	  
	  SecurityManager security = System.getSecurityManager();
	  if (security!=null)
		  log.warn("#### security manager installed, this will slow down RTP packets");
	  
	  
      // test and setup codecs
	  setupCodecs();
	  
	  if (skype_profile.connect)
	  {	  
		  this.skypeUserId=initSkype();
		  
		  if (skype_profile.SkypeInboundAllChannelsBusyAction.startsWith("transferto:"))
	 	  {
	 			transferSkypeId=skype_profile.SkypeInboundAllChannelsBusyAction.replaceAll("transferto:","");
		 		if (transferSkypeId.equalsIgnoreCase(this.skypeUserId))
		 		{
		 			log.fatal("Configuration error - Skype User ID and transferTo ID are the same");
		 			System.exit(SkypeUA.ExitCode.CONFIGERROR.code());
		 		}
	 	  }
	  }
	  else
	  {	  
		  this.skypeUserId="bogus";
		  log.info("Skype disconnected mode");
	  }
	  
	  if (this.skypeUserId==null)
	  {
		  log.info("not connected to skype.");
		  return;
	  }

	  
	  if (connectorWatchDog!=null)
	  {
		  connectorWatchDog.stopTimer();
		  connectorWatchDog=null;
	  }
	  
	  
	  if (shutdownTimer!=null)
	  {
		  shutdownTimer.stopTimer();
		  shutdownTimer=null;
	  }
	  
	  if (skype_profile.autoShutdownMinutes>0)
	  {	  
		 int intMin=shutdownTimerIntervalSeconds/60; 
		 skype_profile.autoShutdownMinutes=(int)((skype_profile.autoShutdownMinutes+intMin-1)/intMin)*intMin;
		 
		 log.info("Auto Shutdown in "+skype_profile.autoShutdownMinutes+" minutes.");
		 shutdownTimer=new ShutdownTimer(shutdownTimerIntervalSeconds,this);
	  }

	  if (configWatcher!=null)
	  {
		  configWatcher.stopTimer();
		  configWatcher=null;
	  }
	  
	  
	  if (skype_profile.configWatchInterval>0)
	  {
		  	 configFiles=new Vector<ConfigFileInfo>();
		     configFiles.add(new ConfigFileInfo(file,true));
		     configFiles.add(new ConfigFileInfo(skype_profile.SipOutDialingRulesFile,false));
		     configFiles.add(new ConfigFileInfo(skype_profile.SipToSkypeAuthFile,false));
		     configFiles.add(new ConfigFileInfo(skype_profile.SkypeOutDialingRulesFile,false));
		     configFiles.add(new ConfigFileInfo(skype_profile.SkypeToSipAuthFile,false));
		  
		  	 int intMin=configWatchTimerIntervalSeconds/60; 
			 skype_profile.configWatchInterval=(int)((skype_profile.configWatchInterval+intMin-1)/intMin)*intMin;
			 log.info("configWatchInterval every "+skype_profile.configWatchInterval+" minutes.");
			 configWatcher=new ConfigWatchTimer(configWatchTimerIntervalSeconds,this);
	  }

	  if (skype_profile.connectorWatchDogMinutes>0)
	  {
		  connectorWatchDog=new ConnectorWatchDogTimer(skype_profile.connectorWatchDogMinutes,this);
	  }
	  
	  
      ra=new RegisterAgent(sip_provider,this,user_profile);

      /*
      // fix the from_url - make sure it has a name
      NameAddress tmpUrl=new NameAddress(user_profile.from_url);
      if (!tmpUrl.hasDisplayName())
      {	  
    	  tmpUrl.setDisplayName("unknown");
    	  user_profile.from_url=tmpUrl.toString();
      }
      */
      
      int rtpPort=this.user_profile.audio_port;
      int skypePort=skype_profile.SkypeAudioPortBase;
      
      log.info("Config - skypeClientSupportsMultiCalls:"+skype_profile.skypeClientSupportsMultiCalls+"  concurrentCallLimit:"+user_profile.concurrentCallLimit);
      log.info("SipToSis contact_url="+ra.contact);
      String dspRealm=user_profile.realm;
      if (dspRealm==null)
    	  dspRealm="";
    	  
      log.info("via_addr="+sip_provider.getViaAddress()+"  realm="+dspRealm);
      if (ra.contact.getAddress().getPort()!=sip_provider.getPort())
      {	  
    	  log.error("####### Config Error - host_port != contact_url port ########");
    	  System.exit(SkypeUA.ExitCode.CONFIGERROR.code());
      }
      
      log.info("RTP Ports: "+rtpPort+"-"+(rtpPort+(user_profile.concurrentCallLimit*2)-2)+"  Local Skype Ports: "+skypePort+"-"+(skypePort+(user_profile.concurrentCallLimit*2)-1));
	  if (user_profile.concurrentCallLimit<1)
		  log.error("######### invalid concurrentCallLimit ##############");
	  

	  initAuthMaps();

	  callChannels=new Vector<SSCallChannel>();
      if (!skype_profile.forceChannelBusy)
      {	  
	      for (int c=0;c<user_profile.concurrentCallLimit;c++)
	      {
	    	  SSCallChannel ssChan=new SSCallChannel(sip_provider,user_profile,skype_profile,rtpPort,skypePort,this,c);
	   		  callChannels.add(ssChan);
	    	  skypePort+=2;
	    	  rtpPort+=2;
	      }
      }
      else
    	  sip_provider.addSipProviderListener(new Identifier("FAKE"),new org.zoolu.sip.dialog.InviteDialog(null,null));
      
      sip_provider.setOPTIONHandler(this);
      
      if (server_profile.is_registrar)
    	  mjServer=new SSRegistrar(sip_provider, server_profile);

      
      NameAddress tmpfrom=new NameAddress(user_profile.from_url);
      if (user_profile.do_register && tmpfrom.getAddress().getHost().equals(sip_provider.getViaAddress()) && tmpfrom.getAddress().getPort()==sip_provider.getPort())
          log.error("###### INVALID CONFIGURATION - Registering with self. #####");		  


      String tmpMsg="MaxCallTime: ";
	  if (skype_profile.maxCallTimeLimitMinutes>0)
		  tmpMsg+=skype_profile.maxCallTimeLimitMinutes;
	  else
		  tmpMsg+="not limited";

      tmpMsg+=" MaxPSTNCallTime: ";
	  if (skype_profile.MaxPstnCallTimeLimitMinutes>0)
		  tmpMsg+=skype_profile.MaxPstnCallTimeLimitMinutes;
	  else
		  tmpMsg+="not limited";
	  
	  
	  log.info(tmpMsg);

      String tmpMsg2="MaxDailyPSTNUniqueNumberCount: ";
	  if (skype_profile.dailyPstnUniqueNumberLimit>0)
		  tmpMsg2+=skype_profile.dailyPstnUniqueNumberLimit;
	  else
		  tmpMsg2+="not limited";

      tmpMsg2+=" MaxDailyPSTNMinutes: ";
	  if (skype_profile.dailyPstnLimitMinutes>0)
		  tmpMsg2+=skype_profile.dailyPstnLimitMinutes;
	  else
		  tmpMsg2+="not limited";
	  
	  log.info(tmpMsg2);
	  
	  this.callHistoryHandler = new CallHistoryHandler(this.skypeUserId,skype_profile.callLogPath,skype_profile.tollFreeNumberPrefixes);
	  if (skype_profile.loadSkypeClientCallHistory)
		  loadSkypeCallHistory();
	  log.info("PSTN counters reset at: "+callHistoryHandler.getResetTime());
	  showCallHist();
	  
      SipStack.init(file);
      
      
      if (skype_profile.emailRecipients==null || skype_profile.emailRecipients.length()<1
    	   || skype_profile.emailHost==null || skype_profile.emailHost.length()<1
    	   || skype_profile.emailWhenBalanceDropsTo<0
    	  )
      {	  
    	  skype_profile.emailWhenBalanceDropsTo=-1;
      }
      else
    	 log.info("Low balance notifier active."); 
      
      
      double curBal=util.getSkypeCreditBalance();
      log.info(util.formatSkypeCreditBalance(curBal));
      this.updateSkypeCreditBalance(curBal);
      
      
      if (skype_profile.emailTest)
      {
    	   new MailerThread("Skype Low Balance Alert Test","It works!",skype_profile.emailHost,skype_profile.emailPort,skype_profile.emailUsername,skype_profile.emailPassword,skype_profile.emailRecipients,skype_profile.emailFrom);
      }
      
	  startRA();
   }


   /** Register with the registrar server.
     * @param expire_time expiration time in seconds */
   public void register(int expire_time)
   {  if (ra.isRegistering()) ra.halt();
      ra.register(expire_time);
   }


   /** Periodically registers the contact address with the registrar server.
     * @param expire_time expiration time in seconds
     * @param renew_time renew time in seconds
     * @param keepalive_time keep-alive packet rate (inter-arrival time) in milliseconds */
   public void loopRegister(int expire_time, int renew_time, long keepalive_time)
   {  if (ra.isRegistering()) ra.halt();
      ra.loopRegister(expire_time,renew_time,keepalive_time);
   }


   /** Unregister with the registrar server */
   public void unregister()
   {  if (ra.isRegistering()) ra.halt();
      ra.unregister();
   }


   /** Unregister all contacts with the registrar server */
   public void unregisterall()
   {  if (ra.isRegistering()) ra.halt();
      ra.unregisterall();
   }

   /** Starts everything */
   public void run()
   {
   }

   private void startRA()
   {
		if (user_profile.do_unregister_all)
		     // ########## unregisters ALL contact URLs
		     {  log.info("UNREGISTER ALL contact URLs");
		        unregisterall();
		     } 
		
		     if (user_profile.do_unregister)
		     // unregisters the contact URL
		     {  log.info("UNREGISTER the contact URL");
		        unregister();
		     } 
		
		     if (user_profile.do_register)
		     // ########## registers the contact URL with the registrar server
		     {  log.info("REGISTRATION");
		        loopRegister(user_profile.expires,user_profile.expires/2,user_profile.keepalive_time);
		     }               
   }
   

   /** Exits */
   public void exit()
   {  try {  Thread.sleep(1000);  } catch (Exception e) {}
      System.exit(SkypeUA.ExitCode.NORMAL.code());
   }



   // **************** RegisterAgent callback functions *****************

   /** When a UA has been successfully (un)registered. */
   public void onUaRegistrationSuccess(RegisterAgent ra, NameAddress target, NameAddress contact, String result)
   {  log.info("Registration success: "+result);
   }

   /** When a UA failed on (un)registering. */
   public void onUaRegistrationFailure(RegisterAgent ra, NameAddress target, NameAddress contact, String result)
   {  log.info("Registration failure: "+result);
   }
   


	private String initSkype()  throws Exception
	{
		String skypeUserId=null;

		if (skype_profile.skypeUserId!=null)
    		log.info("Connecting to Skype Client with User: "+skype_profile.skypeUserId);
		
		log.info("initSkype - If stuck, check Skype online & API auth");

		
    	setupSkype();
    	
    	skypeUserId=Skype.getProfile().getId();
    	
    	log.info("Attached SkypeUserId:"+skypeUserId);

    	return skypeUserId;
	}

	private void cleanupUA()
	{
		
		  if (callChannels!=null)
		  {
			  for (int c=0;c<callChannels.size();c++)
			  {	  
				  SSCallChannel cc=callChannels.get(c);
				  cc.halt();
			  }
			  callChannels=null;
		  }
		   
		  if (ra!=null)
		  {
			ra.haltRA();
			ra=null;
		  }	

		  if (mjServer!=null)
			  mjServer=null;
		  
		  if (sip_provider!=null)
		  {	  
			  sip_provider.halt();
			  sip_provider=null;
		  }
		  
		
        Skype.removeCallListener(this);
	}


	private void setupSkype() throws SkypeException
	{
		int retrycnt=0;
		while (true)
		{
			try
			{
				// this should compile with either connector
				if (skype_profile.skypeUserId!=null && System.getProperty("os.name").toLowerCase().indexOf("windows")>=0)
				{ // only windows can do this	
					Class connectorClass = Class.forName("com.skype.connector.Connector");
				    Object[] argVals = {skype_profile.skypeUserId};
		            Class[] argClasses = { Class.forName ("java.lang.String") };
	                Method getInstance = connectorClass.getMethod("getInstance",argClasses);
			        try
			        { 
			        	 _instance = (Connector) getInstance.invoke(connectorClass,argVals);
			        }
			        catch (Exception e)
			        {  
			        	 log.error("Connector error",e);
			        	 System.exit(SkypeUA.ExitCode.CONFIGERROR.code());
			        }
				}

				if (skype_profile.skypeAPITrace)
				{
				  Connector.getInstance().setDebugOut(new ConnectorDebugWriter());
				  Skype.setDebug(true);
				}
				else
				  Skype.setDebug(false);
				
				
				Connector.getInstance().setCommandTimeout(10000);
				log.info("SkypeVer:"+Skype.getVersion());
				Connector.getInstance().setCommandTimeout(2000);
				break;
			}
			catch (NotAttachedException e)
			{
				retrycnt++;
				if (retrycnt>15)
				{
					log.error("Connect attempt limit reached - exiting.",e);
					System.exit(SkypeUA.ExitCode.STARTERROR.code());
				}
				
				String sStatus=Connector.getInstance().getStatus().name();
			    log.info("Skype Status: "+sStatus+" - retrying every 5 seconds");
				try {Thread.sleep(5000);} catch(Exception te){}
			}
			catch (Throwable e)
			{
				testLoadLibrary();
				log.fatal("skype4java connect error: ",e);
				System.exit(SkypeUA.ExitCode.SKYPE4JAVACONNECTERROR.code());
			}
		}
	  
		Skype.addCallListener(this);
	}
	
	


	public void callReceived(Call call) throws SkypeException
	{
		 	log.info("callReceived - incoming Skype Call from:"+call.getPartnerDisplayName()+" ["+call.getPartnerId()+"] status:"+call.getStatus());

		 	SSCallChannel callChannel=getFreeChannel(true); // find a free channel and lock it
		 	if (callChannel==null)
	        {	
		 		log.info("All Channels in Use.");
		 		if (skype_profile.SkypeInboundAllChannelsBusyAction.equals("refuse"))
		 		{	
			 		log.info("Rejected Skype Call");
		        	call.cancel(); // sorry multiple calls not supported
		        	return;
		 		}
		 		else if (skype_profile.SkypeInboundAllChannelsBusyAction.equals("voicemail"))
		 		{
			 		log.info("Redirecting to VoiceMail");
		 			call.redirectToVoiceMail();
		 			return;
		 		}
		 		else if (skype_profile.SkypeInboundAllChannelsBusyAction.startsWith("transferto:"))
		 		{
			 		log.info("Redirecting Skype Call to "+transferSkypeId);
		 			call.transferTo(transferSkypeId);
		 			return;
		 		}
		 		else
		 		{
		 			log.info("Skype call ignored.");
		 			// ignore or not valid option
		 			return;
		 		}
	        }

		 	callChannel.setCurrentSkypeCall(call);
		    String sipDest=callChannel.getSipDest();
			if (sipDest==null || sipDest.length()==0)
			{
				  log.info("callReceived - rejected call");
				  callChannel.cancelSkypeCall();
				  callChannel.unLock();
				  return;
			}
			else
			{
				callChannel.skypeCallMaked();
				if (isSkypeCallClosed(call.getStatus())) // test to make sure call didn't get cancelled before the status listener got added
				{
					  log.info("callReceived - call cancelled");
					  callChannel.cancelSkypeCall();
					  callChannel.unLock();
					  return;
				}
			     
				if (sipDest.toLowerCase().indexOf("sip:")>=0)
				{
					 sipDest=getSipUserContact(sipDest);
					 //"Skype" <sip:skype@127.0.0.1>
				     
					 NameAddress tmpfrom;
					 String skypeid=callChannel.getCurrentSkypeCall().getPartnerId();	
					 if (skype_profile.replaceFromWithSkypeId)
					 {
						 SipURL tmpurl=new SipURL(callChannel.ua.user_profile.contact_url);
				    	 tmpfrom=new NameAddress(skypeid,new SipURL(skypeid,tmpurl.getHost(),tmpurl.getPort()));
					 }
					 else
						 tmpfrom=new NameAddress(skypeid,new SipURL(callChannel.ua.user_profile.contact_url));
				     
				     callChannel.ua.user_profile.from_url=tmpfrom.toString();

			    	 log.info("callReceived - Direct Sip Dial to:"+sipDest.replaceAll(";.*","")+" from:"+callChannel.ua.user_profile.from_url);
			    	 callChannel.call(sipDest);
				
				}
				else
				{
					log.info("callReceived - Command List:"+sipDest);
					if (!callChannel.answerSkypeCall(sipDest))
					{
						log.error("Skype Answer failed");
						callChannel.listen();
					}
				}
			}	
	}
	
	public void callMaked(Call call) throws SkypeException
	{
		   log.debug("callMaked: Skype "+call.getStatus()+" id="+call.getId());
		   
		   // seems this can be called by skype4java before skype actually returns back from the skypeCall function in SSChannel
		   int chkCount=0;
		   SSCallChannel callChannel=null;
		   
		   while (chkCount++<30)
		   {	   
			   callChannel=locateSkypeChannel(call);
			   if (callChannel!=null)
				   break;
			   else
				   try {Thread.sleep(5);}catch(Exception e){}
		   }
		   
		   if (callChannel==null)
		   {
			   boolean cancelUnknown=isLastSipCallRecent(5); 
			   boolean runAway=isPossibleRunaway(call.getId());  // make sure it's not a runaway call
			   if (!cancelUnknown && !runAway)
			   {
				   if (this.skype_profile.JoinManualSkypeOutboundCallToSip)
				   {
					   // need to stop the skype call, make sip call using the called skype userid as the context
					   // when sip answers, make the skype call.
					   log.error("Manual SIP<-->Skype Link mode 1.");
					   String calledUserId=call.getPartnerId();
					   call.cancel(); // cancel the skype call now
					   
					   // locate a free sip channel to use
					   SSCallChannel sipChan=getFreeChannel(true);
					   if (sipChan!=null)
						   sipChan.doSkypeSIPJoin(calledUserId);
					   else
						   log.error("No SIP channels available.");
				   }
				   else
					   log.info("Manual Outbound Skype Call.");
				   
			   }
			   else
			   {	   
				   log.error("Could not locate CallChannel for this Skype Call! id="+call.getId()+" ActiveSkypeCallIds="+getActiveSkypeCallIds());
				   if ((cancelUnknown || runAway) && (call.getType()==Call.Type.OUTGOING_P2P || call.getType()==Call.Type.OUTGOING_PSTN))
				   {	
					   if (!runAway)
						   log.error("Sip call within 5 second window, Forced hangup.");
					   else
						   log.error("Stopping runaway call.");
						   
					   try
					   {
					    call.cancel(); // cancel any unknown outgoing calls - possible if sip user cancelled before skype call started
					   }
					   catch (CommandFailedException e)
					   {
						 String msg=e.getLocalizedMessage();
						 if (msg==null)
							 log.debug("Cancel skype call error",e);
						 else if (!msg.contains("Cannot hangup inactive call"))
							 log.debug("Cancel skype call error",e);
					   }
				   }
			   }
			   return;
		   }

		   callChannel.skypeCallMaked();
	}
	
	
	public boolean isLastSipCallRecent(int seconds)
	{
		// return true if a sip was received in the last x seconds
	    long curTime=System.currentTimeMillis();
	   
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			if (Math.abs(curTime-callChannel.getLastSipCallTime())<=1000*seconds)
			{	
				return true;
			}
		}
	    return false;
	}

	public boolean isPossibleRunaway(String skypecallid)
	{
		// return true a channel has a matching callid
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			String lastId=callChannel.getPossibleRunawaySkypeCallid();
			if (lastId!=null && skypecallid.equals(lastId))
			{	
				return true;
			}
		}
	    return false;
	}

	
	public void onShutdownTimerTimeout()
	{
		shutdownTimerFiredCount++;
		
		if (allIdle())
		{
			if (shutdownTimerFiredCount>=skype_profile.autoShutdownMinutes*60/shutdownTimerIntervalSeconds)
			{
				shutdownTimer.stopTimer();
				log.info("Auto shutting down.");
				cleanupUA();
				System.exit(SkypeUA.ExitCode.NORMAL.code);
			}
		}
		
	}
	
	private synchronized SSCallChannel getFreeChannel(boolean lockChannel)
	{
		
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			if (callChannel.isIdle())
			{	
				if (lockChannel)
					callChannel.lock();
				return callChannel;
			}
		}
		return null;
	}
	
	private SSCallChannel locateSkypeChannel(Call skypeCall)
	{
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			if (callChannel.getCurrentSkypeCall()!=null && callChannel.getCurrentSkypeCall().getId()==skypeCall.getId())
			{	
				return callChannel;
			}
		}
		return null;
	}
	
	private String getActiveSkypeCallIds()
	{
		String retvar="";
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			if (callChannel.getCurrentSkypeCall()!=null)
			{
				if (retvar.length()>0)
					retvar+=",";
				retvar+=callChannel.getCurrentSkypeCall().getId();
			}
		}
		return retvar;
	}
	
	
	/*
	 * called when an SIP Option message is received
	 * 
	 */
	public String onOptionMsgReceived()
	{
		SSCallChannel callChannel=getFreeChannel(false);
		if (callChannel!=null)
			return callChannel.ua.getCapabilities();
		else
			return null;
	}
	
	/*
	 * called by SSCallChannel thru ControllerChannelInterface
	 */
	public void holdOtherSkypeCalls(Call excludedSkypeCall) throws Exception
	{
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			if (!callChannel.isIdle() && callChannel.getCurrentSkypeCall()!=null && (excludedSkypeCall==null || callChannel.getCurrentSkypeCall()!=excludedSkypeCall))
			{	
					try
					{
						callChannel.holdSkypeCall();
					}
					catch (Exception e)
					{
						log.error("Skype Hold Failed! ",e);
						throw new Exception(e);
					}
			}
		}
		
		// wait for all the skype calls to be on hold
		int timerCnt=0;
		int cntLimit=10; // 10 seconds
		while (timerCnt<cntLimit)
		{
			Thread.sleep(1000);
			if (allSkypeCallsHolding(excludedSkypeCall))
				return;
		}
		throw new TimeoutException("Hold Failed.");
	}
	
	private boolean allSkypeCallsHolding(Call excludedSkypeCall)
	{
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			if (!callChannel.isIdle() && callChannel.getCurrentSkypeCall()!=null && (excludedSkypeCall==null || callChannel.getCurrentSkypeCall()!=excludedSkypeCall))
			{	
					if (!callChannel.ua.skypeHoldingLocal)
						return false;
			}
		}
		return true;
	}
	
	private void testLoadLibrary()
    {
		
		String libName="skype";

		log.info("###testLoadLibrary###");
		log.info("javaLibPath:"+System.getProperty("java.library.path"));
		log.info("libName="+libName+" platform libname="+System.mapLibraryName(libName));
		
    	try
    	{
             System.loadLibrary(libName);
             log.info("Load library worked??? Why are we here then?");
    	}
    	catch (UnsatisfiedLinkError err)
    	{
    		log.error("Load Library failed",err);
    	}
    	catch (Throwable e)
    	{
    		log.error("Load Library failed",e);
    	}
    }


	private void setupCodecs()
	{
		  if (user_profile.audio_frame_size.length!=user_profile.audio_codec.length)
		  {
			  log.fatal("Codec Configuration Incorrect (You must specify audio_frame_size for each codec.");
			  System.exit(SkypeUA.ExitCode.CONFIGERROR.code());
		  }
		  
		  
		  boolean userDefinedTypes=false;
		  if (user_profile.audio_avp.length>0)
			  userDefinedTypes=true;

		  SSCodecFactory.init();
		  
		  Vector<String> codecsOK=new Vector<String>();
	      Vector<Integer> codecsRates=new Vector<Integer>();
	      Vector<Integer> codecsPayTypes=new Vector<Integer>();
	      Vector<Integer> codecsFrameRates=new Vector<Integer>();
	      Vector<Double> codecsAudionInGains=new Vector<Double>();
	      Vector<Double> codecsAudionOutGains=new Vector<Double>();
	      for (int cc=0;cc<user_profile.audio_codec.length;cc++)
	      {	  
	    	  	double inGain=skype_profile.SkypeAudioInGain[0];
	    	    if (cc<skype_profile.SkypeAudioInGain.length)
	    	    	inGain=skype_profile.SkypeAudioInGain[cc];

	    	  	double outGain=skype_profile.SkypeAudioOutGain[0];
	    	    if (cc<skype_profile.SkypeAudioOutGain.length)
	    	    	outGain=skype_profile.SkypeAudioOutGain[cc];
	    	    	
	    	  SSCodec sscodec=SSCodecFactory.configureCodec(user_profile.audio_codec[cc],user_profile.audio_frame_size[cc],inGain,outGain);
	    	  if (sscodec!=null)  
	    	  { 
		    	    codecsOK.add(sscodec.getCodecName());
		    	    codecsRates.add(sscodec.getSampleRate());
		    	    codecsFrameRates.add(sscodec.getFrameSize());
		    	    int ptype=sscodec.getPayloadType();
		    	    if (userDefinedTypes && user_profile.audio_avp[cc]>=0)
		    	    {
		    	    	try
		    	    	{
		    	    	  SSCodecFactory.modifyCodecMap(ptype,user_profile.audio_avp[cc]);
			    	      ptype=user_profile.audio_avp[cc];
		    	    	}
		    	    	catch(Exception e)
		    	    	{
		    	    		log.error("Codec: "+user_profile.audio_codec[cc]+ " error: "+e.getLocalizedMessage());
		    	    		System.exit(SkypeUA.ExitCode.INITFAIL.code());
		    	    	}
		    	    }
		    	    
	    	    	if (ptype<0)
	    	    	{
	    	    		log.error("Codec: "+user_profile.audio_codec[cc]+" is a dynamic payload type - you must set the RTP payload type in the config file");
	    	    		System.exit(SkypeUA.ExitCode.CONFIGERROR.code());
	    	    	}

		    	    codecsPayTypes.add(ptype);
	    	    	codecsAudionInGains.add(inGain);
		    	   	codecsAudionOutGains.add(outGain);
	    	  }
	    	  else
	    	  {
	    		  log.error("Codec: "+user_profile.audio_codec[cc]+" not loaded.");
	    	  }
	      }
	      
	      String codecList="";
	      user_profile.audio_codec=new String[codecsOK.size()];
	      user_profile.audio_sample_rate=new int[codecsRates.size()];
	      user_profile.audio_avp=new int[codecsPayTypes.size()];
	      user_profile.audio_frame_size=new int[codecsPayTypes.size()];
	      skype_profile.SkypeAudioInGain=new double[codecsAudionInGains.size()];
	      skype_profile.SkypeAudioOutGain=new double[codecsAudionOutGains.size()];
	      for (int cc=0;cc<codecsOK.size();cc++)
	      {
	          user_profile.audio_codec[cc]=codecsOK.get(cc);
	          user_profile.audio_sample_rate[cc]=codecsRates.get(cc);
	          user_profile.audio_avp[cc]=codecsPayTypes.get(cc);
	          user_profile.audio_frame_size[cc]=codecsFrameRates.get(cc);
		      skype_profile.SkypeAudioInGain[cc]=codecsAudionInGains.get(cc);
		      skype_profile.SkypeAudioOutGain[cc]=codecsAudionOutGains.get(cc);
	          if (codecList.length()>0)
	        	  codecList+=",";
	          codecList+=user_profile.audio_codec[cc];
	          if (user_profile.audio_sample_rate[cc]!=8000)
	        	  codecList+="/"+(user_profile.audio_sample_rate[cc]/1000)+"k";
	          codecList+="("+user_profile.audio_avp[cc]+")";
	      }
	      
	      if (user_profile.audio_codec.length==0)
	      {
	    	  log.fatal("No Codecs available. Exiting.");
	    	  System.exit(SkypeUA.ExitCode.CONFIGERROR.code());
	      }
	      log.info("Available Codecs: "+codecList);
	      if (this.user_profile.dtmf2833PayloadType>=0)
	    	  log.info("DTMF rfc2833("+this.user_profile.dtmf2833PayloadType+")");
	      
	      log.debug("Codec Config:\r\n"+SSCodecFactory.getConfiguredCodecs());
	}
	
	
	public void onConfigWatcherTimeout()
	{
		configWatchTimerFiredCount++;

		if (configWatchTimerFiredCount>=skype_profile.configWatchInterval*60/configWatchTimerIntervalSeconds)
		{
			// time to check
			boolean secondaryChanged=false;
			boolean primaryChanged=false;
			String priFile=null;
			
			for (int x=0;x<configFiles.size();x++)
			{
				ConfigFileInfo cfi=configFiles.get(x);
				File tmpFile=new File(cfi.filename);
				if (cfi.lastModified!=tmpFile.lastModified())
				{
					log.debug("Config File: "+cfi.filename+" changed");
					
					if (cfi.primaryFile)
					{
						  primaryChanged=true;
						  priFile=cfi.filename;
						  break; // no point in going on
					}
					else
					{	
					  secondaryChanged=true;
					  cfi.lastModified=tmpFile.lastModified();
					}  
				}
				Thread.yield();
			}
			
			
			if (primaryChanged)
			{
			
				if (!allIdle())
					return;

				log.info("Primary Configuration changed - Restarting.");
				try
				{
					cleanupUA();
					initUA(priFile);
				}	
			    catch (Exception e)  
			    {  
			   	  e.printStackTrace(); 
			   	  System.exit(SkypeUA.ExitCode.REINITERROR.code());  
			    }
			}
			else if (secondaryChanged)
			{
				// just reload auth maps
				log.info("Configuration changed - Reloading maps/rules.");
				resetChannelAuthMaps();
			}
			
			configWatchTimerFiredCount=0; // reset
		}
	}

	private boolean allIdle()
	{
		// wait for all channels idle before restart
		for (int c=0;c<callChannels.size();c++)
		{
			SSCallChannel callChannel=callChannels.get(c);
			if (!callChannel.isIdle())
			{	
				return false;
			}
		}
		return true;
	}
	
	private void initAuthMaps()
	{
	  this.authMaps=new AuthMap(skype_profile.SipToSkypeAuthFile,skype_profile.SkypeToSipAuthFile,skype_profile.SkypeOutDialingRulesFile,skype_profile.SipOutDialingRulesFile);
	}
	
	private void resetChannelAuthMaps()
	{
		initAuthMaps();
	}
	
	public AuthMap getAuthMap()
	{
		return this.authMaps;
	}
	
	
	public boolean lostSkypeConnection()
	{
		log.error("Lost Skype client connection - reconnecting.");
		
		// try to connect a few times
		int retries=0;
		while (retries++<5)
		{	
			try
			{
				this.skypeUserId=Skype.getProfile().getId();
				log.info("Reconnected to Skype client.");
				return true;
			}
			catch (NotAttachedException e)
			{
				try {Thread.sleep(3000);} catch(Exception te){}
			}
			catch (Throwable e)
			{
				break;
			}
	    }
		return false;
	}	
		
	public void restart()
	{
   	    log.error("Performing full restart.");
		try {Thread.sleep(2000);} catch(Exception te){}
		try
		{
			cleanupUA();
			initUA(initFile);
		}	
	    catch (Exception e)  
	    {  
	   	  e.printStackTrace(); 
	   	  System.exit(SkypeUA.ExitCode.REINITERROR.code());  
	    }
	}
	
	public String getSipUserContact(String sipDest)
	{
		if (mjServer==null)
			return sipDest;
		
	    Pattern sipidPat=Pattern.compile("sip:<?([^@<]+)@",Pattern.CASE_INSENSITIVE);
	    Matcher cuidMat=sipidPat.matcher(sipDest);
	    String callUID=null;
	    if (!cuidMat.find())
		   return sipDest;
	    callUID=cuidMat.group(1);
		   
	    
		LocationService ls=mjServer.getLocationService();
		String[] testDomains=server_profile.domain_names;
		if (testDomains.length==0)
		{
			 testDomains=new String[1];
			 testDomains[0]=new String(sip_provider.getViaAddress());
		}
	    for (int d=0;d<testDomains.length;d++)
	    {	
			String searchUser=callUID+"@"+testDomains[d];
			Enumeration c=ls.getUserContactURLs(searchUser);
			if (c!=null)
			{	
				while (c.hasMoreElements())
		        { 
				   String contact=(String)c.nextElement();
		           if (ls.isUserContactExpired(searchUser,contact)==false)
		           {
		        	  sipDest=contact.replaceAll(";.*","");
		        	  log.info("Route to registered target:"+sipDest);
		        	  return sipDest;
		           }
		        }
			}
	    }
		
		return sipDest;
	}

	public void onConnectorWatcherTimeout()
	{
		// test connection and attempt reconnect if invalid
		boolean reconnect=false;
		//log.info("ConnectorWatch Fired");
		try
		{
			Skype.getVersion();
		}
		catch(Throwable e)
		{
			reconnect=true;
			log.error("Watcher: Lost Skype client connection - attempting reconnect.");
		}
		
		if (reconnect)
		{	
			int retries=0;
			while (retries++<3)
			{
				try
				{
					this.skypeUserId=Skype.getProfile().getId();
					log.info("Watcher: Skype client reconnected to: "+this.skypeUserId);
					return;
				}
				catch (Throwable e)
				{
					try{Thread.sleep(1000);}catch(Exception se){}
				}
			}
			log.error("Watcher: Reconnect to Skype client failed.");
		}
	}
	
	public void showCallHist()
	{
		  log.info("Qualified PSTN calls today: "+callHistoryHandler.getPstnTodayCountQualified()+" Time: "+callHistoryHandler.getPstnTodayTimeQualified()+" minutes");  
	}
	
	public CallHistoryHandler getCallHistoryHandler()
	{
		return this.callHistoryHandler;
	}
	
	private void loadSkypeCallHistory()
	{
	       log.info("Loading Skype PSTN Call History");
	       String callsResp=util.parseSkypeResponse("SEARCH CALLS","CALLS");
	       log.debug("CallsList:"+callsResp);
	       if (callsResp.trim().length()==0)
	       {
	    	   log.info("0 possible calls to import.");
	    	   return;
	       }
	       String[] calls=callsResp.split(",");
	       log.info(calls.length+" possible calls to import.");

	       // seems the list is in descending order - so we can cut off when the day changes
	       Calendar cutOffDate=Calendar.getInstance();
	       cutOffDate.set(Calendar.HOUR_OF_DAY, 0);
	       cutOffDate.set(Calendar.MINUTE, 0);
	       cutOffDate.set(Calendar.SECOND, 0);
	       cutOffDate.add(Calendar.DATE,-1);
	       int impCount=0;
	       for (int x=0;x<calls.length;x++)
	       {
	         CallHistoryEntry ce=new CallHistoryEntry();             
	         String curId=calls[x].trim();
	         if (curId.length()>0)
	         {
	        	 long unixStamp=-1;
	             try
	             {
	               ce.callId=util.parseSkypeResponse("GET CALL "+curId+" TIMESTAMP","CALL "+curId+" TIMESTAMP").trim();
	               unixStamp=Long.parseLong(ce.callId);
	             }
	             catch(Exception e)
	             {
	                 log.error("History Load Error: Skype CallId="+curId,e);   
	             }
	             
	             if (unixStamp>0)
	             {   
	            	 Calendar callStamp=Calendar.getInstance();
	                 //seems to be the correct time  callStamp.setTimeZone(java.util.SimpleTimeZone.getTimeZone("GMT"));
	                 callStamp.setTimeInMillis(unixStamp*1000);
	                 ce.startTime=callStamp.getTime();
	                 
	                 if (cutOffDate.before(callStamp))
	                 {
	                     ce.callType=Call.Type.valueOf(util.parseSkypeResponse("GET CALL "+curId+" TYPE","CALL "+curId+" TYPE"));
	                     if (ce.callType==Call.Type.OUTGOING_PSTN)
	                     {
	                    	 ce.durationSeconds=Long.parseLong(util.parseSkypeResponse("GET CALL "+curId+" DURATION","CALL "+curId+" DURATION"));
	                         ce.callFrom=this.skypeUserId;
	                         ce.callTo=util.parseSkypeResponse("GET CALL "+curId+" PARTNER_HANDLE","CALL "+curId+" PARTNER_HANDLE");
	                         ce.callCost="Imported Call";
	                         if (callHistoryHandler.addCall(ce))
	                             impCount++;
	                         //log.info(ce);
	                     }
	                 }
	                 else
	                     break; // out of needed date range
	             }
	         }         
	      }
	      log.info(impCount+" PSTN calls imported");
	}
	 




	private boolean isSkypeCallClosed(Call.Status status)
	{
		if (status==Call.Status.MISSED || status==Call.Status.CANCELLED 
			|| status==Call.Status.FAILED || status==Call.Status.FINISHED 
			|| status==Call.Status.BUSY || status==Call.Status.REFUSED 
			|| status==Call.Status.UNPLACED)
			return true;
		else
			return false;
	}
	
	  private static int getVMBitSize()
	  {
		  String dmBits = System.getProperty("sun.arch.data.model", "novalue");
		  if (dmBits.equals("32"))
			  return 32;
		  else if (dmBits.equals("64"))
		      return 64;
		  else if (dmBits.equals("novalue")) 
		  {
		      if (System.getProperty("java.vm.name").indexOf("64") >= 0)
		    	  return 64;
		  } 
  		  return 32;
	  }	  
	

	  public void updateSkypeCreditBalance(double bal)
	  {
		  // track balance and send email notification if switched under threshold
		  if (skype_profile.emailWhenBalanceDropsTo<0)
			  return;
		  else if (bal>skype_profile.emailWhenBalanceDropsTo)
		  {	  
			  if (lowBalanceNotificationSent)
			  {	  
				log.info("Low balance condition has been reset.");
			  	lowBalanceNotificationSent=false;
			  }
		  }
		  else if (!lowBalanceNotificationSent)
		  {
			  log.info("Sending low balance notification.");
			  String msg="Skype Account: \""+this.skypeUserId+"\" balance is now: "+util.formatAmount(bal,2);
			  new MailerThread("Skype Low Balance Alert", msg, skype_profile.emailHost, skype_profile.emailPort,skype_profile.emailUsername,skype_profile.emailPassword, skype_profile.emailRecipients,skype_profile.emailFrom); 
			  lowBalanceNotificationSent=true;
		  }
	  }
}


