/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * 
 * FIR Algorithms from http://www.dsptutor.freeuk.com/FIRFilterDesign/FIRFiltDes102.html
 * 
 * Author(s):
 * Greg Dorfuss 
 */

package local.ua.sscodecs;


public class FIRFilter implements FilterInterface
{
	public enum FILTTYPE {LP,HP,BP};
	public enum WINDOW {RECTANGULAR,HANNING,HAMMING,BLACKMAN};
	private int sampleRate;
	private float freqMax;
	private float freqMin;
	private int filterOrder;
	private WINDOW windowType;
	private FILTTYPE filterType;
	private float[] windowCoef=null;
	private int chunkByteSize;
	private int chunkShortSize;

	private float sampleSum;
	private int outPos;
    private float fN;
    private short[] sampleBuffer;  


	public FIRFilter(int order,WINDOW winType,FILTTYPE filtType,float minFreq,float maxFreq,int argSampleRate,int argChunkSize)
	{
		windowType=winType;
		filterType=filtType;
		sampleRate=argSampleRate;
		fN=(float) (sampleRate*.5);
		filterOrder=order;
		freqMax=maxFreq;
		freqMin=minFreq;
		chunkByteSize=argChunkSize;
		chunkShortSize=chunkByteSize/2;
		windowCoef=design();
		windowCoef=gainCompCoef(windowCoef,1);
		sampleBuffer=new short[chunkShortSize+filterOrder];
	}

	

	
	  private float trBand() 
	  {
		    switch (windowType) {
		      case RECTANGULAR: return 1.84f*fN/filterOrder;
		      case HANNING: return 6.22f*fN/filterOrder;
		      case HAMMING: return 6.64f*fN/filterOrder;
		      case BLACKMAN: return 11.13f*fN/filterOrder;
		      default: return 999.0f;
		    }
	 }
	
	
	  private float[] design() 
	 {
		    // window function values
		    int m = filterOrder / 2;
		    float[] win = new float[m+1];
		    float r;
		    switch (windowType) {
		      case RECTANGULAR:
		        for (int n=1; n <= m; n++)
		          win[n] = 1.0f;
		        break;
		      case HANNING:
		        r = (float)Math.PI/(m+1);
		        for (int n=1; n <= m; n++)
		          win[n] = 0.5f + 0.5f*(float)Math.cos(n*r);
		        break;
		      case HAMMING:
		        r = (float)Math.PI/m;
		        for (int n=1; n <= m; n++)
		          win[n] = 0.54f + 0.46f*(float)Math.cos(n*r);
		        break;
		      case BLACKMAN:
		        r = (float)Math.PI/m;
		        for (int n=1; n <= m; n++)
		          win[n] = 0.42f + 0.5f*(float)Math.cos(n*r) + 0.08f*(float)Math.cos(2*n*r);
		        break;
		     default: for (int n=1; n <= m; n++) win[n] = 1.0f;
		    }

		    float w0 = 0.0f;
		    float w1 = 0.0f;
		    switch (filterType) 
		    {
		      case LP: w0 = 0.0f;
		               w1 = (float)Math.PI*(freqMax + 0.5f*trBand())/fN;
		               break;
		      case HP: w0 = (float)Math.PI;
			       w1 = (float)Math.PI*(1.0f - (freqMin - 0.5f*trBand())/fN);
			       break;
		      case BP: w0 = 0.5f * (float)Math.PI * (freqMin + freqMax) / fN;
		               w1 = 0.5f * (float)Math.PI * (freqMax - freqMin + trBand()) / fN;
			       break;
		    }

		    float[] a = new float[filterOrder+1];
		    a[0] = w1 / (float)Math.PI;
		    for (int n=1; n <= m; n++)
		      a[n] = (float)Math.sin(n*w1)*(float)Math.cos(n*w0)*win[n]/(n*(float)Math.PI);
		    
		   
		    for (int n=m+1; n<=filterOrder; n++) 
		    	a[n] = a[n - m];
		    
		    for (int n=0; n<=m-1; n++) 
		    	a[n] = a[filterOrder - n];
		    
		    a[m] = w1 / (float)Math.PI;
		    
		   // for (int t=0;t<a.length;t++)
		   // 	System.out.println(t+"="+a[t]);
		    
		    
		    return a;
	 }
	 
	 private float[] gainCompCoef(float[] coef,float gainComp)
	 {
		 for (int c=0;c<coef.length;c++)
			 coef[c]*=gainComp;
		 
		 return coef;
	 }
	
	
	public void filter(byte[] srcBuf)
	{
		  // output is always behind by half FilterOrder samples
		 
		  // fill the sample buffer
		  int spos=filterOrder;
		  for (int b=0;b<chunkByteSize;b+=2)
			 sampleBuffer[spos++]=(short) (srcBuf[b + 1] << 8 | srcBuf[b] & 0xff);
		
		
		  // apply the filter
		  outPos=0;
		  for (int sampIndex=0;sampIndex<chunkShortSize;sampIndex++)
		  {	  
			  sampleSum = 0;
			  for (int r=0;r<=filterOrder;r++)
				    sampleSum   = sampleSum + (windowCoef[r] * sampleBuffer[sampIndex + r]);
				  
		    
			 /*
			 System.out.print(" "+sampleBuffer[sampIndex+halfFilterOrder]+"="+ (short) sampleSum);
	    	 if ((sampIndex+1) % 10==0)
	    		System.out.println("");
	    	 */
	    	 
	    	 // output the sample
	    	 srcBuf[outPos++] = (byte)((short)sampleSum);
	    	 srcBuf[outPos++] = (byte)((short)sampleSum >> 8);
		  }
		  
		  //copy the tail over for next round
		  System.arraycopy(sampleBuffer, sampleBuffer.length-filterOrder, sampleBuffer, 0, filterOrder);
	}

}
