/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Based on mjsip 1.6 software and skype4java
 * 
 * Author(s):
 * Greg Dorfuss
 */
package local.ua.sscodecs;
import java.util.Enumeration;
import java.util.Hashtable;
import java.lang.IllegalArgumentException;

import org.apache.log4j.Logger;

public class SSCodecFactory
{
	private static Logger log = null;
	
	/*
	static int PCMU=0;
	static int GSM=3;
	static int G723=4;
	static int DVI4=5; // 8k
	static int DVI4=6; // 16k
	static int LPC=7;
	static int PCMA=8;
	static int G722=9;
	static int L16=10; //44100 stereo
	static int L16=11; //44100 mono
	static int QCELP=12;
	static int CN=13;
	static int MPA=14; //90k
	static int G728=15;
	static int DVI4=16; //11025
	static int DVI4=17; //22050
	static int G729=18;
	
	// following are dynamic
	static int G726-32=2;
	static int G726-40=96;
	static int G726-24=97;
	static int G726-16=98;
	static int speex=??;
	static int iLBC=??;
	*/

	private static final String[] officialNames={"PCMU","GSM","G723","DVI4","LPC","PCMA","G722","L16","QCELP","CN","MPA","G728","G729","G726-16","G726-24","G726-32","G726-40","speex","iLBC"};
	
	
	private static Hashtable<Integer,SSCodecInfo> codecNumberMap=null;
	private static Hashtable<String,SSCodecInfo> codecClassNameMap=null;
	private static Hashtable<String,SSCodecInfo> codecDynamicMap=null;
	
	
	public static void init()
	{
		codecNumberMap=new Hashtable<Integer,SSCodecInfo>();
		codecClassNameMap=new Hashtable<String,SSCodecInfo>();
		codecDynamicMap=new Hashtable<String,SSCodecInfo>();
		log = Logger.getLogger(SSCodecFactory.class);
	}
	
	
	public static SSCodec configureCodec(String codecClassName,int frameSize,double inGain,double outGain)
	{
		// test it first
		SSCodecInfo codecInfo=new SSCodecInfo(codecClassName,"N/A",frameSize,inGain,outGain);
		SSCodec sscodec=null;
		try
		{
			sscodec=getCodec(codecInfo);
		}
		catch (Exception e)
		{
  		  log.error("Codec Error: "+codecClassName+" - "+e.getLocalizedMessage());
  		  return null;
		}
		
		if (codecNumberMap.containsKey(sscodec.getPayloadType()))
		{
	  		  log.error("Error: payload Type "+sscodec.getPayloadType()+" already configured.");
	  		  return null;
		}

		codecInfo.codecOfficialName=sscodec.getCodecName();
		
		// test if official name
		boolean matched=false;
		for (int o=0;o<officialNames.length;o++)
		{
			if (officialNames[o].equals(sscodec.getCodecName()))
			{
				matched=true;
				break;
			}
		}
		
		if (!matched)
			log.warn("Codec: "+codecClassName+" - internal Codec name:"+sscodec.getCodecName()+" is not official name");
		
		// passed, add to maps
		codecNumberMap.put(sscodec.getPayloadType(), codecInfo);
		codecClassNameMap.put(codecClassName, codecInfo);
		if (sscodec.getPayloadType()<0 || (sscodec.getPayloadType()>=96 && sscodec.getPayloadType()<=127))
			codecDynamicMap.put(sscodec.getCodecName().toUpperCase() + "/" + sscodec.getSampleRate(), codecInfo);
		return sscodec;
	}
	
	public static void modifyCodecMap(int oldPayType,int newPayloadType) throws IllegalArgumentException
	{
		if (codecNumberMap.containsKey(newPayloadType))
		{
	  		  throw new IllegalArgumentException("payload type number: "+newPayloadType+" already in use");
		}
		
		SSCodecInfo codecInfo=codecNumberMap.get(oldPayType);
		codecNumberMap.remove(oldPayType);
		codecNumberMap.put(newPayloadType, codecInfo);
	}
	
	
	public static String getConfiguredCodecs()
	{
		// not in any particular order
		String info="";
		
		for (Enumeration e=codecClassNameMap.elements(); e.hasMoreElements(); )
        {  
			SSCodecInfo ci=(SSCodecInfo)e.nextElement();
			info+="CodecClassSuffix:"+ci.codecClassName+" internalName:"+ci.codecOfficialName+" frameSize:"+ci.frameSize+" inGain:"+ci.inGain+" outGain:"+ci.outGain+"\r\n";
        }
		return info;
	}
	
	
	public static SSCodec getCodec(SSCodecInfo codecInfo) throws Exception
	{
		    
        	String codecClassName="local.ua.sscodecs.SSCodec_"+codecInfo.codecClassName;
        	Class codecClass = null;
        	SSCodec sscodec = null;
        	try
        	{
        	  codecClass = Class.forName(codecClassName);
        	}
        	catch (ClassNotFoundException e)
        	{
                throw new IllegalArgumentException("Codec not found - "+codecClassName);

        	}
        	catch (Exception e)
        	{
        		throw new Exception(e); 
        	}

        	try
        	{
        		sscodec = (SSCodec)(codecClass.newInstance());
        	}
        	catch (Throwable e)
        	{
                throw new IllegalArgumentException("Codec won't instantiate - "+codecClassName);

        	}

        	boolean frameOK=false;
        	int[] vfs=sscodec.getValidFrameSizes();
            String frameSizeList="";
            for (int t=0;t<vfs.length;t++)
            {
            	if (vfs[t]==codecInfo.frameSize)
            	{
            		frameOK=true;
            		break;
            	}
            		
            	if (frameSizeList.length()>0)
            		frameSizeList+=",";

            	frameSizeList+=vfs[t];
            }
            
            if (!frameOK)
            	throw new IllegalArgumentException("Invalid frameSize: "+codecInfo.frameSize+" - supported sizes are: "+frameSizeList);
        	
        	try
        	{
                sscodec.init(codecInfo.frameSize,codecInfo.inGain,codecInfo.outGain);
        	}
        	catch (Throwable e)
        	{
                throw new IllegalArgumentException("Codec failed init - "+codecClassName);
        	}
            
        	return sscodec;
	}
	
	public static SSCodec getCodecByNumber(int codecNum) throws Exception
	{
	    SSCodecInfo codecInfo=codecNumberMap.get(codecNum);
	    SSCodec newCodec=getCodec(codecInfo);
	    if (newCodec!=null)
	    	newCodec.setPayloadType(codecNum);
	    return newCodec;
	}

	/*
	public static String getCodecNameFromNumber(int codecNum)
	{
		return codecActNameMap.get(codecNum);
	}
	*/

	public static SSCodec getDynamicCodec(int dynamicNum,String codecParms) throws Exception
	{
		String fixParms=codecParms.replaceAll("/1$","").toUpperCase();
	    SSCodecInfo codecInfo=codecDynamicMap.get(fixParms);
	    if (codecInfo==null)
	    	throw new Exception("Error locating codec: "+fixParms);
	    SSCodec newCodec=getCodec(codecInfo);
	    if (newCodec!=null)
	    	newCodec.setPayloadType(dynamicNum);
	    return newCodec;
	}

	public static void intToBytes32(int sample, byte buffer[], int byteOffset, boolean bigEndian)
    {
        if(bigEndian)
        {
            buffer[byteOffset++] = (byte)(sample >> 24);
            buffer[byteOffset++] = (byte)(sample >>> 16 & 0xff);
            buffer[byteOffset++] = (byte)(sample >>> 8 & 0xff);
            buffer[byteOffset] = (byte)(sample & 0xff);
        } else
        {
            buffer[byteOffset++] = (byte)(sample & 0xff);
            buffer[byteOffset++] = (byte)(sample >>> 8 & 0xff);
            buffer[byteOffset++] = (byte)(sample >>> 16 & 0xff);
            buffer[byteOffset] = (byte)(sample >> 24);
        }
    }

	/* test code 
	public static void main( String [] args) 
	{
		int codecBufOffset=12;
		byte[] encPcmBuf=new byte[640];
		byte[] decPcmBuf=new byte[640];
		int pcmbuflen=encPcmBuf.length;
		byte[] codecBuf=new byte[encPcmBuf.length/320*33+codecBufOffset];
		
		byte tb=0;
		for (int s=0;s<encPcmBuf.length;s++)
		{
			encPcmBuf[s]=tb++;
		}
		
		String codecName="GSMTRI";
		int frame_size=160;
		int payloadtype=3;
		
		configureCodec(codecName,frame_size);
		configureCodec("PCMU",320);
		configureCodec("PCMA",320);
		
		System.out.println("configured Codecs="+getConfiguredCodecs());
		
		System.out.println("inData:"+encPcmBuf[638]+","+encPcmBuf[639]);
		
		try
		{
	      SSCodec sscodec=SSCodecFactory.getCodecByNumber(payloadtype,frame_size,1,1);
	      
	      System.out.println("Testing codecName=" + sscodec.getCodecName());
	      
	      int encodedSize=codecBufOffset+sscodec.PcmToCodec(encPcmBuf, pcmbuflen, codecBuf, codecBufOffset);
	      
          System.out.println("encode worked. " + encodedSize);

	      int decodedSize=sscodec.CodecToPcm(codecBuf, codecBufOffset, encodedSize, decPcmBuf);
	      
          System.out.println("decode worked. " + decodedSize +" "+decPcmBuf[638]+","+decPcmBuf[639]);
		
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}

	}
	*/
}

