/*
 * Copyright (C) 2009 Greg Dorfuss - mhspot.com
 * 
 * SipToSis is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * SipToSis is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this source code; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 * Author(s):
 * Greg Dorfuss
 */

package local.ua.test;




import local.ua.ConnectorDebugWriter;
import local.ua.SkypeUA;


import com.skype.Call;
import com.skype.CallListener;
import com.skype.CallStatusChangedListener;
import com.skype.Skype;
import com.skype.SkypeException;
import com.skype.NotAttachedException;
import com.skype.connector.Connector;

import org.apache.log4j.Logger;
import org.apache.log4j.PropertyConfigurator;

public class SkypeConnectorReliabilityTest implements CallListener, CallStatusChangedListener
{           
	private Logger log=null;
    private String skypeUserId=null;
    private Call currentSkypeCall=null;
    private Call.Status curCallStatus=null;
    private long failcnt=0;

    
    /** The main method. */
	public static void main(String[] args) 
    {         
  	    PropertyConfigurator.configure("log.properties"); // needed only first time
    	new SkypeConnectorReliabilityTest();
    }    
	   	
   
   /** Costructs a UA with a default media port */
   public SkypeConnectorReliabilityTest()
   {
	   
	   if (init())
		   doTest();

	   cleanup();
	   System.exit(0);
   }

   public boolean init()
   {
	  log = Logger.getLogger(this.getClass().getName());
	  int bits=getVMBitSize();
	  log.info("Starting Connector Reliability Test ");
	  log.info("os="+System.getProperty("os.name")+" arch="+System.getProperty("os.arch")+" ver="+System.getProperty("os.version"));
	  log.info("javaVer="+System.getProperty("java.version")+" - "+System.getProperty("java.vendor")+" ("+bits+" bit)");
	  log.debug("javaLibPath="+System.getProperty("java.library.path"));
	  log.debug("javaClassPath="+System.getProperty("java.class.path"));

      if (bits==64 && System.getProperty("os.name").toLowerCase().contains("windows") && (System.getenv("JAVAEXEPATH")==null || System.getenv("JAVAEXEPATH").length()==0))
      {
    	  log.warn("*********************************************************");
    	  log.warn("* WARNING: JAVAEXEPATH not set - Required for 64 bit VM *");
    	  log.warn("*********************************************************");
      }    
      
	  try
	  {
	    skypeUserId=initSkype();
	    return true;
	  }
	  catch(Exception e)
	  {
		  log.error("initSkype failed",e);
	  }
	  return false;
   }


	private String initSkype()  throws Exception
	{
		log.info("initSkype - If stuck, check Skype online & API auth");

		
    	setupSkype();
    	
    	skypeUserId=Skype.getProfile().getId();
    	
    	log.info("Attached SkypeUserId:"+skypeUserId);

    	return skypeUserId;
	}

	private void cleanup()
	{
        Skype.removeCallListener(this);
	}


	private void setupSkype() throws SkypeException
	{
		int retrycnt=0;
		while (true)
		{
			try
			{

			    Connector.getInstance().setDebugOut(new ConnectorDebugWriter());
			    Connector.getInstance().setCommandTimeout(2000);
				Skype.setDebug(true);
				
				log.info("SkypeVer:"+Skype.getVersion());
				break;
			}
			catch (NotAttachedException e)
			{
				retrycnt++;
				if (retrycnt>15)
				{
					log.error("Connect attempt limit reached - exiting.",e);
					System.exit(SkypeUA.ExitCode.STARTERROR.code());
				}
				
				String sStatus=Connector.getInstance().getStatus().name();
			    log.info("Skype Status: "+sStatus+" - retrying every 5 seconds");
				try {Thread.sleep(5000);} catch(Exception te){}
			}
			catch (Throwable e)
			{
				log.fatal("skype4java connect error: ",e);
				System.exit(SkypeUA.ExitCode.SKYPE4JAVACONNECTERROR.code());
			}
		}
	  
		Skype.addCallListener(this);
	}
	
	


	public void callReceived(Call call) throws SkypeException
	{
		 	log.info("callReceived - incoming Skype Call from:"+call.getPartnerDisplayName()+" ["+call.getPartnerId()+"] status:"+call.getStatus());
	}
	
	public void callMaked(Call call) throws SkypeException
	{
		   log.info("callMaked: Skype "+call.getStatus()+" id="+call.getId());
		   this.currentSkypeCall.addCallStatusChangedListener(this);

	}
	
	
	public boolean lostSkypeConnection()
	{
		log.error("Lost Skype client connection - reconnecting.");
		
		// try to connect a few times
		int retries=0;
		while (retries++<5)
		{	
			try
			{
				skypeUserId=Skype.getProfile().getId();
				log.info("Reconnected to Skype client.");
				return true;
			}
			catch (NotAttachedException e)
			{
				try {Thread.sleep(3000);} catch(Exception te){}
			}
			catch (Throwable e)
			{
				break;
			}
	    }
		return false;
	}	
		



	  private static int getVMBitSize()
	  {
		  String dmBits = System.getProperty("sun.arch.data.model", "novalue");
		  if (dmBits.equals("32"))
			  return 32;
		  else if (dmBits.equals("64"))
		      return 64;
		  else if (dmBits.equals("novalue")) 
		  {
		      if (System.getProperty("java.vm.name").indexOf("64") >= 0)
		    	  return 64;
		  } 
  		  return 32;
	  }	  
	
	  
		/* when skype call status changes */
		public void statusChanged(Call.Status status) throws SkypeException
		{
			curCallStatus=status;
			if (status==Call.Status.FINISHED || status==Call.Status.CANCELLED || status==Call.Status.MISSED || status==Call.Status.REFUSED || status==Call.Status.FAILED || status==Call.Status.UNPLACED || status==Call.Status.BUSY)
			{
				// tear down the call
				log.info("skypeCallStatus - Complete: "+status);
				currentSkypeCall.removeCallStatusChangedListener(this);
				currentSkypeCall=null;
			}
			else if (status==Call.Status.EARLYMEDIA)
			{
				// skypeout earlymedia handling
				log.info("skypeCallStatus - "+status);
			}
			else if (status==Call.Status.INPROGRESS)
			{
					log.info("skypeCallStatus - "+status);
			}
			else if (status==Call.Status.LOCALHOLD)
			{
				log.info("skypeCallStatus - "+status);
			}
			else if (status==Call.Status.REMOTEHOLD)
			{
				log.info("skypeCallStatus - "+status);
			}
			else if (status==Call.Status.ROUTING || status==Call.Status.RINGING)
			{
				log.info("skypeCallStatus - "+status);
			}	
			else
			{	
				log.info("skypeCallStatus - "+status);
			}
			
		}
		
	  
		private boolean loadSkypeCallHistory()
		{
		       log.info("Loading Skype PSTN Call History");
		       String callsResp=local.ua.util.parseSkypeResponse("SEARCH CALLS","CALLS");
		       log.debug("CallsList:"+callsResp);
		       if (callsResp.trim().length()==0)
		       {
		    	   log.info("0 possible calls in history.");
		    	   return true;
		       }
		       String[] calls=callsResp.split(",");
		       log.info(calls.length+" calls in history.");

		       for (int x=0;x<calls.length;x++)
		       {
		         String curId=calls[x].trim();
		         if (curId.length()>0)
		         {
		        	 String callId;
		        	 Call.Type callType;
		        	 long durationSeconds;
		        	 String callTo;
		        	 long unixStamp=-1;
		             try
		             {
		               callId=local.ua.util.parseSkypeResponse("GET CALL "+curId+" TIMESTAMP","CALL "+curId+" TIMESTAMP").trim();
		               unixStamp=Long.parseLong(callId);
		             
			             if (unixStamp>0)
			             {   
		                     callType=Call.Type.valueOf(local.ua.util.parseSkypeResponse("GET CALL "+curId+" TYPE","CALL "+curId+" TYPE"));
		                     if (callType==Call.Type.OUTGOING_PSTN)
		                     {
		                    	 durationSeconds=Long.parseLong(local.ua.util.parseSkypeResponse("GET CALL "+curId+" DURATION","CALL "+curId+" DURATION"));
		                         callTo=local.ua.util.parseSkypeResponse("GET CALL "+curId+" PARTNER_HANDLE","CALL "+curId+" PARTNER_HANDLE");
		                     }
			             }
		             }
		             catch(Exception e)
		             {
		                 log.error("History Load Error: Skype CallId="+curId,e); 
		                 return false;
		             }
		         }         
		      }
              return true;
		}
		 		
		
		private boolean isSkypeCallClosed(Call.Status status)
		{
			if (status==Call.Status.MISSED || status==Call.Status.CANCELLED 
				|| status==Call.Status.FAILED || status==Call.Status.FINISHED 
				|| status==Call.Status.BUSY || status==Call.Status.REFUSED 
				|| status==Call.Status.UNPLACED)
				return true;
			else
				return false;
		}
		
	  private void doTest()
	  {

		  int maxTests=999999999;
		  int testcnt=0;
		  log.info("Testing - press a key to exit");
		  
		  while (!keyPressed() || testcnt>=maxTests)
		  {	  
			  testcnt++;
			  Pinger pinger=new Pinger();
			  pinger.start();
			  for (int x=0;x<5;x++)
			  {	  
				  if (!loadSkypeCallHistory())
				  {	  
					  log.error("FAILED HISTORY LOAD TEST");
					  failcnt++;
				  }
				  
				  try
				  {
				    log.info("ver:"+Skype.getVersion());
				  }
				  catch(Exception e)
				  {
					 log.error("FAILED",e); 
					 failcnt++;
				  }
			  }

			  /*
			  String dest="echo123";
			  if (!makeTestCall(dest))
				  log.error("FAILED TEST CALL TEST");
			  */
			  pinger.halt();
			  log.info("Complete Pass: "+testcnt+" failcnt="+failcnt);
			  
			  try
			  {
				  Thread.sleep(5000);
			  }
			  catch(Exception e)
			  {}
			  
			  
		  }
	  }
	
	  private boolean makeTestCall(String dest)
	  {
		  boolean retvar=false;
		  curCallStatus=null;
		  log.info("Calling: "+dest);
		  try
		  {
		     currentSkypeCall=Skype.call(dest);
		  }
		  catch(Exception e)
		  {
			  log.error("FAIL:",e);
		  }

		  if (currentSkypeCall!=null)
		  {
			  try
			  {
				  curCallStatus=currentSkypeCall.getStatus();
				  int wait=0;
				  int connectLimit=15;
				  while (wait++<connectLimit)
				  {
					  if (curCallStatus==Call.Status.INPROGRESS)
					  {	  
						  log.info("OnCall with:"+currentSkypeCall.getPartnerId()); 
						  int connwait=0;
						  while (curCallStatus==Call.Status.INPROGRESS && connwait++<10)
						  {	  
						     Thread.sleep(2000);
						  }
						  
					      if (curCallStatus==Call.Status.INPROGRESS)
					      {
					    	 currentSkypeCall.cancel();
					    	 int cancelcnt=0;
							 while (curCallStatus==Call.Status.INPROGRESS && cancelcnt++<2)
							 {	  
							     Thread.sleep(2000);
							 }
							 if (cancelcnt>=2)
							 {
								 log.info("Call did not end within time limit");
							 }
							 else
							 {	 
								 log.info("CallEnded");
								 currentSkypeCall=null;
						    	 retvar=true;
							 }
					    	 break;
					      }
					      else
							 log.info("Call was cut off prematurely");
					      
					  }
					  else
					      Thread.sleep(2000);
				  }
				  if (currentSkypeCall!=null)
					  currentSkypeCall.cancel();

			      if (wait>=connectLimit)
						 log.info("Call did not connect within time limit");

			  }
			  catch(Exception e)
			  {
				  log.error("Error",e);
			  }
		  }
		  return retvar;
	  }
	  
	  private boolean keyPressed()
	  {
		  boolean retvar=false;
		  try
		  {
			  if (System.in.available()>0)
				  retvar=true;
		  }
		  catch(Exception e)
		  {
		  }
		  return retvar;
	  }
	  
}


