//---------------------------------------------------------------------
//  Copyright (C) Microsoft Corporation.  All rights reserved.
// 
//THIS CODE AND INFORMATION ARE PROVIDED AS IS WITHOUT WARRANTY OF ANY
//KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
//IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
//PARTICULAR PURPOSE.
//---------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Data;
using System.Text;
using System.Windows.Forms;
using System.Media;
using System.IO;

namespace CountDownCSharp
{
    public partial class CDCountdownPanel : UserControl, IPanelNavigating
    {
        TimeSpan _countdownTime;
        SoundPlayer _player;
        bool _timerStarted = false;
        bool _timerExpired = false;
        bool _upButtonAcceleration = false;
        int _numberOfTicks = 0;
        int _oldMouseDownTimerInterval = 0;
        int[] _lastSelectedTime = new int[3];
        DateTime _startedOn;

        public CDCountdownPanel()
        {
            InitializeComponent();

            countdownStartButton.Click += new EventHandler(countdownStartButton_Click);
            upButton.MouseDown += new MouseEventHandler(upButton_MouseDown);
            upButton.MouseUp += new MouseEventHandler(upButton_MouseUp);
            downButton.MouseDown += new MouseEventHandler(downButton_MouseDown);
            downButton.MouseUp += new MouseEventHandler(downButton_MouseUp);
            mouseDownTimer.Tick += new EventHandler(mouseDownTimer_Tick);   
            timer1.Tick += new EventHandler(timer1_Tick);
            quickSetComboBox.SelectedIndexChanged += new EventHandler(quickSetComboBox_SelectedIndexChanged);
            pauseButton.Click += new EventHandler(pauseButton_Click);
        }

        private void countdownStartButton_Click(object sender, System.EventArgs e)
        {
            if ((!(_timerStarted) & !(_timerExpired)))
            {
                _lastSelectedTime[0] = System.Convert.ToInt32(hoursLabel.Text);
                _lastSelectedTime[1] = System.Convert.ToInt32(minutesLabel.Text);
                _lastSelectedTime[2] = System.Convert.ToInt32(secondsLabel.Text);
                _timerStarted = true;
                downButton.Enabled = false;
                upButton.Enabled = false;
                quickSetComboBox.Enabled = false;
                timeLabel.Enabled = false;
                quickSetLabel.Enabled = false;
                hoursLabel.ForeColor = Color.Green;
                minutesLabel.ForeColor = Color.Green;
                secondsLabel.ForeColor = Color.Green;
                pauseButton.Enabled = true;
                timer1.Enabled = true;
                _startedOn = DateTime.Now;
                countdownStartButton.Text = Properties.Resources.StopText;
            }
            else
            {
                FinishCountdown();
            }
        }

        private void FinishCountdown()
        {
            _timerStarted = false;
            if ((!(_timerExpired)))
            {
                timer1.Enabled = false;
            }
            else
            {
                if (!(_player == null) & System.Convert.ToBoolean(Properties.Settings.Default.LoopSoundPlayer))
                {
                    _player.Stop();
                }
                try
                {
                    My.Application.AddLogFileEntry(_startedOn, _lastSelectedTime[0], _lastSelectedTime[1], _lastSelectedTime[2]);
                }
                catch (FileNotFoundException ex)
                {
                    // TODO log error. 
                }
            }
            hoursLabel.ForeColor = Color.Black;
            minutesLabel.ForeColor = Color.Black;
            secondsLabel.ForeColor = Color.Black;
            _countdownTime = new TimeSpan(_lastSelectedTime[0], _lastSelectedTime[1], _lastSelectedTime[2]);
            UpdateTime();
            downButton.Enabled = true;
            upButton.Enabled = true;
            quickSetComboBox.Enabled = true;
            timeLabel.Enabled = true;
            quickSetLabel.Enabled = true;
            pauseButton.Enabled = false;
            countdownStartButton.Text = Properties.Resources.StartText;
            _timerExpired = false;
        }

        private void CDCountdownPanel_Load(object sender, EventArgs e)
        {
            _countdownTime = new TimeSpan(0, 20, 0);
            quickSetComboBox.SelectedIndex = 0; 
        }

        private void AddTime(int minuteIncrement)
        {
            if ((_countdownTime.Minutes > 1))
            {
                downButton.Enabled = true;
            }
            _countdownTime = _countdownTime.Add(new TimeSpan(0, minuteIncrement, 0));
            quickSetComboBox.Text = "Custom";
            UpdateTime();
        }

        private void SubtractTime(int minuteDecrement)
        {
            if ((_countdownTime.Minutes == 1 & _countdownTime.Hours == 0))
            {
                downButton.Enabled = false;
                if ((mouseDownTimer.Enabled))
                {
                    ResetMouseDownTimer();
                }
            }
            else
            {
                _countdownTime = _countdownTime.Subtract(new TimeSpan(0, minuteDecrement, 0));
                quickSetComboBox.Text = "Custom";
                UpdateTime();
            }
        }

        private void upButton_MouseDown(object sender, MouseEventArgs e)
        {
            AddTime(1);
            _upButtonAcceleration = true;
            _numberOfTicks = 0;
            _oldMouseDownTimerInterval = mouseDownTimer.Interval;
            mouseDownTimer.Start();
        }

        private void mouseDownTimer_Tick(object sender, System.EventArgs e)
        {
            if ((_upButtonAcceleration))
            {
                AddTime(1);
            }
            else
            {
                SubtractTime(1);
            }
            if ((_numberOfTicks == 2 | _numberOfTicks == 6 | _numberOfTicks == 9))
            {
                mouseDownTimer.Interval = mouseDownTimer.Interval / 2;
            }
            _numberOfTicks += 1;
        }

        private void upButton_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            ResetMouseDownTimer();
        }

        private void ResetMouseDownTimer()
        {
            _numberOfTicks = 0;
            mouseDownTimer.Interval = _oldMouseDownTimerInterval;
            mouseDownTimer.Stop();
        }

        private void timer1_Tick(object sender, System.EventArgs e)
        {
            if ((TimeSpan.Equals(_countdownTime, new TimeSpan(0, 0, 0))))
            {
                _timerExpired = true;
                timer1.Stop();
                hoursLabel.ForeColor = Color.Black;
                minutesLabel.ForeColor = Color.Black;
                secondsLabel.ForeColor = Color.Black;
                _player = new SoundPlayer(System.Convert.ToString(Properties.Settings.Default.SoundFile));
                if (System.Convert.ToBoolean(Properties.Settings.Default.LoopSoundPlayer))
                {
                    _player.PlayLooping();
                }
                else
                {
                    _player.Play();
                    FinishCountdown();
                }
            }
            else
            {
                _countdownTime = _countdownTime.Subtract(new TimeSpan(0, 0, 1));
                if ((_countdownTime.Hours == 0 & _countdownTime.Minutes == 0))
                {
                    hoursLabel.ForeColor = Color.Red;
                    minutesLabel.ForeColor = Color.Red;
                    secondsLabel.ForeColor = Color.Red;
                }
                UpdateTime();
            }
        }

        private void downButton_MouseDown(object sender, MouseEventArgs e)
        {
            SubtractTime(1);
            _upButtonAcceleration = false;
            _numberOfTicks = 0;
            _oldMouseDownTimerInterval = mouseDownTimer.Interval;
            mouseDownTimer.Start();
        }

        private void downButton_MouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
        {
            ResetMouseDownTimer();
        }

        private void UpdateTime()
        {
            hoursLabel.Text = _countdownTime.Hours.ToString("00");
            minutesLabel.Text = _countdownTime.Minutes.ToString("00");
            secondsLabel.Text = _countdownTime.Seconds.ToString("00");
        }

        private void quickSetComboBox_SelectedIndexChanged(object sender, System.EventArgs e)
        {
            string _quickSetTime = quickSetComboBox.SelectedItem.ToString();
            if (!(_quickSetTime == Properties.Resources.CustomText))
            {
                _quickSetTime = _quickSetTime.Substring(0, _quickSetTime.IndexOf(" "));
                _countdownTime = new TimeSpan(0, System.Convert.ToInt32(_quickSetTime), 0);
                UpdateTime();
                downButton.Enabled = true;
            }
        }

        public bool CanNavigate()
        {
            if (_timerStarted)
            {
                DialogResult _dr = MessageBox.Show(Properties.Resources.CountdownStillRunningText, Properties.Resources.CountdownRunningHeaderText, MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button1);
                if (_dr == DialogResult.Yes)
                {
                    FinishCountdown();
                    return (true);
                }
                else
                {
                    return (false);
                }
            }
            else
            {
                return (true);
            }
        }

        private void pauseButton_Click(object sender, System.EventArgs e)
        {
            if (pauseButton.Tag == null)
            {
                countdownStartButton.Enabled = false;
                timer1.Stop();
                pauseButton.Tag = Properties.Resources.PausedText;
                pauseButton.Text = Properties.Resources.PauseButtonUnpauseText;
            }
            else
            {
                pauseButton.Tag = null;
                pauseButton.Text = Properties.Resources.PauseButtonPausedText;
                countdownStartButton.Enabled = true;
                timer1.Start();
            }
        }
    }
}
