'---------------------------------------------------------------------
'  Copyright (C) Microsoft Corporation.  All rights reserved.

'THIS CODE AND INFORMATION ARE PROVIDED AS IS WITHOUT WARRANTY OF ANY
'KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
'IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A
'PARTICULAR PURPOSE.
'---------------------------------------------------------------------

Imports System.DateTime
Imports System.Media
Imports System.IO

Public Class CDCountdownPanel
    Implements IPanelNavigating

    ' The meditation interval.
    Dim _CountdownTime As TimeSpan
    Dim _Player As SoundPlayer
    Dim _TimerStarted As Boolean = False
    Dim _TimerExpired As Boolean = False
    Dim _UpButtonAcceleration As Boolean = False
    Dim _NumberOfTicks As Integer = 0
    Dim _OldMouseDownTimerInterval As Integer = 0

    ' For countdown logging.
    Dim _LastSelectedTime(3) As Integer
    Dim _StartedOn As DateTime

    Private Sub CountdownStartButton_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles CountdownStartButton.Click
        If (Not _TimerStarted And Not _TimerExpired) Then
            ' Starting the timer. Start the clock.
            _LastSelectedTime(0) = CInt(HoursLabel.Text)
            _LastSelectedTime(1) = CInt(MinutesLabel.Text)
            _LastSelectedTime(2) = CInt(SecondsLabel.Text)

            _TimerStarted = True
            DownButton.Enabled = False
            UpButton.Enabled = False
            QuickSetComboBox.Enabled = False
            TimeLabel.Enabled = False
            QuickSetLabel.Enabled = False

            ' Set time to green while counting down.
            HoursLabel.ForeColor = Color.Green
            MinutesLabel.ForeColor = Color.Green
            SecondsLabel.ForeColor = Color.Green

            PauseButton.Enabled = True
            Timer1.Enabled = True
            _StartedOn = DateTime.Now
            CountdownStartButton.Text = "Stop!"
        Else
            FinishCountdown()
        End If
    End Sub

    Private Sub FinishCountdown()
        ' Could be in one of two states here:
        ' (1) Interrupting meditation before it's over;
        ' (2) Meditation is over, so turn off the alarm.
        _TimerStarted = False

        ' Log the time.
        If (Not _TimerExpired) Then
            ' The user stopped the countdown early.
            Timer1.Enabled = False
        Else
            '  TODO: Use My.Computer.Audio here.
            If (_Player IsNot Nothing And CBool(My.Settings("LoopSoundPlayer"))) Then
                _Player.Stop()
            End If

            ' Log the countdown time.
            Try
                My.Application.AddLogFileEntry(_StartedOn, _LastSelectedTime(0), _LastSelectedTime(1), _LastSelectedTime(2))
            Catch ex As Exception
                ' For now, log any failures silently. 
                ' TODO Incorporate an alert into the UI.
                My.Application.Log.WriteException(ex)
            End Try
        End If

        HoursLabel.ForeColor = Color.Black
        MinutesLabel.ForeColor = Color.Black
        SecondsLabel.ForeColor = Color.Black

        ' Revert to the previously selected time.
        _CountdownTime = New TimeSpan(_LastSelectedTime(0), _LastSelectedTime(1), _LastSelectedTime(2))
        UpdateTime()

        DownButton.Enabled = True
        UpButton.Enabled = True
        QuickSetComboBox.Enabled = True
        TimeLabel.Enabled = True
        QuickSetLabel.Enabled = True
        PauseButton.Enabled = False

        CountdownStartButton.Text = "Start!"

        _TimerExpired = False
    End Sub

    Private Sub CDCountdownPanel_Load(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MyBase.Load
        _CountdownTime = New TimeSpan(0, 20, 0)
        QuickSetComboBox.SelectedIndex = 0
    End Sub

    Private Sub AddTime(ByVal minuteIncrement As Integer)
        If (_CountdownTime.Minutes > 1) Then
            DownButton.Enabled = True
        End If

        _CountdownTime = _CountdownTime.Add(New TimeSpan(0, minuteIncrement, 0))
        QuickSetComboBox.Text = "Custom"
        UpdateTime()
    End Sub

    Private Sub SubtractTime(ByVal minuteDecrement As Integer)
        ' Minimum time is one minute.
        '  TODO: May want to make this more granular, to drop it down to seconds if it goes under a minute.
        ' Most people won't need this, though.
        If (_CountdownTime.Minutes = 1 And _CountdownTime.Hours = 0) Then
            DownButton.Enabled = False
            ' If mouse down timer is running, disable it.
            If (MouseDownTimer.Enabled) Then
                ResetMouseDownTimer()
            End If
        Else
            _CountdownTime = _CountdownTime.Subtract(New TimeSpan(0, minuteDecrement, 0))
            QuickSetComboBox.Text = "Custom"
            UpdateTime()
        End If
    End Sub

    ' Handle this in MouseDown instead of Click to enable scrolling behavior.
    Private Sub UpButton_MouseDown(ByVal sender As System.Object, ByVal e As MouseEventArgs) Handles UpButton.MouseDown
        AddTime(1)

        ' Set mouse down timer, and auto-increment the time upwards if the user keeps the mouse down.
        _UpButtonAcceleration = True
        _NumberOfTicks = 0
        _OldMouseDownTimerInterval = MouseDownTimer.Interval
        MouseDownTimer.Start()
    End Sub

    ' This timer will handle both mouse button up and mouse button down.
    Private Sub MouseDownTimer_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles MouseDownTimer.Tick
        If (_UpButtonAcceleration) Then
            ' Go up, not down.
            AddTime(1)
        Else
            ' Down, not up.
            SubtractTime(1)
        End If

        If (_NumberOfTicks = 2 Or _NumberOfTicks = 6 Or _NumberOfTicks = 9) Then
            MouseDownTimer.Interval = MouseDownTimer.Interval / 2
        End If

        _NumberOfTicks += 1
    End Sub

    Private Sub UpButton_MouseUp(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles UpButton.MouseUp
        ResetMouseDownTimer()
    End Sub

    Private Sub ResetMouseDownTimer()
        _NumberOfTicks = 0
        MouseDownTimer.Interval = _OldMouseDownTimerInterval
        MouseDownTimer.Stop()
    End Sub

    ' Subtract one second from the count, and update the display.
    Private Sub Timer1_Tick(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles Timer1.Tick
        If (TimeSpan.Equals(_CountdownTime, New TimeSpan(0, 0, 0))) Then
            ' Time is up - sound the alarm.
            _TimerExpired = True
            Timer1.Stop()

            HoursLabel.ForeColor = Color.Black
            MinutesLabel.ForeColor = Color.Black
            SecondsLabel.ForeColor = Color.Black

            _Player = New SoundPlayer(CStr(My.Settings("SoundFile")))
            If (CBool(My.Settings("LoopSoundPlayer"))) Then
                ' Don't call FinishCountdown() here - wait for the user to stop the sound
                ' by clicking the Stop button.
                _Player.PlayLooping()
            Else
                _Player.Play()
                FinishCountdown()
            End If
        Else
            _CountdownTime = _CountdownTime.Subtract(New TimeSpan(0, 0, 1))
            If (_CountdownTime.Hours = 0 And _CountdownTime.Minutes = 0) Then
                HoursLabel.ForeColor = Color.Red
                MinutesLabel.ForeColor = Color.Red
                SecondsLabel.ForeColor = Color.Red
            End If
            UpdateTime()
        End If
    End Sub

    Private Sub DownButton_MouseDown(ByVal sender As System.Object, ByVal e As MouseEventArgs) Handles DownButton.MouseDown
        SubtractTime(1)

        _UpButtonAcceleration = False
        _NumberOfTicks = 0
        _OldMouseDownTimerInterval = MouseDownTimer.Interval
        MouseDownTimer.Start()
    End Sub

    Private Sub DownButton_MouseUp(ByVal sender As Object, ByVal e As System.Windows.Forms.MouseEventArgs) Handles DownButton.MouseUp
        ResetMouseDownTimer()
    End Sub

    Private Sub UpdateTime()
        HoursLabel.Text = _CountdownTime.Hours.ToString("00")
        MinutesLabel.Text = _CountdownTime.Minutes.ToString("00")
        SecondsLabel.Text = _CountdownTime.Seconds.ToString("00")
    End Sub

    Private Sub QuickSetComboBox_SelectedIndexChanged(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles QuickSetComboBox.SelectedIndexChanged
        Dim _QuickSetTime As String = QuickSetComboBox.SelectedItem.ToString()
        If (Not _QuickSetTime = "Custom") Then
            _QuickSetTime = _QuickSetTime.Substring(0, _QuickSetTime.IndexOf(" "))
            _CountdownTime = New TimeSpan(0, CInt(_QuickSetTime), 0)
            UpdateTime()
            DownButton.Enabled = True
        End If
    End Sub

    Public Function CanNavigate() As Boolean Implements IPanelNavigating.CanNavigate
        If (_TimerStarted) Then
            Dim _Dr As DialogResult = MessageBox.Show("The countdown clock is still running. Do you want to stop?", _
                "Countdown in Progress", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button1)
            If (_Dr = DialogResult.Yes) Then
                ' Cancel the countdown.
                FinishCountdown()
                CanNavigate = True
            Else
                ' Cancel the panel navigation.
                CanNavigate = False
            End If
        Else
            CanNavigate = True
        End If
    End Function

    Private Sub PauseButton_Click(ByVal sender As System.Object, ByVal e As System.EventArgs) Handles PauseButton.Click
        If (PauseButton.Tag Is Nothing) Then
            CountdownStartButton.Enabled = False
            Timer1.Stop()
            PauseButton.Tag = "Paused"
            PauseButton.Text = My.Resources.PauseButtonUnpauseText
        Else
            PauseButton.Tag = Nothing
            PauseButton.Text = My.Resources.PauseButtonPausedText
            CountdownStartButton.Enabled = True
            Timer1.Start()
        End If
    End Sub
End Class
