/*
 *                 Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License
 * Version 1.0 (the "License"). You may not use this file except in
 * compliance with the License. A copy of the License is available at
 * http://www.sun.com/
 *
 * The Original Code is NetBeans. The Initial Developer of the Original
 * Code is Sun Microsystems, Inc. Portions Copyright 1997-2000 Sun
 * Microsystems, Inc. All Rights Reserved.
 */
package org.netbeans.lib.cvsclient.file;

import java.util.*;

/**
 * @author Thomas Singer
 */
public final class FileStatus {

	// Constants ==============================================================

	private static final Map<String, FileStatus> nameToStatus = new HashMap<String, FileStatus>();

	/**
	 * The Added status, i.e. the file has been added to the repository
	 * but not committed yet.
	 */
	public static final FileStatus ADDED = new FileStatus("Locally Added", LocalFileStatus.ADDED, RemoteFileStatus.UP_TO_DATE);

	/**
	 * The Removed status, i.e. the file has been removed from the repository
	 * but not committed yet
	 */
	public static final FileStatus REMOVED = new FileStatus("Locally Removed", LocalFileStatus.REMOVED, RemoteFileStatus.UP_TO_DATE);

	/**
	 * The file is locally modified, but no new revision is available in the repository.
	 */
	public static final FileStatus MODIFIED = new FileStatus("Locally Modified", LocalFileStatus.MODIFIED, RemoteFileStatus.UP_TO_DATE);

	/**
	 * The file is not modified and no new revision is available in the repository.
	 */
	public static final FileStatus UP_TO_DATE = new FileStatus("Up-to-date", LocalFileStatus.UP_TO_DATE, RemoteFileStatus.UP_TO_DATE);

	/**
	 * The file is locally not available (eg physically deleted), but available in the repository.
	 */
	public static final FileStatus NEEDS_CHECKOUT = new FileStatus("Needs Checkout", LocalFileStatus.NOT_EXISTING, RemoteFileStatus.UP_TO_DATE);

	/**
	 * The file is not modified, but a new revision available in the repository.
	 */
	public static final FileStatus NEEDS_PATCH = new FileStatus("Needs Patch", LocalFileStatus.UP_TO_DATE, RemoteFileStatus.NEW_REVISION_AVAILABLE);

	/**
	 * The file is locally modified and a new revision in the repository avaible.
	 */
	public static final FileStatus NEEDS_MERGE = new FileStatus("Needs Merge", LocalFileStatus.MODIFIED, RemoteFileStatus.NEW_REVISION_AVAILABLE);

	public static final FileStatus HAS_CONFLICTS = new FileStatus("File had conflicts on merge", LocalFileStatus.MODIFIED, RemoteFileStatus.NEW_REVISION_AVAILABLE);

	/**
	 * The file is locally modified, but removed in the repository.
	 */
	public static final FileStatus UNRESOLVED_CONFLICT = new FileStatus("Unresolved Conflict", LocalFileStatus.MODIFIED, RemoteFileStatus.REMOVED);

	/**
	 * The file is not modified, but removed in the repository.
	 */
	public static final FileStatus ENTRY_INVALID = new FileStatus("Entry Invalid", LocalFileStatus.UP_TO_DATE, RemoteFileStatus.REMOVED);

	/**
	 * The unknown status, i.e. the file is not known to the CVS repository.
	 */
	public static final FileStatus UNKNOWN = new FileStatus("Unknown", null, null);

	// Static =================================================================

	/**
	 * Returns the corresponding FileStatus constant for the specified String.
	 */
	public static FileStatus getStatusForString(String statusString) {
		return nameToStatus.get(statusString);
	}

	// Fields =================================================================

	private final String statusString;
	private final LocalFileStatus localFileStatus;
	private final RemoteFileStatus remoteFileStatus;

	// Setup ==================================================================

	private FileStatus(String statusString, LocalFileStatus localFileStatus, RemoteFileStatus remoteFileStatus) {
		this.statusString = statusString;
		this.localFileStatus = localFileStatus;
		this.remoteFileStatus = remoteFileStatus;

		nameToStatus.put(statusString, this);
	}

	// Implemented ============================================================

	/**
	 * Returns the String representation for thiz.
	 */
	@Override
	public String toString() {
		return statusString;
	}

	// Accessing ==============================================================

	public LocalFileStatus getLocalFileStatus() {
		return localFileStatus;
	}

	public RemoteFileStatus getRemoteFileStatus() {
		return remoteFileStatus;
	}
}