unit Jobs;

interface

uses Classes;

const c_CopyOK = 99999;
      c_NoRoom = 'I/O error 112';
      c_SecsPerDay = 86400;
      smOlder = 0; {Spec modes}
      smSize = 1;
      smList = 2;
      smAlways = 3;
      smNever = 4;

type TJobSpec = record
       SrcRoot, {original, unchanging source folder}
       Src,     {path for current Spec.Spec}
       Spec,    {filename}
       Trg,     {base target folder}
       TrgRoot, {optional new root to stem from Trg base}
       TrgDir,  {new Trg + NewRoot + SrcFrag\}
       SrcFile, {fully expanded}
       TrgFile  {fully expanded}
        : string;
       Sub : boolean;  {include subdirs}
       Mode : integer; {overwrite mode}
       Size : comp;    {collect once}
     end;

type TJob = class
       LastIO,Tempfile : string;
       dtStart,dtUpWait : TDateTime;
       bErr,bCancel : boolean;
       nFileTotal,nBestSpeed : integer;
       nGrandBytes,nSkipBytes,
       nCopyBytes,nOkBytes : comp;
       slOk,slSkip,slErr,
       slRecover,slPrompt,slOther,
       slSrc : TStringList;
       Spec : TJobSpec;
       txtJob : textfile;
       constructor Create; overload;
       destructor Destroy; override;
       procedure ExpandFiles;
       procedure ReadSpec;
       procedure WriteSpec;
       procedure MkTrgDir;
       procedure SetTrg;
       function MD5FileMatch(File1,File2 : string) : boolean;
       function MkCopy : boolean;
       function NewRoot : string;
       function SkipTest : boolean;
       function WriteLog : string;
       procedure Start;
       procedure DoJob;
       procedure DelIfNotInSource;
       procedure DelEmptyDirs(sRoot : string);
       procedure AfterCopy;
       procedure Stop;
     end;

function CleanPath(sPath : string) : string;

implementation

{$WARN SYMBOL_PLATFORM OFF}

uses VPAExt,Main,EditProps,SkipList,MD5,hash,Preview,
     Windows,StrUtils,SysUtils,ComCtrls;

function CleanPath(sPath : string) : string;

var v : string;

begin
  if copy(sPath,1,2) = '\\' then
  begin
    v := '\\';
    System.Delete(sPath,1,2);
  end
  else v := '';

  while pos('\\',sPath) > 0 do
    System.Delete(sPath,pos('\\',sPath),1);

  Result := v + sPath;
end;

constructor TJob.Create;
begin
  inherited;
  bCancel := false;
  slOk := TStringList.Create;
  slSkip := TStringList.Create;
  slErr := TStringList.Create;
  slRecover := TStringList.Create;
  slPrompt := TStringList.Create;
  slOther := TStringList.Create;
  slSrc := TStringList.Create;
  slSrc.Sorted := true;
end;

destructor TJob.Destroy;
begin
  slOk.Free;
  slSkip.Free;
  slErr.Free;
  slRecover.Free;
  slPrompt.Free;
  slOther.Free;
  slSrc.Free;
  inherited Destroy;
end;

procedure TJob.DelIfNotInSource;

{$I+}
procedure DelUnmatchedProcess(DirName : string);

var sr : TSearchRec;
    f  : integer;
    s  : string;

begin
  try
    if (bCancel) or (not DirectoryExists(DirName)) then exit;
    ChDir(DirName);
  except
    begin
      slOther.Add('Access error - ' + DirName);
      exit;
    end;
  end;

  frmMain.pnlFolder.Caption := DirName;
  frmMain.Yield;

  try
    f := FindFirst('*.*',faDirectory,sr);
  except on e : exception do
    begin
      slOther.Add('FindFirst() - ' + e.Message);
      try SysUtils.FindClose(sr); except end;
      exit;
    end;
  end;

  while f = 0 do
  begin
    if (sr.name <> '.') AND (sr.name <> '..') AND
      ((sr.attr AND faDirectory) <> 0) then
    begin
      if DirName[length(DirName)] = '\' then
        DelUnmatchedProcess(DirName + sr.Name)
      else
        DelUnmatchedProcess(DirName + '\' + sr.Name);

      try
        ChDir(DirName);
      except
        begin
          slOther.Add('Access error - ' + DirName);
          try SysUtils.FindClose(sr); except end;
          exit;
        end;
      end;
    end;

    try
      f := FindNext(sr);
    except on e : exception do
      begin
        slOther.Add('FindNext() - ' + e.Message);
        try SysUtils.FindClose(sr); except end;
        exit;
      end;
    end;
  end;

  try SysUtils.FindClose(sr); except end;

  try
    f := FindFirst('*.*',faArchive,sr);
  except on e : exception do
    begin
      slOther.Add('FindFirst()2 - ' + e.Message);
      try SysUtils.FindClose(sr); except end;
      exit;
    end;
  end;

  while f = 0 do
  begin
    if bCancel then break;

    if (sr.name <> '.') AND (sr.name <> '..') then
    begin
      s := FinalBS(DirName) + sr.Name;
      frmMain.Yield;

      if slSrc.IndexOf(s) = -1 then
      begin
        if DeleteFile(s) then
          slOther.Add('Sync Ok: unmatched file deleted: ' + s)
        else
          slOther.Add('Sync Error: unable to delete unmatched file: ' + s);
      end;

      try
        f := FindNext(sr);
      except on e : exception do
        begin
          slOther.Add('FindNext() - ' + e.Message);
          try SysUtils.FindClose(sr); except end;
          exit;
        end;
      end;
    end;
  end;

  try SysUtils.FindClose(sr); except end;
end;

begin  {DelIfNotInSource() [Sync / Mirror]}
  {files now, dirs later}
  frmMain.pnlFile.Caption := 'Removing unmatched files...';
  DelUnmatchedProcess(CleanPath(Spec.Trg + NewRoot));
end;

function TJob.NewRoot : string;
begin
  if Spec.TrgRoot <> '' then
    Result := NoFinalBS(Spec.TrgRoot)
  else
    Result := Spec.TrgRoot;
end;

function TJob.SkipTest : boolean;

var nSkip : integer;

begin
  Result := false;

  for nSkip := 0 to frmSkipList.lvSkip.Items.Count - 1 do
  begin
    if pos(frmSkipList.lvSkip.Items[nSkip].Caption,Spec.SrcFile) > 0 then
    begin
      Result := true;
      break;
    end;
  end;
end;

{$I+}
procedure TJob.MkTrgDir;

var n,nRootEnd : integer;
    s,sRoot : string;

procedure MkDir2;
begin
  try
    if not DirectoryExists(sRoot + s) then
      MkDir(sRoot + s);
  except
    slOther.Add('Creation error - ' + sRoot + s);
  end;
end;

begin {MkTrgDir}
  if (SkipTest) or (frmMain.chkNoCopy.Checked) then exit;
  s := '';

  if pos('\\',Spec.TrgDir) > 0 then
  begin
    sRoot := copy(Spec.TrgDir,1,posex('\',Spec.TrgDir,3));
    nRootEnd := length(sRoot) + 1;
  end
  else
  begin
    sRoot := copy(Spec.TrgDir,1,3);
    nRootEnd := 4;
  end;

  for n := nRootEnd to length(Spec.TrgDir) do
  begin
    if Spec.TrgDir[n] = '\' then
    begin
      MkDir2;
      s := s + '\';
    end
    else s := s + Spec.TrgDir[n];
  end;

  MkDir2;
end;

procedure TJob.DelEmptyDirs(sRoot : string);

{$I+}
procedure DelEmptyProcess(DirName : string);

var sr : TSearchRec;
    f  : integer;

begin
  try
    if (bCancel) or (not DirectoryExists(DirName)) then exit;
    ChDir(DirName);
  except
    begin
      slOther.Add('Access error - ' + DirName);
      exit;
    end;
  end;

  frmMain.pnlFolder.Caption := DirName;
  frmMain.Yield;

  try
    f := FindFirst('*.*',faDirectory,sr);
  except on e : exception do
    begin
      slOther.Add('FindFirst() - ' + e.Message);
      try SysUtils.FindClose(sr); except end;
      exit;
    end;
  end;

  while f = 0 do
  begin
    if (sr.name <> '.') AND (sr.name <> '..') AND
      ((sr.attr AND faDirectory) <> 0) then
    begin
      if DirName[length(DirName)] = '\' then
        DelEmptyProcess(DirName + sr.Name)
      else
        DelEmptyProcess(DirName + '\' + sr.Name);

      try
        ChDir(DirName);
      except
        begin
          slOther.Add('Access error - ' + DirName);
          try SysUtils.FindClose(sr); except end;
          exit;
        end;
      end;
    end;

    try
      f := FindNext(sr);
    except on e : exception do
      begin
        slOther.Add('FindNext() - ' + e.Message);
        try SysUtils.FindClose(sr); except end;
        exit;
      end;
    end;
  end;

  try SysUtils.FindClose(sr); except end;

  if length(DirName) > 3 then
  begin
    if DirectoryEmpty(DirName) then
    begin
      try {to be somewhere else}
        if pos('\\',Spec.Src) = 0 then
          ChDir(copy(Spec.Src,1,3))
        else
          ChDir('c:\');
      except
      end;

      try
        RmDir(DirName);
        slOther.Add('Removed empty folder - ' + DirName);
      except on e : exception do
        slOther.Add('Error removing empty folder - ' +
         DirName + ' - ' + e.Message);
      end;
    end;
  end;
end;

begin  {DelEmptyDirs()}
  frmMain.pnlFile.Caption := 'Removing empty folders...';
  DelEmptyProcess(sRoot);
end;

{$I+}
function TJob.WriteLog : string;

var txt : textfile;
    n : integer;
    sID : string;

function ConfigFile : string;
begin
  if frmMain.cfg = '' then
    Result := 'console'
  else
    Result := frmMain.cfg;
end;

function Errs : string;
begin
  if slErr.Count > 0 then
    Result := ' - ERRORS REPORTED'
  else
    Result := ' ' + c_NoErr;
end;

function SessionSettings : string;

var j : integer;
    i : TListItem;

function RetrySetting : string;
begin
  if not frmMain.chkRetry.Checked then
    Result := 'disabled'
  else if frmMain.chkRetryForever.Checked then
    Result := 'retry indefinately'
  else
    Result := 'retry ' + IntToStr(frmMain.spnRetry.Value) +
     ' times with ' + IntToStr(frmMain.spnRetryInt.Value) +
     's interval';
end;

begin {SessionSettings()}
  with frmMain do Result :=
   'Session Settings' + CRLF +
   '----------------' + CRLF +
   'Running Config      ' + ConfigFile + Errs + CRLF +
   'Retry Setting       ' + RetrySetting + CRLF +
   'MD5 Verification    ' + YN(chkMD5.Checked) + CRLF +
   'No-Copy             ' + YN(chkNoCopy.Checked) + CRLF +
   'Mirror              ' + YN(chkDelTrg.Checked) + CRLF +
   'Delete Empty Dirs   ' + YN(chkDelEmpty.Checked) + CRLF;

  Result := Result + CRLF + CRLF + 'Copy Specifications' +
   CRLF + '-------------------' + CRLF;

  for j := 0 to frmMain.lv.Items.Count - 1 do
  begin
    i := frmMain.lv.Items[j];

    Result := Result +
     'Source Folder : ' + i.Caption + CRLF +
     'FileSpec      : ' + i.SubItems[c_Spec] + CRLF +
     'Subdirs       : ' + i.SubItems[c_Sub] + CRLF +
     'Target Folder : ' + i.SubItems[c_Trg] + CRLF +
     'Target Root   : ' + i.SubItems[c_TrgRoot] + CRLF +
     'Overwrite     : ' + i.SubItems[c_Mode] + CRLF +
     'Size          : ' + i.SubItems[c_Size] + CRLF +
     '     ****' + CRLF;
  end;

  if frmSkipList.lvSkip.Items.Count > 0 then
  begin
    Result := Result + CRLF + 'Skip Specifications' +
     CRLF + '-------------------' + CRLF;

    for j := 0 to frmSkipList.lvSkip.Items.Count - 1 do
    begin
      i := frmSkipList.lvSkip.Items[j];
      Result := Result + i.Caption + CRLF;
    end;
  end;
end;

begin
  Result := UniqueID;

  repeat
    sID := UniqueID;
  until sID <> Result;

  try
    Result := UDir + 'Logs\' + sID + '_' + ConfigFile + Errs + '.txt';
    AssignFile(txt,Result);

    try
      Rewrite(txt);
      WriteLn(txt,c_App + ' ' + c_Ver + ' ' + c_Copyright);
      WriteLn(txt,c_URL);
      WriteLn(txt,CRLF + 'Session Log');
      WriteLn(txt,'-----------' + CRLF);

      if frmMain.chkNoCopy.Checked then
      begin
        WriteLn(txt,'[--------------------------------------------------------]');
        WriteLn(txt,'[ NOTE - no actual data was transferred (NoCopy enabled) ]');
        WriteLn(txt,'[--------------------------------------------------------]' + CRLF);
      end;

      WriteLn(txt,'Session started at ' +
       DateTimeToStr(dtStart) + ' as ' +
       GetCurrentUserName + CRLF);

      WriteLn(txt,SessionSettings);

      WriteLn(txt,'Total Source Files      : ' +
       PadL(CommaStr(FloatToStrF(nFileTotal,ffFixed,10,0)),10) +
       ' (' + FileSizeText(nGrandBytes) + ')');

      WriteLn(txt,'Total Files Copied Ok   : ' +
       PadL(CommaStr(IntToStr(slOk.Count)),10) +
       ' (' + FileSizeText(nOkBytes) + ')');

      WriteLn(txt,'Total Files Skipped     : ' +
       PadL(CommaStr(IntToStr(slSkip.Count)),10) +
       ' (' + FileSizeText(nSkipBytes) + ')');

      WriteLn(txt,'Total Files Errored     : ' + PadL(CommaStr(IntToStr(slErr.Count)),10));
      WriteLn(txt,'Total Errors Recovered  : ' + PadL(CommaStr(IntToStr(slRecover.Count)),10));
      frmMain.UpdateStatus; {set correct pnlElapsed.Caption}
      WriteLn(txt,'Total Time Elapsed      : ' + frmMain.pnlElapsed.Caption);
      WriteLn(txt,'Best Transfer Speed     : ' + CommaStr(IntToStr(nBestSpeed)) + ' MB/sec');

      if bCancel then
        WriteLn(txt,CRLF + '  --- CANCELLED BY USER ---');

      WriteLn(txt,CRLF + 'Session ended at ' + DateTimeToStr(Now));
      WriteLn(txt,CRLF + CRLF + 'Search for these text tags to navigate the log:');
      WriteLn(txt,CRLF + '_OK  _SKIPPED  _ERROR  _RECOVERED  _OTHER' + CRLF);
      WriteLn(txt,CRLF + CRLF + '_OK' + CRLF);

      for n := 0 to slOk.Count - 1 do
        WriteLn(txt,slOk[n] + CRLF);

      WriteLn(txt,CRLF + '_SKIPPED' + CRLF);

      for n := 0 to slSkip.Count - 1 do
        WriteLn(txt,slSkip[n] + CRLF);

      WriteLn(txt,CRLF + '_ERRORED' + CRLF);

      for n := 0 to slErr.Count - 1 do
        WriteLn(txt,slErr[n] + CRLF);

      WriteLn(txt,CRLF + '_RECOVERED' + CRLF);

      for n := 0 to slRecover.Count - 1 do
        WriteLn(txt,slRecover[n] + CRLF);

      WriteLn(txt,CRLF + '_OTHER' + CRLF);

      for n := 0 to slOther.Count - 1 do
        WriteLn(txt,slOther[n] + CRLF);
    finally
      CloseFile(txt);
    end;
  except
    frmMain.red.Text := 'Error writing log.';
  end;
end;

function TJob.MD5FileMatch(File1,File2 : string) : boolean;

var s1,s2 : string;

function CalcMD5Sum(sFile : string) : string;

var buffer : array[0..$8000 - 1] of char;
    err : word;
    md5 : TMD5Digest;

function MD5DigestToString(md5 : TMD5Digest) : string;

var i : integer;

begin
  Result := '';

  for i := 0 to 15 do
    Result := Result + IntToHex(md5[i],2);
end;

begin {CalcMD5Sum()}
  try
    MD5File(sFile,md5,buffer,$8000,err);
  except
    err := 9;
  end;

  if err = 0 then
    Result := MD5DigestToString(md5)
  else
    Result := '';
end;

begin {MD5FileMatch()}
  Result := false;

  with frmMain do
  begin
    pnlMsg.Caption := 'MD5 Src Calc...';
    pnlMsg.Refresh;
    if bCancel then exit;
    s1 := CalcMD5Sum(File1);
    pnlMsg.Caption := 'MD5 Trg Calc...';
    pnlMsg.Refresh;
    Yield;
    if bCancel then exit;
    s2 := CalcMD5Sum(File2);
    Result := s1 = s2;
    pnlMsg.Caption := '';
  end;
end;

{$I+}
function TJob.MkCopy : boolean;

var ToF,FromF : file;
    nPercent,NumRead,NumWritten : integer;
    bErr : boolean;
    nTheseBytes : comp;
    Buffer : array[1..2048] of byte;

begin
  bErr := false;
  LastIO := '';

  if frmMain.chkNoCopy.Checked then
  begin
    Result := true;
    nOkBytes := nOkBytes + Spec.Size;
    exit;
  end;

  try
    AssignFile(FromF,Spec.SrcFile);
    Reset(FromF,1);
  except on e : exception do
    begin
      LastIO := 'Reset() - ' + e.Message + ' - ' + Spec.SrcFile;
      Result := false;
      exit;
    end;
  end;

  try
    AssignFile(ToF,Spec.TrgFile);
    Rewrite(ToF,1);
  except on e : exception do
    begin
      CloseFile(FromF);
      LastIO := 'Rewrite() - ' + e.Message + ' - ' + Spec.TrgFile;
      Result := false;
      exit;
    end;
  end;

  frmMain.pnlFolder.Caption := Spec.Src;
  frmMain.pnlFile.Caption := Spec.Spec;
  frmMain.pnlDest.Caption := Spec.TrgDir;

  frmMain.prgFile.Hint := 'size = ' + CommaStr(
   FloatToStrF(Spec.Size,ffFixed,24,0)) + ' bytes';

  nTheseBytes := 0;

  repeat
    try
      BlockRead(FromF,Buffer,SizeOf(Buffer),NumRead);
    except on e : exception do
      begin
        LastIO := 'BlockRead() - ' + e.Message + ' - ' + Spec.SrcFile;
        bErr := true;
      end;
    end;

    try
      if not bErr then
        BlockWrite(ToF,Buffer,NumRead,NumWritten);
    except on e : exception do
      begin
        LastIO := 'BlockWrite() - ' + e.Message + ' - ' + Spec.TrgFile;
        bErr := true;
      end;
    end;

    nTheseBytes := nTheseBytes + NumWritten;
    nOkBytes := nOkBytes + NumRead;

    if Spec.Size > 0 then
      nPercent := Round((nTheseBytes / Spec.Size) * 100)
    else
      nPercent := 0;

    frmMain.prgFile.Position := nPercent;
    frmMain.UpdateStatus;
  until (bErr)
   or (bCancel)
   or (NumRead = 0)
   or (NumWritten <> NumRead);

  try
    CloseFile(FromF);
    CloseFile(ToF);
  except
  end;

  if (bCancel) or (bErr) then
  begin
    DeleteFile(Spec.TrgFile);
    slOther.Add('Partial copy removed: ' + Spec.TrgFile);
  end;

  Result := (not bErr) and (not bCancel);

  if Result then
    SetFileDate(Spec.TrgFile,GetFileDT(Spec.SrcFile));
end;

procedure TJob.AfterCopy;
begin
  slOk.Add(Spec.TrgFile);
  frmMain.UpdateCounts;
  if bCancel then exit;

  if frmMain.chkMD5.Checked then
  begin
    if not MD5FileMatch(Spec.SrcFile,Spec.TrgFile) then
    begin
      slErr.Add('MD5 mismatch - ' + Spec.SrcFile);
      frmMain.UpdateCounts;
    end;
  end;
end;

procedure TJob.WriteSpec;
begin
  WriteLn(txtJob,Spec.SrcRoot);
  WriteLn(txtJob,Spec.Src);
  WriteLn(txtJob,Spec.Spec);
  WriteLn(txtJob,YN(Spec.Sub));
  WriteLn(txtJob,Spec.Trg);
  WriteLn(txtJob,Spec.TrgRoot);
  WriteLn(txtJob,Spec.Mode);
  WriteLn(txtJob,Spec.Size);
end;

procedure TJob.ReadSpec;

var s : string;

begin
  ReadLn(txtJob,Spec.SrcRoot);
  ReadLn(txtJob,Spec.Src);
  ReadLn(txtJob,Spec.Spec);
  ReadLn(txtJob,s);
  Spec.Sub := YN(s);
  ReadLn(txtJob,Spec.Trg);
  ReadLn(txtJob,Spec.TrgRoot);
  ReadLn(txtJob,Spec.Mode);
  ReadLn(txtJob,Spec.Size);
  Spec.SrcFile := Spec.Src + Spec.Spec;
  SetTrg;

  if (bAuto) and (Spec.Mode = smList) then
    Spec.Mode := smOlder;
end;

procedure TJob.Start;

var nFree,nSize : int64;

begin
  TempFile := SerFile(HDir + c_App + '.tmp');
  nGrandBytes := 0;
  nOkBytes := 0;
  nCopyBytes := 0;
  nSkipBytes := 0;
  nFileTotal := 0;
  dtStart := Now;
  ExpandFiles;

  if (not bAuto) and
   (FreeSpace(Spec.Trg[1],nFree,nSize)) and
   (nFree < nCopyBytes) then
  begin
    if not Confirm(c_App,
     'Free space required  : ' +
     FileSizeText(nCopyBytes) + #13 +
     'Free space available : ' +
     FileSizeText(nFree) +
     #13#13 + 'Proceed anyway?') then
      bCancel := true;
  end;

  if (not bCancel) and
   (slSkip.Count < nFileTotal) then
    DoJob;

  if not bCancel then
  begin
    if slPrompt.Count > 0 then
      frmPreview.ShowModal;

    if frmMain.chkDelTrg.Checked then
      DelIfNotInSource;

    if frmMain.chkDelEmpty.Checked then
    begin
      DelEmptyDirs(Spec.SrcRoot);
      DelEmptyDirs(CleanPath(Spec.Trg + NewRoot));
    end;
  end;

  Stop;
end;

procedure TJob.SetTrg;
begin
  Spec.TrgDir := CleanPath(Spec.Trg + NewRoot + copy(Spec.Src,length(Spec.SrcRoot),999));
  Spec.TrgFile := Spec.TrgDir + Spec.Spec;
end;

procedure TJob.ExpandFiles;

var n,j : integer;
    sl : TFileStrings;
    jsize : comp;

procedure Include(i : TListItem; sFile : string);

var size2 : comp;
    dt1,dt2 : TDateTime;
    bCopy : boolean;

begin
  if bCancel then exit;
  bErr := false;
  bCopy := true;
  Spec.SrcRoot := i.Caption;
  Spec.Src := ExtractFilePath(sFile);
  Spec.Spec := ExtractFilename(sFile);
  Spec.SrcFile := sFile;
  Spec.Sub := YN(i.SubItems[c_Sub]);
  Spec.Trg := i.SubItems[c_Trg];
  Spec.TrgRoot := i.SubItems[c_TrgRoot];
  Spec.Mode := frmEditProps.cmbMode.Items.IndexOf(i.SubItems[c_Mode]);
  if Spec.Mode = -1 then Spec.Mode := smOlder; {older .scc files}
  SetTrg;
  Spec.Size := VPAExt.GetFileSize(Spec.SrcFile);
  inc(nFileTotal);

  if Spec.Size = -1 then
  begin
    bCopy := false;
    bErr := true;
    slErr.Add(Spec.SrcFile);

    if slOther.IndexOf('Access error - ' + Spec.SrcFile) = -1 then
      slOther.Add('Access error - ' + Spec.SrcFile);
  end
  else nGrandBytes := nGrandBytes + Spec.Size;

  frmMain.pnlFileTotal.Caption := CommaStr(IntToStr(nFileTotal));
  frmMain.pnlGrand.Caption := FileSizeText(nGrandBytes);

  if frmMain.chkDelTrg.Checked then
    slSrc.Add(Spec.TrgFile);

  {repeating for bAuto and post-check failure}
  if not DirectoryExists(Spec.Trg) then
  begin
    bCopy := false;
    bErr := true;
    slErr.Add(Spec.SrcFile);

    if slOther.IndexOf('Invalid target - ' + Spec.Trg) = -1 then
      slOther.Add('Invalid target - ' + Spec.Trg);
  end;

  if bCopy then
    bCopy := (not SkipTest);

  if (bCopy) and (FileExists(Spec.TrgFile)) then
  begin
    if Spec.Mode = smSize then
    begin
      size2 := VPAExt.GetFileSize(Spec.TrgFile);

      if size2 = -1 then
      begin
        bCopy := false;
        bErr := true;
        slErr.Add('Access error - ' + Spec.TrgFile);
      end
      else bCopy := Spec.Size <> size2;
    end
    else if Spec.Mode = smOlder then
    begin {only mode for bidir sync, which can reverse meaning of Src Trg as-is}
      dt1 := GetFileDT(Spec.SrcFile);

      if dt1 = -1 then
      begin
        bCopy := false;
        bErr := true;
        slErr.Add('Access error - ' + Spec.SrcFile);
      end;

      if bCopy then
      begin
        dt2 := GetFileDT(Spec.TrgFile);

        if dt2 = -1 then
        begin
          bCopy := false;
          bErr := true;
          slErr.Add('Access error - ' + Spec.TrgFile);
        end
        else bCopy := dt1 > dt2;
      end;
    end
    else if Spec.Mode = smAlways then
      bCopy := true
    else if Spec.Mode = smNever then
      bCopy := false
    else if Spec.Mode = smList then
    begin
      slPrompt.Add(Spec.SrcFile + '+++' + Spec.TrgFile);
      bCopy := false;
    end
    else bCopy := true;
  end;

  if not bCopy then
  begin
    if not bErr then
    begin
      nSkipBytes := nSkipBytes + Spec.Size;
      slSkip.Add(Spec.SrcFile);
    end;

    frmMain.UpdateCounts;
  end
  else {accept}
  begin
    jsize := jsize + Spec.Size;
    nCopyBytes := nCopyBytes + Spec.Size;
    WriteSpec;
  end;

  frmMain.UpdateStatus;
end;

begin {ExpandFiles}
  DeleteFile(TempFile);
  AssignFile(txtJob,TempFile);
  Rewrite(txtJob);

  with frmMain do
  begin
    for n := 0 to lv.Items.Count - 1 do
    begin
      if bCancel then break;
      jsize := 0;
      pnlFolder.Caption := lv.Items[n].Caption;

      if FileSpec(lv.Items[n].SubItems[c_Spec]) then
      begin
        sl := TFileStrings.Create(
         lv.Items[n].Caption,
         lv.Items[n].SubItems[c_Spec],false,
         YN(lv.Items[n].SubItems[c_Sub]));

        if (sl.Count = 0) and
         (not DirectoryExists(lv.Items[n].Caption)) and
         (slOther.IndexOf('Invalid source: ' + lv.Items[n].Caption) = -1) then
          slOther.Add('Invalid source: ' + lv.Items[n].Caption);

        try
          for j := 0 to sl.Count - 1 do
            Include(lv.Items[n],sl[j]);
        finally
          sl.Free;
        end;
      end
      else Include(lv.Items[n],
       lv.Items[n].Caption +
       lv.Items[n].SubItems[c_Spec]);

      lv.Items[n].SubItems[c_Size] := FileSizeText(jsize);
    end;

    pnlFileTotal.Caption := CommaStr(IntToStr(nFileTotal));
    pnlGrand.Caption := FileSizeText(nGrandBytes);
  end;

  Reset(txtJob);
end;

procedure TJob.DoJob;

var nRetry : integer;
    bRetry : boolean;
    dtWait : TDateTime;

function Secs : integer;
begin
  Result := (Round(Frac(Now) * SecsPerDay)) -
   (Round(Frac(dtWait) * SecsPerDay));
end;

begin {DoJob}
  while not eof(txtJob) do
  begin
    ReadSpec;
    MkTrgDir;

    if FileExists(Spec.Src + Spec.Spec) then
    begin
      if not MkCopy then
      begin
        if pos(c_NoRoom,LastIO) > 0 then
        begin
          slOther.Add('Out of disk space.');
          bCancel := true;
        end;

        slErr.Add(LastIO);

        if frmMain.chkRetry.Checked then
        begin
          nRetry := 0;
          bRetry := frmMain.spnRetry.Value > 0;

          while (bRetry) and (not bCancel) do
          begin
            inc(nRetry);

            if frmMain.chkRetryForever.Checked then
              frmMain.spnRetry.Value := nRetry;

            dtWait := Now;

            while Secs < frmMain.spnRetryInt.Value do
            begin
              if bCancel then break;

              if frmMain.chkRetryForever.Checked then
                frmMain.pnlMsg.Caption := 'Pausing [' +
                 IntToStr(frmMain.spnRetryInt.Value - Secs) + 's]'
              else
                frmMain.pnlMsg.Caption := 'Retry ' +
                 IntToStr(nRetry) + '/' +
                 IntToStr(frmMain.spnRetry.Value) + ' - pausing ' +
                 IntToStr(frmMain.spnRetryInt.Value - Secs) + 's]';

              Sleep(333);
              frmMain.Yield;
            end;

            frmMain.pnlMsg.Caption := 'Retry attempt ' +
             CommaStr(IntToStr(nRetry));

            if (not bCancel) and (MkCopy) then
            begin
              nRetry := c_CopyOK;
              slRecover.Add(Spec.SrcFile);
              slErr.Delete(slErr.IndexOf(Spec.SrcFile));
              frmMain.UpdateCounts;
              break;
            end;

            if frmMain.chkRetryForever.Checked then
              bRetry := true
            else
              bRetry := nRetry < frmMain.spnRetry.Value;
          end; {bRetry}

          if nRetry = c_CopyOK then AfterCopy;
        end;
      end
      else AfterCopy;
    end;

    frmMain.UpdateStatus;
    if bCancel then break;
  end;
end;

procedure TJob.Stop;

var i : TListItem;
    LogFile : string;

begin
  CloseFile(txtJob);
  DeleteFile(TempFile);
  try ChDir('C:\'); except; end;

  with frmMain do
  begin
    prgFile.Hint := '';
    prgAll.Hint := '';
    pnlFileTotal.Hint := '';

    if bCancel then
      pnlRemain.Caption := 'Cancelled'
    else
      pnlRemain.Caption := 'Done';

    if chkLogs.Checked then
    begin
      LogFile := WriteLog;

      if bAuto then
        i := nil
      else
        i := AddLog(LogFile);

      if i <> nil then
      begin
        pag.ActivePage := tabResults;
        i.MakeVisible(false);
        lvRes.Selected := i;
        red.SetFocus;
      end;
    end;
  end;
end;

end.
